/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapreduce.security.token;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenIdentifier;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.JobID;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.UserGroupInformationThreadLocal;
import org.apache.hadoop.security.token.SecretManager;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;

public class DelegationTokenRenewal {
    private static final Log LOG = LogFactory.getLog(DelegationTokenRenewal.class);
    public static final String SCHEME = "hdfs";
    public static UserGroupInformation mrOwnerUGI;
    private static Timer renewalTimer;
    private static Set<DelegationTokenToRenew> delegationTokens;
    private static final Text kindHdfs;

    private static void addTokenToList(DelegationTokenToRenew t) {
        delegationTokens.add(t);
    }

    public static synchronized void registerDelegationTokensForRenewal(JobID jobId, Credentials ts, Configuration conf, UserGroupInformation ugi) {
        mrOwnerUGI = ugi;
        DelegationTokenRenewal.registerDelegationTokensForRenewal(jobId, ts, conf);
    }

    public static synchronized void registerDelegationTokensForRenewal(JobID jobId, Credentials ts, Configuration conf) {
        if (ts == null) {
            return;
        }
        Collection<Token<? extends TokenIdentifier>> tokens = ts.getAllTokens();
        long now = System.currentTimeMillis();
        for (Token<? extends TokenIdentifier> t : tokens) {
            if (!t.getKind().equals(kindHdfs)) continue;
            Token<? extends TokenIdentifier> dt = t;
            DelegationTokenToRenew dtr = new DelegationTokenToRenew(jobId, dt, conf, now);
            DelegationTokenRenewal.addTokenToList(dtr);
            DelegationTokenRenewal.setTimerForTokenRenewal(dtr, true);
            LOG.info((Object)("registering token for renewal for service =" + dt.getService() + " and jobID = " + jobId));
        }
    }

    private static long renewDelegationToken(DelegationTokenToRenew dttr) throws Exception {
        long newExpirationDate = System.currentTimeMillis() + 3600000L;
        Token<DelegationTokenIdentifier> token = dttr.token;
        Configuration conf = dttr.conf;
        if (token.getKind().equals(kindHdfs)) {
            try {
                DistributedFileSystem dfs = DelegationTokenRenewal.getDFSForToken(token, conf);
                newExpirationDate = dfs.renewDelegationToken(token);
            }
            catch (SecretManager.InvalidToken ite) {
                LOG.warn((Object)"invalid token - not scheduling for renew");
                DelegationTokenRenewal.removeFailedDelegationToken(dttr);
                throw new IOException("failed to renew token", ite);
            }
            catch (AccessControlException ioe) {
                LOG.warn((Object)("failed to renew token:" + token), (Throwable)ioe);
                DelegationTokenRenewal.removeFailedDelegationToken(dttr);
                throw new IOException("failed to renew token", ioe);
            }
            catch (Exception e) {
                LOG.warn((Object)("failed to renew token:" + token), (Throwable)e);
            }
        } else {
            throw new Exception("unknown token type to renew+" + token.getKind());
        }
        return newExpirationDate;
    }

    private static DistributedFileSystem getDFSForToken(Token<DelegationTokenIdentifier> token, final Configuration conf) throws Exception {
        DistributedFileSystem dfs = null;
        try {
            String[] ipaddr = token.getService().toString().split(":");
            InetAddress iaddr = InetAddress.getByName(ipaddr[0]);
            String dnsName = iaddr.getCanonicalHostName();
            final URI uri = new URI("hdfs://" + dnsName + ":" + ipaddr[1]);
            dfs = UserGroupInformation.getLoginUser().doAs(new PrivilegedExceptionAction<DistributedFileSystem>(){

                @Override
                public DistributedFileSystem run() throws IOException {
                    return (DistributedFileSystem)FileSystem.get(uri, conf);
                }
            });
        }
        catch (Exception e) {
            LOG.warn((Object)("Failed to create a dfs to renew for:" + token.getService()), (Throwable)e);
            throw e;
        }
        return dfs;
    }

    private static void setTimerForTokenRenewal(DelegationTokenToRenew token, boolean firstTime) {
        long renewIn;
        long now = System.currentTimeMillis();
        if (firstTime) {
            renewIn = now;
        } else {
            long expiresIn = token.expirationDate - now;
            renewIn = now + expiresIn - expiresIn / 10L;
        }
        try {
            RenewalTimerTask tTask = new RenewalTimerTask(token, mrOwnerUGI);
            token.setTimerTask(tTask);
            renewalTimer.schedule(token.timerTask, new Date(renewIn));
        }
        catch (Exception e) {
            LOG.warn((Object)"failed to schedule a task, token will not renew more", (Throwable)e);
        }
    }

    public static void close() {
        renewalTimer.cancel();
        delegationTokens.clear();
    }

    private static void cancelToken(DelegationTokenToRenew t) {
        Token<DelegationTokenIdentifier> token = t.token;
        Configuration conf = t.conf;
        if (token.getKind().equals(kindHdfs)) {
            try {
                DistributedFileSystem dfs = DelegationTokenRenewal.getDFSForToken(token, conf);
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("canceling token " + token.getService() + " for dfs=" + dfs));
                }
                dfs.cancelDelegationToken(token);
            }
            catch (Exception e) {
                LOG.warn((Object)("Failed to cancel " + token), (Throwable)e);
            }
        }
    }

    private static void removeFailedDelegationToken(DelegationTokenToRenew t) {
        JobID jobId = t.jobId;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("removing failed delegation token for jobid=" + jobId + ";t=" + t.token.getService()));
        }
        delegationTokens.remove(t);
        if (t.timerTask != null) {
            t.timerTask.cancel();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void removeDelegationTokenRenewalForJob(JobID jobId) {
        Set<DelegationTokenToRenew> set = delegationTokens;
        synchronized (set) {
            Iterator<DelegationTokenToRenew> it = delegationTokens.iterator();
            while (it.hasNext()) {
                DelegationTokenToRenew dttr = it.next();
                if (!dttr.jobId.equals(jobId)) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("removing delegation token for jobid=" + jobId + ";t=" + dttr.token.getService()));
                }
                if (dttr.timerTask != null) {
                    dttr.timerTask.cancel();
                }
                DelegationTokenRenewal.cancelToken(dttr);
                it.remove();
            }
        }
    }

    static {
        renewalTimer = new Timer(true);
        delegationTokens = Collections.synchronizedSet(new HashSet());
        kindHdfs = DelegationTokenIdentifier.HDFS_DELEGATION_KIND;
    }

    private static class RenewalTimerTask
    extends TimerTask {
        private DelegationTokenToRenew dttr;
        private UserGroupInformation ugi;

        RenewalTimerTask(DelegationTokenToRenew t) {
            this.dttr = t;
        }

        RenewalTimerTask(DelegationTokenToRenew t, UserGroupInformation ugi) {
            this(t);
            this.ugi = ugi;
        }

        @Override
        public void run() {
            UserGroupInformationThreadLocal.set(this.ugi);
            Token<DelegationTokenIdentifier> token = this.dttr.token;
            long newExpirationDate = 0L;
            try {
                newExpirationDate = DelegationTokenRenewal.renewDelegationToken(this.dttr);
            }
            catch (Exception e) {
                return;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("renewing for:" + token.getService() + ";newED=" + newExpirationDate));
            }
            this.dttr.expirationDate = newExpirationDate;
            DelegationTokenRenewal.setTimerForTokenRenewal(this.dttr, false);
        }
    }

    private static class DelegationTokenToRenew {
        public final Token<DelegationTokenIdentifier> token;
        public final JobID jobId;
        public final Configuration conf;
        public long expirationDate;
        public TimerTask timerTask;

        public DelegationTokenToRenew(JobID jId, Token<DelegationTokenIdentifier> t, Configuration newConf, long newExpirationDate) {
            this.token = t;
            this.jobId = jId;
            this.conf = newConf;
            this.expirationDate = newExpirationDate;
            this.timerTask = null;
            if (this.token == null || this.jobId == null || this.conf == null) {
                throw new IllegalArgumentException("invalid params for Renew Token;t=" + this.token + ";j=" + this.jobId + ";c=" + this.conf);
            }
        }

        public void setTimerTask(TimerTask tTask) {
            this.timerTask = tTask;
        }

        public String toString() {
            return this.token + ";exp=" + this.expirationDate;
        }

        public boolean equals(Object obj) {
            return obj instanceof DelegationTokenToRenew && this.token.equals(((DelegationTokenToRenew)obj).token);
        }

        public int hashCode() {
            return this.token.hashCode();
        }
    }
}

