/*
 * Copyright 2006,2007 WSO2, Inc. http://www.wso2.org\
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wso2.wsas.sample.sts.client;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.util.Base64;
import org.apache.axiom.soap.SOAP12Constants;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.ConfigurationContextFactory;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyEngine;
import org.apache.rahas.RahasConstants;
import org.apache.rahas.Token;
import org.apache.rahas.TokenStorage;
import org.apache.rahas.TrustUtil;
import org.apache.rahas.client.STSClient;
import org.apache.rampart.RampartMessageData;
import org.apache.rampart.policy.model.CryptoConfig;
import org.apache.rampart.policy.model.RampartConfig;
import org.apache.ws.secpolicy.Constants;
import org.apache.ws.security.WSPasswordCallback;
import org.opensaml.XML;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.xml.namespace.QName;

import java.io.File;
import java.io.IOException;
import java.util.Properties;

/**
 * Client of the STS sample
 */
public class Client implements CallbackHandler {

    private static final String wso2wsasHome = System.getProperty("wso2wsas.home");
    
    public static void main(String[] args) throws Exception {
        
        String sts;
        String service;
        if(args.length != 2) {
            sts = "http://localhost:9762/services/wso2wsas-sts";
            service = "http://localhost:9762/services/echo";
        } else {
            sts = args[0];
            service = args[1];
        }

        System.out.println("Security token service endpoint address: " + sts);
        System.out.println("Secured Service endpoint address: " + service);
        
        String clientSSLStore = wso2wsasHome + File.separator + "conf"
                + File.separator + "client-truststore.jks";
        
        System.getProperties().remove("javax.net.ssl.trustStore");
        System.getProperties().remove("javax.net.ssl.trustStoreType");
        System.getProperties().remove("javax.net.ssl.trustStorePassword");

        System.setProperty("javax.net.ssl.trustStore", clientSSLStore);
        System.setProperty("javax.net.ssl.trustStoreType", "JKS");
        System.setProperty("javax.net.ssl.trustStorePassword", "wso2wsas");

        ConfigurationContext ctx  =
            ConfigurationContextFactory.createConfigurationContextFromFileSystem(
                wso2wsasHome + "/samples/sts-sample/temp/client-repo");
        //STS Invocation 
        STSClient stsClient = new STSClient(ctx);
        
        Policy stsPolicy = loadPolicy(wso2wsasHome +  
                "/samples/sts-sample/conf/sts.policy.xml");
        Policy servicePolicy = loadPolicy(wso2wsasHome +  
                "/samples/sts-sample/conf/service.policy.xml");
        
        stsClient.setRstTemplate(getRSTTemplate());
        stsClient.setAction(RahasConstants.WST_NS_05_02 + RahasConstants.RST_ACTION_ISSUE);
        Token responseToken = stsClient.requestSecurityToken(servicePolicy,  sts, stsPolicy, service);
        
        System.out.println("\nRECEIVED SECRET: "
                + Base64.encode(responseToken.getSecret()) + "\n");
        System.out.println("RECEIVED TOKEN: " + responseToken.getToken() + "\n");
        
        
        //Store token
        TokenStorage store = TrustUtil.getTokenStore(ctx);
        store.add(responseToken);
        
        //Service invocation
        ServiceClient client = new ServiceClient(ctx, null);
        client.engageModule("rampart");
        client.engageModule("addressing");
        
        client.getServiceContext().setProperty(RampartMessageData.KEY_RAMPART_POLICY, servicePolicy);
        Options options = client.getOptions();
        options.setAction("urn:echoOMElement");
        options.setSoapVersionURI(SOAP12Constants.SOAP_ENVELOPE_NAMESPACE_URI);
        options.setTo(new EndpointReference(service));
        options.setProperty(RampartMessageData.KEY_CUSTOM_ISSUED_TOKEN, responseToken.getId());
        OMElement respEelm = client.sendReceive(getPayload());
        System.out.println("RESPONSE FROM SERVICE: \n" + respEelm);
    }
  

    private static Policy loadPolicy(String xmlPath) throws Exception {
        StAXOMBuilder builder = new StAXOMBuilder(xmlPath);
        Policy policy = PolicyEngine.getPolicy(builder.getDocumentElement());
        
        RampartConfig rc = new RampartConfig();

        rc.setUser("client");
        rc.setEncryptionUser("wso2wsas");
        rc.setPwCbClass(Client.class.getName());

        CryptoConfig sigCryptoConfig = new CryptoConfig();

        sigCryptoConfig.setProvider("org.apache.ws.security.components.crypto.Merlin");

        String keystore = wso2wsasHome + File.separator + "samples" +
                          File.separator + "sts-sample" + File.separator +
                          "conf" + File.separator + "client-truststore.jks";

        Properties prop1 = new Properties();
        prop1.put("org.apache.ws.security.crypto.merlin.keystore.type", "JKS");
        prop1.put("org.apache.ws.security.crypto.merlin.file", keystore);
        prop1.put("org.apache.ws.security.crypto.merlin.keystore.password", "wso2wsas");
        sigCryptoConfig.setProp(prop1);

        CryptoConfig encrCryptoConfig = new CryptoConfig();
        encrCryptoConfig.setProvider("org.apache.ws.security.components.crypto.Merlin");

        Properties prop2 = new Properties();

        prop2.put("org.apache.ws.security.crypto.merlin.keystore.type", "JKS");
        prop2.put("org.apache.ws.security.crypto.merlin.file", keystore);
        prop2.put("org.apache.ws.security.crypto.merlin.keystore.password", "wso2wsas");
        encrCryptoConfig.setProp(prop2);

        rc.setSigCryptoConfig(sigCryptoConfig);
        rc.setEncrCryptoConfig(encrCryptoConfig);

        policy.addAssertion(rc);

        return policy;
    }

    private static OMElement getRSTTemplate() throws Exception {
        OMFactory fac = OMAbstractFactory.getOMFactory();
        OMElement elem = fac.createOMElement(Constants.RST_TEMPLATE);
        TrustUtil.createTokenTypeElement(RahasConstants.VERSION_05_02, elem).setText(XML.SAML_NS);
        TrustUtil.createKeyTypeElement(RahasConstants.VERSION_05_02, elem, RahasConstants.KEY_TYPE_SYMM_KEY);
        TrustUtil.createKeySizeElement(RahasConstants.VERSION_05_02, elem, 256);
        return elem;
    }
    
    private static OMElement getPayload() {
        OMFactory fac = OMAbstractFactory.getOMFactory();
        OMElement elem = fac.createOMElement(new QName("http://echo.services.wsas.wso2.org", "Echo" ));
        elem.setText("WSO2 WSAS Rocks!!!");
        return elem;
    }
    
    public void handle(Callback[] callbacks) throws IOException,
            UnsupportedCallbackException {
        WSPasswordCallback cb = (WSPasswordCallback)callbacks[0];
        cb.setPassword("wso2wsas");
    }

}
