/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.wso2.wsas.sample.xkms.recover;

import java.security.Key;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateCrtKey;
import java.util.List;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.xml.security.keys.KeyInfo;
import org.wso2.wsas.sample.xkms.utils.Utils;
import org.wso2.xkms2.Authentication;
import org.wso2.xkms2.KeyBinding;
import org.wso2.xkms2.PrivateKey;
import org.wso2.xkms2.RecoverKeyBinding;
import org.wso2.xkms2.RecoverRequest;
import org.wso2.xkms2.RecoverResult;
import org.wso2.xkms2.RespondWith;
import org.wso2.xkms2.Status;
import org.wso2.xkms2.StatusValue;
import org.wso2.xkms2.UseKeyWith;
import org.wso2.xkms2.XKMSException;
import org.wso2.xkms2.builder.RecoverResultBuilder;
import org.wso2.xkms2.util.XKMSKeyUtil;

public class RecoverServiceDemo {

    private static String XKMS_SERVICE_URL = "http://127.0.0.1:9762/services/xkms";
    private static String PASS_PHRASE = "secret";
    
    static {
        org.apache.xml.security.Init.init();
    }

    public static void main(String[] args) throws Exception {

        RecoverRequest request = Utils.createRecoverRequest();
        request.setServiceURI(XKMS_SERVICE_URL);
        
        Authentication authentication = new Authentication();
        Key authenKey = XKMSKeyUtil.getAuthenticationKey(PASS_PHRASE);
        authentication.setKeyBindingAuthenticationKey(authenKey);
        request.setAuthentication(authentication);

        RecoverKeyBinding keyBinding = Utils.createRecoverKeyBinding();
        keyBinding.addUseKeyWith(UseKeyWith.PKIX,
                "CN=Bob, OU=OASIS Interop Test Cert, O=OASIS");
        
        Status status = new Status();
        status.setStatusValue(StatusValue.INDETERMINATE);
        keyBinding.setStatus(status);
        
        request.setRecoverKeyBinding(keyBinding);

        request.addRespondWith(RespondWith.X_509_CERT);
        request.addRespondWith(RespondWith.PRIVATE_KEY);

        OMElement element = getAsOMElement(request);
        OMElement result = Utils.sendReceive(element, XKMS_SERVICE_URL);
        result = Utils.buildElement(result);

        RecoverResult recoverResult = getRecoverResult(result);
        List keybindings = recoverResult.getKeyBindings();
        KeyBinding keybinding = (KeyBinding) keybindings.get(0);

        KeyInfo keyInfo = keybinding.getKeyInfo();
        X509Certificate cert = keyInfo.getX509Certificate();
        System.out.println("Recovered X509Certificate");
        Utils.printCert(cert);

        PrivateKey privateKey = recoverResult.getPrivateKey();
        Key decryptionKey = XKMSKeyUtil.getPrivateKey("secret", "DESede");
        privateKey.setKey(decryptionKey);

        KeyPair rsaKeyPair = privateKey.getRSAKeyPair();
        System.out.println("Recovered Private Key");
        Utils.printRSAPrivateKey((RSAPrivateCrtKey) rsaKeyPair.getPrivate());
    }

    private static OMElement getAsOMElement(RecoverRequest request)
            throws XKMSException {
        OMFactory factory = DOOMAbstractFactory.getOMFactory();
        return request.serialize(factory);
    }

    private static RecoverResult getRecoverResult(OMElement recoverResultElem)
            throws Exception {
        return (RecoverResult) RecoverResultBuilder.INSTANCE
                .buildElement(recoverResultElem);
    }

}
