/*
 * Copyright 2007  WSO2 Inc. (http://www.wso2.org)
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.wso2.wsas.sample.xkms.locate;

import java.io.File;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.List;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.xml.security.keys.KeyInfo;
import org.wso2.wsas.sample.xkms.utils.Utils;
import org.wso2.xkms2.LocateRequest;
import org.wso2.xkms2.LocateResult;
import org.wso2.xkms2.QueryKeyBinding;
import org.wso2.xkms2.RespondWith;
import org.wso2.xkms2.ResultMinor;
import org.wso2.xkms2.UnverifiedKeyBinding;
import org.wso2.xkms2.XKMSException;
import org.wso2.xkms2.builder.LocateResultBuilder;

/**
 * This sample desmonstrates how to retrieve a X509Certificate using its alias
 * from a XKMS service.
 */
public class LocateServiceDemo {

    private static String XKMS_SERVICE_URL = "http://127.0.0.1:9762/services/xkms";

    private static String WSO2_HOME = System.getProperty("wso2wsas.home");

    private static String separator = File.separator;

    public static void main(String[] args) throws Exception {
        // alice is the alias of a X509Certificate which is already uploaded to
        // XKMS service.
        fetchCert("alice");

        // jane is a fake alias. Hence we should get NO_MATCH response from the
        // server.
        fetchCert("jane");
    }

    private static void fetchCert(String alias) throws Exception {
        String location = WSO2_HOME + separator + "samples" + separator
                + "XKMS" + separator + "conf" + separator + "keystore.jks";
        
        String password = "password";

        // loading client keystore
        FileInputStream fis = new FileInputStream(location);
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(fis, password.toCharArray());

        LocateRequest request = Utils.createLocateRequest();
        request.setServiceURI(XKMS_SERVICE_URL);

        QueryKeyBinding keyBinding = Utils.createQueryKeyBinding();
        keyBinding.setKeyName(alias);
        request.setQueryKeyBinding(keyBinding);

        request.addRespondWith(RespondWith.X_509_CERT);

        OMElement element = getAsOMElement(request);
        OMElement result = Utils.sendReceive(element, XKMS_SERVICE_URL);
        result = Utils.buildElement(result);

        LocateResult locateResult = getLocateResult(result);

        if (ResultMinor.NO_MATCH.equals(locateResult.getResultMinor())) {
            System.out.println("No X509Certificate is found for the alias : "
                    + alias);

        } else {
            System.out.println(" X509Certificate found for alias : " + alias);
            // pic the first keybinding from the keybinding list returned
            List keybindings = locateResult.getUnverifiedKeyBindingList();
            UnverifiedKeyBinding keybinding = (UnverifiedKeyBinding) keybindings
                    .get(0);
            KeyInfo keyInfo = keybinding.getKeyInfo();
            X509Certificate cert = keyInfo.getX509Certificate();
            Utils.printCert(cert);
        }
    }

    private static OMElement getAsOMElement(LocateRequest request)
            throws XKMSException {
        OMFactory factory = DOOMAbstractFactory.getOMFactory();
        return request.serialize(factory);
    }

    private static LocateResult getLocateResult(OMElement result)
            throws Exception {
        LocateResult locateResult = (LocateResult) LocateResultBuilder.INSTANCE
                .buildElement(result);
        return locateResult;
    }

}
