/*
* Copyright 2005,2006 WSO2, Inc. http://wso2.com
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
*/
package org.wso2.throttle.module;


import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.*;
import org.apache.axis2.modules.Module;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Assertion;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyEngine;
import org.wso2.throttle.*;

import javax.xml.namespace.QName;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

public class ThrottleModule implements Module {

    private static Log log = LogFactory.getLog(ThrottleModule.class.getName());

    private Policy defaultPolicy = null;
    private Throttle defaultThrottle = null;
    private ConfigurationContext configctx;

    /**
     * initialize the module
     */
    public void init(ConfigurationContext configContext, AxisModule module) throws AxisFault {
        this.configctx = configContext;
        initDefaultPolicy();
        initDefaultThrottle();
        Throttle throttle;
        PolicyInclude policyInclude = module.getPolicyInclude();
        if (policyInclude != null) {
            Policy policy = policyInclude.getEffectivePolicy();
            if (policy != null) {
                try {
                    throttle = ThrottlePolicyProcessor.processPolicy(policy);
                }
                catch (ThrottleException e) {

                    log.error("Error was ocuured when initing throttle module " + e.getMessage());
                    log.info("Throttling will occur using default module policy");

                    String id = policy.getId();
                    policyInclude.removePolicyElement(id);
                    defaultPolicy.setId(id);
                    policyInclude.addPolicyElement(PolicyInclude.AXIS_MODULE_POLICY, defaultPolicy);
                    throttle = defaultThrottle;
                }
                if (throttle != null) {
                    Map throttles =
                        (Map) configctx.getPropertyNonReplicable(ThrottleConstants.THROTTLES_MAP);
                    if (throttles == null) {
                        throttles = new HashMap();
                        configctx.setNonReplicableProperty(ThrottleConstants.THROTTLES_MAP, throttles);
                    }
                    if (!throttles.containsKey(ThrottleConstants.GLOBAL_THROTTLE_KEY)) {
                        throttle.setId(ThrottleConstants.GLOBAL_THROTTLE_ID);
                        throttles.put(ThrottleConstants.GLOBAL_THROTTLE_KEY, throttle);
                    }
                    ConcurrentAccessController cac = throttle.getConcurrentAccessController();
                    if (cac != null) {
                        String cacKey = ThrottleConstants.THROTTLE_PROPERTY_PREFIX
                            + ThrottleConstants.GLOBAL_THROTTLE_ID + ThrottleConstants.CAC_SUFFIX;
                        if (configctx.getPropertyNonReplicable(cacKey) == null) {
                            configctx.setProperty(cacKey, cac);
                        }
                    }
                }
            }
        }

    }

    public void engageNotify(AxisDescription axisDescription) throws AxisFault {

        String currentServiceName;
        if (axisDescription instanceof AxisService) {
            Throttle throttle = null;
            AxisService currentService = ((AxisService) axisDescription);
            PolicyInclude policyInclude = currentService.getPolicyInclude();
            if (policyInclude != null) {
                try {
                    Policy currentPolicy = policyInclude.getPolicy();
                    if (currentPolicy == null) {
                        currentPolicy = policyInclude.getEffectivePolicy();
                    }
                    if (currentPolicy != null) {
                        throttle = ThrottlePolicyProcessor.processPolicy(currentPolicy);
                    }
                }
                catch (ThrottleException e) {

                    log.error("Error was ocuured when engaging throttle module for the service :" +
                        currentService.getName() + e.getMessage());
                    log.info("Throttling will occur using default module policy");
                    throttle = defaultThrottle;
                }
                if (throttle != null) {
                    Map throttles =
                        (Map) configctx.getPropertyNonReplicable(ThrottleConstants.THROTTLES_MAP);
                    if (throttles == null) {
                        throttles = new HashMap();
                        configctx.setNonReplicableProperty(ThrottleConstants.THROTTLES_MAP, throttles);
                    }
                    String serviceName = currentService.getName();
                    if (!throttles.containsKey(serviceName)) {
                        throttle.setId(serviceName);
                        throttles.put(serviceName, throttle);
                    }
                    ConcurrentAccessController cac = throttle.getConcurrentAccessController();
                    if (cac != null) {
                        String cacKey = ThrottleConstants.THROTTLE_PROPERTY_PREFIX
                                                  + serviceName + ThrottleConstants.CAC_SUFFIX;
                        if (configctx.getPropertyNonReplicable(cacKey) == null) {
                            configctx.setProperty(cacKey, cac);
                        }
                    }
                }
            }
        } else if (axisDescription instanceof AxisOperation) {

            Throttle throttle = null;
            AxisOperation currentOperation = ((AxisOperation) axisDescription);
            AxisService axisService = (AxisService) currentOperation.getParent();
            if (axisService != null) {
                currentServiceName = axisService.getName();
                PolicyInclude policyInclude = currentOperation.getPolicyInclude();
                if (policyInclude != null) {
                    try {
                        Policy currentPolicy = policyInclude.getPolicy();
                        if (currentPolicy == null) {
                            currentPolicy = policyInclude.getEffectivePolicy();
                        }
                        if (currentPolicy != null) {
                            throttle = ThrottlePolicyProcessor.processPolicy(currentPolicy);
                        }
                    } catch (ThrottleException e) {
                        log.error("Error was ocuured when engaging throttle module for operation : " +
                            currentOperation.getName() + " in the service :" +
                            currentServiceName + e.getMessage());
                        log.info("Throttling will occur using default module policy");
                        throttle = defaultThrottle;
                    }

                    if (throttle != null) {
                        Map throttles =
                            (Map) configctx.getPropertyNonReplicable(ThrottleConstants.THROTTLES_MAP);
                        if (throttles == null) {
                            throttles = new HashMap();
                            configctx.setNonReplicableProperty(ThrottleConstants.THROTTLES_MAP
                                                                                    , throttles);
                        }
                        QName opQName = currentOperation.getName();
                        if (opQName != null) {
                            String opName = opQName.getLocalPart();
                            String key = currentServiceName + opName;
                            if (!throttles.containsKey(key)) {
                                throttle.setId(key);
                                throttles.put(key, throttle);
                            }
                            ConcurrentAccessController cac = throttle.getConcurrentAccessController();
                            if (cac != null) {
                                String cacKey = ThrottleConstants.THROTTLE_PROPERTY_PREFIX
                                    + key + ThrottleConstants.CAC_SUFFIX;
                                if (configctx.getPropertyNonReplicable(cacKey) == null) {
                                    configctx.setProperty(cacKey, cac);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    public void shutdown(ConfigurationContext configurationContext) throws AxisFault {
        //Todo
    }

    public void applyPolicy(Policy policy, AxisDescription axisDescription) throws AxisFault {

        // TODO
    }

    public boolean canSupportAssertion(Assertion assertion) {
        // TODO
        return true;
    }

    private void initDefaultPolicy() throws AxisFault {
        InputStream inputStream =
            this.getClass().getResourceAsStream("/resources/policy/default_module_policy.xml");
        if (inputStream != null) {
            defaultPolicy = PolicyEngine.getPolicy(inputStream);
        } else {
            throw new AxisFault("Couldn't load the default throttle policy .the module is invalid ");
        }
    }

    private void initDefaultThrottle() throws AxisFault {
        try {
            if (defaultPolicy != null) {
                defaultThrottle = ThrottlePolicyProcessor.processPolicy(defaultPolicy);
                if (defaultThrottle == null) {
                    throw new AxisFault("Couldn't create the default throttle  .the module is invalid ");
                }
            } else {
                throw new AxisFault("Couldn't find the default throttle policy .the module is invalid ");
            }
        } catch (ThrottleException e) {
            String msg = "Error during processing default throttle policy + system will not works" +
                                e.getMessage();
            log.error(msg);
            throw new AxisFault(msg);
        }
    }
}