<?xml version="1.0" encoding="UTF-8"?>
<!-- 
    /*
    * Copyright 2007 WSO2, Inc. http://www.wso2.org
    *
    * Licensed under the Apache License, Version 2.0 (the "License");
    * you may not use this file except in compliance with the License.
    * You may obtain a copy of the License at
    *
    * http://www.apache.org/licenses/LICENSE-2.0
    *
    * Unless required by applicable law or agreed to in writing, software
    * distributed under the License is distributed on an "AS IS" BASIS,
    * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    * See the License for the specific language governing permissions and
    * limitations under the License.
    */
    
    This stylesheet does some predigestion of WSDL 2.0 documents to make it easier to locate 
    binding and message type information for each operation in a service.  The type information
    is further digested to indicate a mapping into simple function signatures.

    Created by Jonathan Marsh, jonathan@wso2.com
-->
 <xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0"
    xmlns:wsdl="http://www.w3.org/ns/wsdl" xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:wrpc="http://www.w3.org/ns/wsdl/rpc" xmlns:wsoap="http://www.w3.org/ns/wsdl/soap" xmlns:whttp="http://www.w3.org/ns/wsdl/http"
    xmlns:wsdlx="http://www.w3.org/ns/wsdl-extensions" xmlns:wsaw="http://www.w3.org/2006/05/addressing/wsdl"
    exclude-result-prefixes="wsdl wrpc wsoap whttp wsdlx wsaw">
    <xsl:output method="xml" indent="yes"/>
    <!-- copy and paste from http://dev.w3.org/2002/ws/desc/test-suite/results/wsdl-xslt/wsdl-component-model.xslt
         defining collections (nodesets) representing all the top-level types whereever we can find them (through)
         imports, includes, etc.)
         
         copy from here:***************** -->
    <xsl:variable name="root" select="/wsdl:description"/>
    <xsl:variable name="imported-wsdl"
        select="document($root[wsdl:include]/wsdl:include/@location)/wsdl:description |
        document($root[wsdl:import]/wsdl:import/@location)/wsdl:description"/>
    <!-- global (including imports/includes) collections of types that may be referred to later -->
    <xsl:variable name="all-interfaces" select="$root/wsdl:interface |
        $imported-wsdl/wsdl:interface"/>
    <xsl:variable name="all-operations" select="$all-interfaces/wsdl:operation"/>
    <xsl:variable name="all-faults" select="$all-interfaces/wsdl:fault"/>
    <xsl:variable name="all-bindings" select="$root/wsdl:binding | $imported-wsdl/wsdl:binding"/>
    <xsl:variable name="all-services" select="$root/wsdl:service| $imported-wsdl/wsdl:service"/>
    <!--
        Note - limited levels of Schema import/include functionality:
        Supported scenarios:
        - Embedded schemas (multiple schemas OK)
        - Schema directly pointed to by wsdl:types/xs:import/@schemaLocation.
        - Schema directly pointed to by wsdl:types/xs:schema/xs:import/@schemaLocation.
        - Above scenarios, when resulting from direct import/include of WSDL.
        - Chameleon includes (namespace specified on the include, not in the included schema.)
        Unsupported scenarios:
        - Schema imported or included indirectly (except as above).
        - Schema imported without @schemaLocation attribute (no catalog support).
    -->
    <xsl:variable name="imported-schema"
        select="document($root/wsdl:types[xs:import]/xs:import[@schemaLocation and
        not(starts-with(@schemaLocation,'#'))]/@schemaLocation)/xs:schema |
        document($root/wsdl:types/xs:schema[xs:import]/xs:import[@schemaLocation and
        not(starts-with(@schemaLocation,'#'))]/@schemaLocation)/xs:schema |
        document($imported-wsdl/wsdl:types[xs:import]/xs:import[@schemaLocation and
        not(starts-with(@schemaLocation,'#'))]/@schemaLocation)/xs:schema |
        $imported-wsdl/wsdl:types/xs:schema"/>
    <xsl:variable name="included-schema"
        select="document($root/wsdl:types/xs:schema[xs:include]/xs:include/@schemaLocation)/xs:schema"/>
    <!-- global (including imports/includes) collections of types that may be referred to later -->
    <xsl:variable name="all-elements" select="$root/wsdl:types/xs:schema/xs:element |
        $imported-wsdl/wsdl:types/xs:schema/xs:element | $imported-schema/xs:element |
        $included-schema/xs:element"/>
    <xsl:variable name="all-types" select="$root/wsdl:types/xs:schema/xs:simpleType |
        $imported-wsdl/wsdl:types/xs:schema/xs:simpleType | $imported-schema/xs:simpleType |
        $included-schema/xs:simpleType | $root/wsdl:types/xs:schema/xs:complexType |
        $imported-wsdl/wsdl:types/xs:schema/xs:complexType | $imported-schema/xs:complexType
        | $included-schema/xs:complexType"/>
    <!-- to here.***************** -->
    <xsl:template match="wsdl:description">
        <!-- <services> : container for all the services represented in this signature file. -->
        <services>
            <!-- Hoist from endpoints, through bindings, through interfaces, to message schemas.
                 Most of the data in each service will be duplicated but having it in predigested 
                 format is more valuable than duplication. -->
            <xsl:for-each select="$all-services/wsdl:endpoint">
                <xsl:variable name="interfaceLocalName" select="substring-after(../@interface,':')"/>
                <!-- <service> : representing all the capabilities exposed by a single endpoint. 
                     Generate one per endpoint so a downstream processor can choose which 
                     service they'd like to process (possibly dynamically). 
                       @name : name of the service
                       @endpoint : name of the endpoint
                       @address : endopint address (url)
                       @type : canonical token representing the type of binding
                     -->
                <service name="{ancestor::wsdl:service/@name}" endpoint="{@name}"
                    address="{@address}">
                    <!-- resolve indirection to a wsdl:binding -->
                    <xsl:variable name="binding" select="substring-after(@binding,':')"/>
                    <xsl:variable name="thisBinding" select="$all-bindings[@name = $binding]"/>
                    <xsl:for-each select="$thisBinding">
                        <!-- Only the WSDL 2.0 standard bindings are supported. -->
                        <xsl:choose>
                            <xsl:when test="@type='http://www.w3.org/ns/wsdl/soap' and
                                (not(@wsoap:version) or @wsoap:version='1.2')">
                                <xsl:attribute name="type">SOAP12</xsl:attribute>
                            </xsl:when>
                            <xsl:when test="@type='http://www.w3.org/ns/wsdl/http'">
                                <xsl:attribute name="type">HTTP</xsl:attribute>
                            </xsl:when>
                            <xsl:when test="@type='http://www.w3.org/ns/wsdl/soap' and
                                @wsoap:version='1.1'">
                                <xsl:attribute name="type">SOAP11</xsl:attribute>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:comment>Unknown binding type: </xsl:comment>
                                <xsl:copy-of select="."/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:for-each>
                    <documentation>
                        <xsl:copy-of select="/wsdl:description/wsdl:documentation/node()"/>
                    </documentation>
                    <!-- <operations> : container for operations exposed by the interface for this endpoint.
                            @name : interface name
                    -->
                    <operations name="{$interfaceLocalName}">
                        <xsl:for-each select="$all-interfaces[@name = $interfaceLocalName]">
                            <!-- returns a list of all the interface names, including extended interfaces, in the form
                                 '[interfacename1][interfacename2]...'.  Workaround for the lack of sets of strings in XSLT.-->
                            <xsl:variable name="interfaces-extended-by-this-one">
                                <xsl:call-template name="interface-list">
                                    <xsl:with-param name="interface" select="."/>
                                </xsl:call-template>
                            </xsl:variable>
                            <!-- loop through all the operations of this interface, or any interface it extends.  -->
                            <xsl:for-each select="$all-operations[contains($interfaces-extended-by-this-one,
                                concat('[',ancestor::wsdl:interface/@name,']'))]">
                                <xsl:variable name="thisInterfaceOperation" select="."/>
                                <!-- <operation> : represents the details for a particular operation.
                                       @name : operation name
                                       @pattern : uri representing the MEP
                                       @safe : canonical value of the safety property
                                -->
                                <operation name="{@name}" pattern="{@pattern}" safe="{@wsdlx:safe = 'true' or @wsdlx:safe = '1'}">
                                    <documentation>
                                        <xsl:copy-of select="wsdl:documentation/node()"/>
                                    </documentation>
                                    <!-- <signature> : capture the details of the schema types relevant to generating a function signature
                                           @method : how the signature was computed ('rpc-signature' hints, or 'inference' from the 
                                                     schema structures themselves.)
                                    -->
                                    <signature>
                                        <xsl:choose>
                                            <xsl:when test="@wrpc:signature">
                                                <xsl:attribute name="method">rpc-signature</xsl:attribute>
                                                <xsl:call-template name="rpc-signature">
                                                    <xsl:with-param name="operation" select="."/>
                                                </xsl:call-template>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:attribute name="method">inference</xsl:attribute>
                                                <xsl:call-template name="infer-params">
                                                    <xsl:with-param name="operation" select="."/>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </signature>
                                    <xsl:for-each select="$thisBinding/wsdl:operation[substring-after(@ref, ':') = current()/@name]">
                                        <!-- <binding-details> : record any other information specific to an operation that
                                                                 will enable successful communication with the endpoint.
                                        -->
                                        <binding-details>
                                            <!-- soap-only binding details include:
                                                   @wsawaction : WS-Addressing explicit action value
                                                   @soapaction : explicitly declared soap action value
                                                 soap and http binding details include:
                                                   @method : {http method} property
                                                   @httplocation : {http location} property
                                                   @httignoreUncited : {http location ignore uncited} property
                                                   @httpqueryParameterSeparator : cascading value for the {http query parameter separator} property
                                                 http-only binding details include:
                                                   @httpinputSerialization : {http input serialization} property
                                            -->
                                            <xsl:if test="../@type = 'http://www.w3.org/ns/wsdl/soap'">
                                                <xsl:if test="$thisInterfaceOperation/wsdl:input/@wsaw:Action">
                                                    <xsl:attribute name="wsawaction"><xsl:value-of select="$thisInterfaceOperation/wsdl:input/@wsaw:Action"/></xsl:attribute>
                                                </xsl:if>
                                                <xsl:if test="@wsoap:action">
                                                    <xsl:attribute name="soapaction"><xsl:value-of select="@wsoap:action"/></xsl:attribute>
                                                </xsl:if>
                                                <!-- method can't be explicitly set for a soap binding, but can be implied by the SOAP 1.2 Response MEP. -->
                                                <xsl:if test="../@wsoap:version='1.2' and ../@wsoap:protocol='http://www.w3.org/2003/05/soap/mep/soap-response/' and (@wsoap:mep='http://www.w3.org/2003/05/soap/mep/soap-response/' or (not(@wsoap:mep) and ../@wsoap:mepDefault='http://www.w3.org/2003/05/soap/mep/soap-response/'))">
                                                    <xsl:attribute name="httpmethod">GET</xsl:attribute>
                                                </xsl:if>
                                            </xsl:if>
                                            <xsl:if test="@whttp:location">
                                                <xsl:attribute name="httplocation"><xsl:value-of select="@whttp:location"/></xsl:attribute>
                                            </xsl:if>
                                            <xsl:if test="@whttp:ignoreUncited">
                                                <xsl:attribute name="httpignoreUncited"><xsl:value-of select="@whttp:ignoreUncited"/></xsl:attribute>
                                            </xsl:if>
                                            <xsl:choose>
                                                <xsl:when test="@whttp:queryParameterSeparator">
                                                    <xsl:attribute name="httpqueryParameterSeparator"><xsl:value-of select="@whttp:queryParameterSeparator"/></xsl:attribute>
                                                </xsl:when>
                                                <xsl:when test="../@whttp:queryParameterSeparatorDefault">
                                                    <xsl:attribute name="httpqueryParameterSeparator"><xsl:value-of select="../@whttp:queryParameterSeparatorDefault"/></xsl:attribute>
                                                </xsl:when>
                                            </xsl:choose> 
                                            <xsl:if test="../@type = 'http://www.w3.org/ns/wsdl/http'">
                                                <xsl:if test="@whttp:inputSerialization">
                                                    <xsl:attribute name="httpinputSerialization"><xsl:value-of select="@whttp:inputSerialization"/></xsl:attribute>
                                                </xsl:if>
                                                <!-- HTTP method can be set explicitly, defaulted, or implied by wsdlx:safe -->
                                                <xsl:choose>
                                                    <xsl:when test="@whttp:method">
                                                        <xsl:attribute name="httpmethod"><xsl:value-of select="@whttp:method"/></xsl:attribute>
                                                    </xsl:when>
                                                    <xsl:when test="../@whttp:methodDefault">
                                                        <xsl:attribute name="httpmethod"><xsl:value-of select="../@whttp:methodDefault"/></xsl:attribute>
                                                    </xsl:when>
                                                    <xsl:when test="$thisInterfaceOperation/@wsdlx:safe = 'true' or $thisInterfaceOperation/@wsdlx:safe = '1'">
                                                        <xsl:attribute name="httpmethod">GET</xsl:attribute>
                                                    </xsl:when>
                                                </xsl:choose> 
                                            </xsl:if>
                                            <xsl:for-each select="*/wsoap:header">
                                                <soapheader for="{local-name(parent::*)}">
                                                    <xsl:copy-of select="@required | @mustUnderstand"/>
                                                    <xsl:attribute name="type">
                                                        <xsl:call-template name="local-name">
                                                            <xsl:with-param name="qname" select="@element"/>
                                                        </xsl:call-template>
                                                    </xsl:attribute>
                                                    <xsl:attribute name="type-namespace">
                                                        <xsl:value-of select="namespace::*[local-name() = substring-before(@element,':')]"/>
                                                    </xsl:attribute>
                                                    <xsl:copy-of select="*"/>
                                                </soapheader>
                                            </xsl:for-each>
                                            <xsl:for-each select="whttp:header">
                                                <httpheader for="{local-name(parent::*)}">
                                                    <xsl:copy-of select="@required"/>
                                                    <xsl:attribute name="type">
                                                        <xsl:call-template name="local-name">
                                                            <xsl:with-param name="qname" select="@element"/>
                                                        </xsl:call-template>
                                                    </xsl:attribute>
                                                    <xsl:attribute name="type-namespace">
                                                        <xsl:value-of select="namespace::*[local-name() = substring-before(@element,':')]"/>
                                                    </xsl:attribute>
                                                    <xsl:copy-of select="*"/>
                                                </httpheader>
                                            </xsl:for-each>
                                        </binding-details>
                                    </xsl:for-each>
                                </operation>
                            </xsl:for-each>
                        </xsl:for-each>
                    </operations>
                </service>
            </xsl:for-each>
        </services>
    </xsl:template>

    <!-- A pair of recursive templates to list extended interfaces 
         since we operat on flat lists, just having a list of names is sufficient.  An actual nodeset 
         would be quite difficult to construct without copying (which loses context and the ability
         to further query the result in a standard way). Result should look like:
            [interfacename1][interfacename2]...
         The square brackets provide convenient delimiters so that we can search the list for accurate
         matches (including delimiters) and not get false positives when one interface name contains
         another.
    -->
    <xsl:template name="interface-list">
        <xsl:param name="interface"/>
        <xsl:text>[</xsl:text>
        <xsl:value-of select="$interface/@name"/>
        <xsl:text>]</xsl:text>
        <!-- append the names of any interfaces that this one extends -->
        <xsl:if test="$interface/@extends">
            <xsl:call-template name="extended-interface-list">
                <xsl:with-param name="interface-names" select="$interface/@extends"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    <xsl:template name="extended-interface-list">
        <xsl:param name="interface-names"/>
        <!-- split off the first QName in a space-delimited list -->
        <xsl:variable name="qnamesplit" select="concat(normalize-space($interface-names),' ')"/>
        <xsl:variable name="firstqname" select="substring-before($qnamesplit,' ')"/>
        <xsl:variable name="remainder" select="substring-after($qnamesplit,' ')"/>
        <xsl:variable name="interface-local-name">
            <xsl:call-template name="local-name">
                <xsl:with-param name="qname" select="$firstqname"/>
            </xsl:call-template>
        </xsl:variable>
        <!-- add the first QName to the list (recursively processing any other extends attributes) -->
        <xsl:call-template name="interface-list">
            <xsl:with-param name="interface" select="$all-interfaces[@name=$interface-local-name]"/>
        </xsl:call-template>
        <!-- recursively process the remainder of the list until nothing is left. -->
        <xsl:if test="$remainder != ''">
            <xsl:call-template name="extended-interface-list">
                <xsl:with-param name="interface-names" select="$remainder"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>


    <!-- Template to extract relevant details about the signature from the wrpc:signature extension. 
           $signature : string representing the value of the wrpc:signature extension.
           $operation : node representing the operation for which to generate parameters -->
     <xsl:template name="rpc-signature">
        <xsl:param name="operation"/>

        <!-- inputs -->
        <xsl:call-template name="generate-rpc-params">
            <xsl:with-param name="signature" select="@wrpc:signature"/>
            <xsl:with-param name="direction" select="'#in'"/>
            <xsl:with-param name="element-name" select="$operation/wsdl:input/@element"/>
            <xsl:with-param name="element-namespace" select="$operation/wsdl:input/namespace::*[local-name() =
                substring-before($operation/wsdl:input/@element,':')]"/>
        </xsl:call-template>
         
        <!-- outputs -->
        <!-- The following <if> implies we only support the in-out, in-only, and robust-in-only MEPs -->
        <xsl:if test="$operation/@pattern = 'http://www.w3.org/ns/wsdl/in-out'">
            <xsl:call-template name="generate-rpc-params">
                <xsl:with-param name="signature" select="@wrpc:signature"/>
                <xsl:with-param name="direction" select="'#return'"/>
                <xsl:with-param name="element-name" select="$operation/wsdl:output/@element"/>
                <xsl:with-param name="element-namespace" select="$operation/wsdl:output/namespace::*[local-name() =
                    substring-before($operation/wsdl:output/@element,':')]"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
     <!-- Helper template for "rpc-signature", generating the <params>/<returns> element 
            $signature : rpc-signature string to use
            $direction : '#in' | '#return' signifying input/return parameter types 
            $element-name : local name of the element declaration associated with this parameter 
            $element-namespace : namespace of the element declaration associated with this parameter.
     -->
     <xsl:template name="generate-rpc-params">
        <xsl:param name="signature"/>
        <xsl:param name="direction"/>
        <xsl:param name="element-name"/>
        <xsl:param name="element-namespace"/>
        <!-- <params>/<returns> : wrapper for input/output parameters 
                @wrapper-element : localName of the RPC wrapper element (implied by the WSDL 2.0 RPC style) if there is one
                @wrapper-element-ns : namespace URI of the RPC wrapper element
             <param> : represent a parameter in a function signature
                @name : name of the parameter
                @type : localName of schema type for the parameter
                @type-namespace : namespace URI of schema type for the parameter
                @token : what type of parameter this is (#in, #return, #any wildcard)
                @simple : whether the type is simple or complex
        -->
        <xsl:variable name="containername">
            <xsl:choose>
                <xsl:when test="$direction = '#in'">params</xsl:when>
                <xsl:otherwise>returns</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:element name="{$containername}">
            <!-- Find the element declaration and its type definition. -->
            <xsl:variable name="this-element"
                select="$all-elements[@name=substring-after($element-name,':')]"/>
            <xsl:variable name="this-type" select="$this-element[not(@type)]/xs:complexType |
                $all-types[@name=substring-after($this-element/@type,':')]"/>

            <xsl:if test="contains($element-name, ':')">
                <xsl:attribute name="wrapper-element">
                    <xsl:value-of select="substring-after($element-name,':')"/>
                </xsl:attribute>
                <xsl:attribute name="wrapper-element-ns">
                    <xsl:value-of select="$element-namespace"/>
                </xsl:attribute>
            </xsl:if>
            <xsl:call-template name="rpc-sig-param">
                <xsl:with-param name="signature" select="$signature"/>
                <xsl:with-param name="direction" select="$direction"/>
                <xsl:with-param name="wrapper-type" select="$this-type"/>
            </xsl:call-template>

            <!-- Handle any element wildcards -->
            <xsl:for-each select="$this-type/xs:sequence/xs:any">
                <param token="#any">
                    <xsl:copy-of select="@minOccurs | @maxOccurs"/>
                    <xsl:if test="not(@minOccurs)"><xsl:attribute name="minOccurs">1</xsl:attribute></xsl:if>
                    <xsl:if test="not(@maxOccurs)"><xsl:attribute name="maxOccurs">1</xsl:attribute></xsl:if>
                </param>
            </xsl:for-each>
        </xsl:element>
    </xsl:template>
    
    <!-- Helper template for "generate-rpc-params", generating the <param/> element representing a complex type
         (possibly recursive)
            $signature : the rpc signature string
            $direction : '#in' | '#return' signifying input/return parameter types 
            $this-element : node representing the element delcaration associated with this parameter 
            $this-type : node representing the type definition associated with this parameter.
    -->
    <xsl:template name="rpc-sig-param">
        <xsl:param name="signature"/>
        <xsl:param name="direction"/>
        <xsl:param name="wrapper-type"/>
        
        <!-- strip off the first pair of tokens, process and recurse.  So first, check that we're not done. -->
        <xsl:if test="$signature != ''">
            <!-- qname is the first item, token is the second. -->
            <xsl:variable name="qname" select="substring-before($signature,' ')"/>
            <xsl:variable name="token" select="substring-before(substring-after(concat($signature,' '),' '),' ')"/>
            
            <xsl:variable name="this-element" select="$wrapper-type/xs:sequence/xs:element[@name=$qname]"/>

            <!-- This list is processed twice, filtered by $direction.  If we're looking for an in and this sig pair
                 is an input, we'll process it.  Or if we're looking for returns and this sig pair represents a 
                 return, we'll process it. -->
            <xsl:if test="($direction='#in' and ($token='#in' or $token='#inout')) or ($direction='#return' and ($token='#return' or $token='#out' or $token='#inout'))">
                <xsl:call-template name="complex-param">
                    <xsl:with-param name="direction" select="$direction"/>
                    <xsl:with-param name="this-element" select="$this-element"/>
                </xsl:call-template>
            </xsl:if>
            
            <!-- recurse, stripping the sig pair we just processed off the sig. Looping XSLT-style. -->
            <xsl:call-template name="rpc-sig-param">
                <xsl:with-param name="signature"
                    select="substring-after(substring-after($signature,' '),' ')"/>
                <xsl:with-param name="direction" select="$direction"/>
                <xsl:with-param name="wrapper-type" select="$wrapper-type"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>

    <!-- Template to extract relevant details about the signature from an rpc-style service, but 
         lacking the optional wrpc:signature extension. 
           $operation : node representing the operation for which to generate parameters -->
     <xsl:template name="infer-params">
        <xsl:param name="operation"/>

        <!-- inputs -->
        <xsl:call-template name="generate-params">
            <xsl:with-param name="direction" select="'#in'"/>
            <xsl:with-param name="element-name" select="$operation/wsdl:input/@element"/>
            <xsl:with-param name="element-namespace" select="$operation/wsdl:input/namespace::*[local-name() =
                substring-before($operation/wsdl:input/@element,':')]"/>
        </xsl:call-template>

        <!-- outputs -->
        <!-- The following <if> implies we only support the in-out, in-only, and robust-in-only MEPs -->
        <xsl:if test="$operation/@pattern = 'http://www.w3.org/ns/wsdl/in-out'">
            <xsl:call-template name="generate-params">
                <xsl:with-param name="direction" select="'#return'"/>
                <xsl:with-param name="element-name" select="$operation/wsdl:output/@element"/>
                <xsl:with-param name="element-namespace" select="$operation/wsdl:output/namespace::*[local-name() =
                    substring-before($operation/wsdl:output/@element,':')]"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <!-- Helper template for "infer-params", generating the <params>/<returns> element -->
    <xsl:template name="generate-params">
        <!-- $direction : '#in' | '#return' signifying input/return parameter types 
             $element-name : local name of the element declaration associated with this parameter 
             $element-namespace : namespace of the element declaration associated with this parameter.
        -->
        <xsl:param name="direction"/>
        <xsl:param name="element-name"/>
        <xsl:param name="element-namespace"/>
        <!-- <params>/<returns> : wrapper for input/output parameters 
                @wrapper-element : localName of the RPC wrapper element (implied by the WSDL 2.0 RPC style) if there is one
                @wrapper-element-ns : namespace URI of the RPC wrapper element
             <param> : represent a parameter in a function signature
                @name : name of the parameter
                @type : localName of schema type for the parameter
                @type-namespace : namespace URI of schema type for the parameter
                @token : what type of parameter this is (#in, #return, #any wildcard)
                @simple : whether the type is simple or complex
        -->
        <xsl:variable name="containername">
            <xsl:choose>
                <xsl:when test="$direction = '#in'">params</xsl:when>
                <xsl:otherwise>returns</xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:element name="{$containername}">
            <xsl:choose>
                <!-- If there is no expected input/output, don't generate any <param>s.  -->
                <xsl:when test="$element-name = '#none'"/>

                <!-- If any single element is expected, generate a default xs:anyType parameter type. -->
                <xsl:when test="$element-name = '#any'">
                    <param token="{$direction}" type="anyType"
                        type-namespace="http://www.w3.org/2001/XMLSchema" simple="no"/>
                </xsl:when>

                <!-- Otherwise there is a referenced XML Schema type. -->
                <xsl:otherwise>
                    <!-- Find the element declaration and its type definition. -->
                    <xsl:variable name="this-element"
                        select="$all-elements[@name=$element-name or @name=substring-after($element-name,':')]"/>
                    <xsl:variable name="this-type" select="$this-element[not(@type)]/xs:complexType
                        | $all-types[@name=substring-after($this-element/@type,':')]"/>
                    <xsl:choose>

                        <!-- If there is a referenced complex type definition -->
                        <xsl:when test="$this-type">
                            <!-- See if the complexType conforms to some of the rquirements of the WSDL 2.0 RPC style,
                                 namely, a sequence of local elements, with xs:any at the end if anywhere. -->
                            <xsl:if test="$this-type[not(xs:simpleContent or xs:complexContent or xs:group or xs:all or xs:choice or xs:attribute or xs:attributeGroup or xs:anyAttribute or xs:any[2])][not(xs:any/following-sibling::xs:*)]/xs:sequence[not(xs:element/@ref)]">
                                <xsl:attribute name="wrapper-element">
                                    <xsl:call-template name="local-name">
                                        <xsl:with-param name="qname" select="$element-name"/>
                                    </xsl:call-template>
                                </xsl:attribute>
                                <xsl:attribute name="wrapper-element-ns">
                                    <xsl:value-of select="$element-namespace"/>
                                </xsl:attribute>
                            </xsl:if>
                            <!-- Generate the <param> for a complex type. -->
                            <!-- local element children in a sequence are interpreted as parameters in the WSDL 2.0 RPC style -->
                            <xsl:for-each select="$this-type/xs:sequence/xs:element">
                                <xsl:call-template name="complex-param">
                                    <xsl:with-param name="this-element" select="."/>
                                    <xsl:with-param name="direction" select="$direction"/>
                                </xsl:call-template>
                            </xsl:for-each>
                            
                            <!-- Handle any wildcards -->
                            <xsl:for-each select="$this-type/xs:sequence/xs:any">
                                <param token="#any">
                                    <xsl:copy-of select="@minOccurs | @maxOccurs"/>
                                    <xsl:if test="not(@minOccurs)"><xsl:attribute name="minOccurs">1</xsl:attribute></xsl:if>
                                    <xsl:if test="not(@maxOccurs)"><xsl:attribute name="maxOccurs">1</xsl:attribute></xsl:if>                                    
                                </param>
                            </xsl:for-each>
                            
                        </xsl:when>

                        <!-- No referenced definition, is the type a built-in schema type other than xs:anyType? -->
                        <xsl:when test="$this-element/namespace::*[local-name() = substring-before($this-element/@type,':')][. = 'http://www.w3.org/2001/XMLSchema'] and substring-after($this-element/@type,':') != 'anyType'">
                            <param name="{substring-after($element-name,':')}" token="#in"
                                type="{substring-after($this-element/@type,':')}" type-namespace="http://www.w3.org/2001/XMLSchema" simple="yes">
                                <xsl:if test="$this-element/ancestor::xs:schema/@elementFormDefault != 'unqualified'">
                                    <xsl:attribute name="targetNamespace">
                                        <xsl:value-of select="$this-element/ancestor::xs:schema/@targetNamespace"/>
                                    </xsl:attribute>
                                </xsl:if>
                            </param>                                
                        </xsl:when>

                        <!-- Either xs:anyType or some other complex type that doesn't conform to the RPC style.
                             Fall back to xs:anyType.  -->
                        <xsl:otherwise>
                            <param name="{substring-after($element-name,':')}" token="#in"
                                type="anyType" type-namespace="http://www.w3.org/2001/XMLSchema" simple="no">
                                <xsl:if test="$this-element/ancestor::xs:schema/@elementFormDefault != 'unqualified'">
                                    <xsl:attribute name="targetNamespace">
                                        <xsl:value-of select="$this-element/ancestor::xs:schema/@targetNamespace"/>
                                    </xsl:attribute>
                                </xsl:if>
                            </param>                                
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:element>
    </xsl:template>

    <!-- Helper template for "infer-params", generating the <param/> element representing a complex type
         (possibly recursive) 
            $direction : '#in' | '#return' signifying input/return parameter types 
            $this-element : node representing the element delcaration associated with this parameter 
    -->
    <xsl:template name="complex-param">
        <xsl:param name="direction"/>
        <xsl:param name="this-element"/>

        <param token="{$direction}">
            <xsl:if test="$this-element/ancestor::xs:schema/@elementFormDefault != 'unqualified'">
                <xsl:attribute name="targetNamespace">
                    <xsl:value-of select="$this-element/ancestor::xs:schema/@targetNamespace"/>
                </xsl:attribute>
            </xsl:if>
            <!-- set the context for attribute scraping -->
            <xsl:for-each select="$this-element">
                <xsl:copy-of select="@nillable | @minOccurs | @maxOccurs"/>
                <!-- canonical values for minOccurs and maxOccurs -->
                <xsl:if test="not(@minOccurs)"><xsl:attribute name="minOccurs">1</xsl:attribute></xsl:if>
                <xsl:if test="not(@maxOccurs)"><xsl:attribute name="maxOccurs">1</xsl:attribute></xsl:if>
                <xsl:attribute name="name">
                    <xsl:call-template name="elementname-2-paramname">
                        <xsl:with-param name="elementname" select="@name | @ref"/>
                    </xsl:call-template>
                </xsl:attribute>
                <xsl:choose>
                    <!-- If the type is a built-in schema type -->
                    <xsl:when test="namespace::*[local-name() = substring-before(current()/@type,':')] = 'http://www.w3.org/2001/XMLSchema'">
                        <xsl:attribute name="type"><xsl:value-of select="substring-after(@type,':')"/></xsl:attribute>
                        <xsl:attribute name="type-namespace">http://www.w3.org/2001/XMLSchema</xsl:attribute>
                        <xsl:attribute name="simple">
                            <xsl:choose>
                                <xsl:when test="substring-after(@type,':') != 'anyType'">yes</xsl:when>
                                <xsl:otherwise>no</xsl:otherwise>
                            </xsl:choose>
                        </xsl:attribute>
                    </xsl:when>
    
                    <!-- Not in a built-in type, do some more inference -->
                    <xsl:otherwise>
                        <xsl:if test="@type">
                            <xsl:attribute name="type">
                                <xsl:call-template name="local-name">
                                    <xsl:with-param name="qname" select="@type"/>
                                </xsl:call-template>
                            </xsl:attribute>                            
                            <xsl:attribute name="type-namespace"><xsl:value-of select="namespace::*[local-name() = substring-before(current()/@type,':')]"/></xsl:attribute>
                        </xsl:if>

                        <!-- this-type-def will either point to the named type (if there is a @type) or to the child type (if there is a @name) -->
                        <xsl:variable name="this-type-def" select="$all-types[@name=substring-after(current()/@type,':')] | xs:simpleType | xs:complexType"/>
                        <xsl:attribute name="simple">
                            <xsl:choose>
                                <xsl:when test="local-name($this-type-def) = 'simpleType'">yes</xsl:when>
                                <xsl:otherwise>no</xsl:otherwise>
                            </xsl:choose>
                        </xsl:attribute>
                        
                        <!-- look into the type and see if we can determine whether it's a restriction of another type -->
                        <xsl:if test="$this-type-def/xs:restriction">
                            <xsl:for-each select="$this-type-def/xs:restriction">
                                <xsl:attribute name="restriction-of"><xsl:value-of select="substring-after(@base,':')"/></xsl:attribute>
                                <xsl:attribute name="restriction-namespace"><xsl:value-of select="namespace::*[local-name() =
                                substring-before(current()/@base,':')]"/></xsl:attribute>
                            </xsl:for-each>
                        </xsl:if>
                        
                        <!-- Does the type represent an enumeration of strings (as generated by the Mashup Server? 
                             If so capture the possible values.  -->
                        <xsl:if test="$this-type-def/xs:restriction[substring-after(@base,':') = 'string']/xs:enumeration">
                            <!--  potential bug: doesn't double-check the xs namespace on this -->
                            <xsl:for-each select="$this-type-def/xs:restriction/xs:enumeration">
                                <enumeration value="{@value}"/>
                            </xsl:for-each>
                        </xsl:if>
                        
                        <!-- If the type itself is complex, try to recurse and infer sub-params, helpful for generating nested forms -->
                        <xsl:if test="local-name($this-type-def) = 'complexType'">
                            <!-- local element children in a sequence are interpreted as parameters in the WSDL 2.0 RPC style -->
                            <xsl:for-each select="$this-type-def/xs:sequence/xs:element">
                                <xsl:call-template name="complex-param">
                                    <xsl:with-param name="this-element" select="."/>
                                    <xsl:with-param name="direction" select="$direction"/>
                                </xsl:call-template>
                            </xsl:for-each>
                            
                            <!-- Handle any wildcards -->
                            <xsl:for-each select="$this-type-def/xs:sequence/xs:any">
                                <param token="#any">
                                    <xsl:copy-of select="@minOccurs | @maxOccurs"/>
                                    <xsl:if test="not(@minOccurs)"><xsl:attribute name="minOccurs">1</xsl:attribute></xsl:if>
                                    <xsl:if test="not(@maxOccurs)"><xsl:attribute name="maxOccurs">1</xsl:attribute></xsl:if>                                    
                                </param>
                            </xsl:for-each>
                        </xsl:if>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:for-each>
        </param>
    </xsl:template>

    <!-- extract a localName from a QName -->
    <xsl:template name="local-name">
        <xsl:param name="qname"/>
        <xsl:choose>
            <xsl:when test="contains($qname,':')">
                <xsl:value-of select="substring-after($qname,':')"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="$qname"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- perform some XML->Javascript (and other programming languages) mapping -->
    <xsl:template name="elementname-2-paramname">
        <xsl:param name="elementname"/>
        <xsl:value-of select="translate($elementname,':.-','___')"/>
    </xsl:template>
</xsl:stylesheet>
