/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.wso2.wsas.sample.xkms.reissue;

import java.io.File;
import java.io.FileInputStream;
import java.security.Key;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Calendar;
import java.util.List;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.xml.security.keys.KeyInfo;
import org.wso2.wsas.sample.xkms.utils.Utils;
import org.wso2.xkms2.Authentication;
import org.wso2.xkms2.KeyBinding;
import org.wso2.xkms2.ReissueKeyBinding;
import org.wso2.xkms2.ReissueRequest;
import org.wso2.xkms2.ReissueResult;
import org.wso2.xkms2.RespondWith;
import org.wso2.xkms2.Status;
import org.wso2.xkms2.StatusValue;
import org.wso2.xkms2.ValidReason;
import org.wso2.xkms2.builder.ReissueResultBuilder;
import org.wso2.xkms2.util.XKMSKeyUtil;

public class ReissueServiceDemo {

    private static String PASS_PHRASE = "secret";

    private static String XKMS_SERVICE_URL = "http://127.0.0.1:9762/services/xkms";

    private static String WSO2_HOME = System.getProperty("wso2wsas.home");

    private static String separator = File.separator;

    static {
        org.apache.xml.security.Init.init();
    }

    public static void main(String[] args) throws Exception {
        String location = WSO2_HOME + separator + "samples" + separator
                + "XKMS" + separator + "conf" + separator + "keystore.jks";
        String password = "password";

        // loading client keystore
        FileInputStream fis = new FileInputStream(location);
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(fis, password.toCharArray());

        ReissueRequest request = Utils.createReissueRequest();
        request.setServiceURI(XKMS_SERVICE_URL);

        // setting the authentication info
        Authentication authentication = Utils.createAuthenticate();
        Key authKey = XKMSKeyUtil.getAuthenticationKey(PASS_PHRASE);
        authentication.setKeyBindingAuthenticationKey(authKey);
        request.setAuthentication(authentication);

        // setting the Keybinding info
        ReissueKeyBinding keyBinding = Utils.createReissueKeyBinding();
        // setting the
        X509Certificate cert = (X509Certificate) keyStore.getCertificate("bob");
        keyBinding.setCertValue(cert);

        // setting
        Calendar start = Calendar.getInstance();
        Calendar expiry = Calendar.getInstance();
        expiry.add(Calendar.DAY_OF_YEAR, 90);
        keyBinding.setValidityInterval(start, expiry);

        // setting the status
        Status status = new Status();
        status.setStatusValue(StatusValue.VALID);
        status.addValidReason(ValidReason.SIGNATURE);
        keyBinding.setStatus(status);

        request.setReissueKeyBinding(keyBinding);

        // setting private key as proof that the client pocess both public and
        // private keys
        PrivateKey privateKey = (PrivateKey) keyStore.getKey("bob", "password"
                .toCharArray());
        request.setProofOfPossessionKey(privateKey);

        request.addRespondWith(RespondWith.X_509_CERT);

        OMElement element = getAsOMElement(request);
        OMElement result = Utils.sendReceive(element, XKMS_SERVICE_URL);
        result = Utils.buildElement(result);

        ReissueResult reissueResult = getRegisterResult(result);
        List keybindings = reissueResult.getKeyBindings();
        KeyBinding keybinding = (KeyBinding) keybindings.get(0);

        KeyInfo keyInfo = keybinding.getKeyInfo();
        X509Certificate cert2 = keyInfo.getX509Certificate();
        if (cert2 != null) {
            System.out.println("X509Certificate : ");
            Utils.printCert(cert);
        }
    }

    private static OMElement getAsOMElement(ReissueRequest reissueRequest)
            throws Exception {
        return reissueRequest.serialize(DOOMAbstractFactory.getOMFactory());
    }

    private static ReissueResult getRegisterResult(OMElement reissueResultElem)
            throws Exception {
        return (ReissueResult) ReissueResultBuilder.INSTANCE
                .buildElement(reissueResultElem);
    }

}
