/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.wso2.wsas.sample.xkms.register;

import java.security.Key;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.util.List;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.xml.security.keys.KeyInfo;
import org.apache.xml.security.keys.content.KeyName;
import org.wso2.wsas.sample.xkms.utils.Utils;
import org.wso2.xkms2.Authentication;
import org.wso2.xkms2.KeyBinding;
import org.wso2.xkms2.KeyUsage;
import org.wso2.xkms2.PrototypeKeyBinding;
import org.wso2.xkms2.RegisterRequest;
import org.wso2.xkms2.RegisterResult;
import org.wso2.xkms2.RespondWith;
import org.wso2.xkms2.UseKeyWith;
import org.wso2.xkms2.builder.RegisterResultBuilder;
import org.wso2.xkms2.util.XKMSKeyUtil;

/**
 * This sample demonstrates how to register a client-side generated keypair and
 * obtain the corresponding X509Certificate from a XKMS registration service.
 */
public class RegisterServiceDemo1 {

    private static String XKMS_SERVICE_URL = "http://127.0.0.1:9762/services/xkms";
    
    private static String PASS_PHRASE = "secret";

    static {
        org.apache.xml.security.Init.init();
    }

    public static void main(String[] args) throws Exception {

        KeyPair keyPair = Utils.generateRSAKeyPair();

        RegisterRequest request = Utils.createRegisterRequest();
        request.setServiceURI(XKMS_SERVICE_URL);

        // setting the authentication info
        Authentication authentication = Utils.createAuthenticate();
        Key authKey = XKMSKeyUtil.getAuthenticationKey(PASS_PHRASE);
        authentication.setKeyBindingAuthenticationKey(authKey);
        request.setAuthentication(authentication);

        // setting the Keybinding info
        PrototypeKeyBinding keyBinding = Utils.createPrototypeKeyBinding();
        // setting the public key to which an X509Certificate should be issued.
        keyBinding.setKeyValue(keyPair.getPublic());

        // setting usages of the key
        keyBinding.addKeyUsage(KeyUsage.SIGNATURE);

        keyBinding.addUseKeyWith(UseKeyWith.PKIX,
                "C=US, ST=NY, L=NYC, O=Yahoo, OU=XKMS, CN=Motukuru");

        request.setPrototypeKeyBinding(keyBinding);

        // setting private key as proof that the client pocess both public and
        // private keys
        request.setProofOfPocessionKey(keyPair.getPrivate());

        request.addRespondWith(RespondWith.KEY_NAME);
        request.addRespondWith(RespondWith.KEY_VALUE);
        request.addRespondWith(RespondWith.X_509_CERT);

        OMElement element = getAsOMElement(request);
        OMElement result = Utils.sendReceive(element, XKMS_SERVICE_URL);
        result = Utils.buildElement(result);

        RegisterResult registerResult = getRegisterResult(result);
        List keybindings = registerResult.getKeyBindings();
        KeyBinding keybinding = (KeyBinding) keybindings.get(0);

        KeyInfo keyInfo = keybinding.getKeyInfo();

        KeyName keyName = keyInfo.itemKeyName(0);
        if (keyName != null) {
            System.out.println("KeyName : " + keyName.getKeyName());
        }

        X509Certificate cert = keyInfo.getX509Certificate();
        if (cert != null) {
            System.out.println("X509Certificate : ");
            Utils.printCert(cert);
        }
    }

    private static OMElement getAsOMElement(RegisterRequest registerRequest)
            throws Exception {
        return registerRequest.serialize(DOOMAbstractFactory.getOMFactory());
    }

    private static RegisterResult getRegisterResult(OMElement registerResult)
            throws Exception {
        return (RegisterResult) RegisterResultBuilder.INSTANCE
                .buildElement(registerResult);
    }
}
