/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.sample.javascript;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.soap.SOAPEnvelope;
import org.apache.axiom.soap.SOAPFactory;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.MessageReceiver;
import org.apache.axis2.i18n.Messages;
import org.apache.axis2.receivers.AbstractInOutSyncMessageReceiver;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import java.io.*;

/**
 * JavaScriptReceiver processes the incoming message by sending it to the
 * JavaScriptEngine and composes the outgoing message.
 */
public class JavascriptMessageReceiver extends AbstractInOutSyncMessageReceiver
        implements MessageReceiver {

    public void invokeBusinessLogic(MessageContext inMessage,
                                    MessageContext outMessage) throws AxisFault {

        JavascriptEngine engine = new JavascriptEngine();
        try {

            //Get the method, arguments and the reader from the MessageContext
            String method = getJSMethod(inMessage);
            String args = getArgs(inMessage);
            Reader reader = readJS(inMessage);

            if (reader == null) {
                throw new AxisFault("Unable to load JavaScript file");
            }
            if (method == null) {
                throw new AxisFault("Unable to read the method");
            }

            //Get the result from executing the javascript file
            Object obj = engine.call(method, reader, args);
            if (obj == null) {
                throw new AxisFault(Messages.getMessage("JavaScriptNoanswer"));
            }

            //Create the out-going message
            SOAPFactory fac;
            if (inMessage.isSOAP11()) {
                fac = OMAbstractFactory.getSOAP11Factory();
            } else {
                fac = OMAbstractFactory.getSOAP12Factory();
            }
            SOAPEnvelope envelope = fac.getDefaultEnvelope();
            OMNamespace ns = fac.createOMNamespace("http://soapenc/", "res");
            OMElement returnElement = fac.createOMElement("return", null);
            OMElement responseElement = fac.createOMElement(method + "Response", ns);
            String outMessageString = obj.toString();

            responseElement.addChild(getpayLoad(outMessageString));
            returnElement.addChild(responseElement);
            envelope.getBody().addChild(returnElement);
            outMessage.setEnvelope(envelope);
        } catch (XMLStreamException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * @param str The string to embed in the OMElement
     * @return an OMElement
     * @throws XMLStreamException
     */
    private OMElement getpayLoad(String str) throws XMLStreamException {
        XMLStreamReader xmlReader =
                XMLInputFactory.newInstance().createXMLStreamReader(
                        new ByteArrayInputStream(str.getBytes()));
        OMFactory fac = OMAbstractFactory.getOMFactory();
        StAXOMBuilder staxOMBuilder =
                new StAXOMBuilder(fac, xmlReader);
        return staxOMBuilder.getDocumentElement();
    }

    /**
     * @param inMessage
     * @return the javascript message to invoke
     * @throws AxisFault
     * @throws XMLStreamException
     */
    public String getJSMethod(MessageContext inMessage) throws AxisFault, XMLStreamException {

        //Look at the method name. if available this should be a javascript method
        AxisOperation op = inMessage.getOperationContext().getAxisOperation();
        if (op == null) {
            throw new AxisFault(Messages.getMessage("notFound", "Operation"));
        }
        return op.getName().getLocalPart();
    }

    /**
     * @param inMessage
     * @return the javascript source file
     * @throws AxisFault
     */
    public Reader readJS(MessageContext inMessage) throws AxisFault {

        AxisService service = inMessage.getOperationContext().getServiceContext().getAxisService();
        Parameter implInfoParam = service.getParameter("ServiceClass");
        if (implInfoParam == null) {
            throw new AxisFault(Messages.getMessage("Parameter is not specified", "ServiceClass"));
        }
        InputStream jsFileStream = service.getClassLoader().getResourceAsStream(implInfoParam.getValue().toString());
        if (jsFileStream == null) {
            throw new AxisFault(Messages.getMessage("javaScriptUnableToLoad", implInfoParam.getValue().toString()));
        }
        return new BufferedReader(new InputStreamReader(jsFileStream));
    }

    /**
     * @param inMessage
     * @return arguments to be sent to the javascript function
     * @throws XMLStreamException
     */
    public String getArgs(MessageContext inMessage) throws XMLStreamException {

        OMElement firstChild = inMessage.getEnvelope().getBody().getFirstElement().getFirstElement(); // This has to be fixed
        String value = firstChild.toStringWithConsume();
        if (value != null) {
            return value;
        }
        return null;

    }
}
