/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.trust;

import junit.framework.TestCase;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.rahas.Token;
import org.apache.rahas.TrustException;
import org.wso2.wsas.ServerConstants;

import java.util.Date;
import java.util.Properties;

/**
 *
 */
public class SecurityTokenStoreTest extends TestCase {

    private SecurityTokenStore store;

    protected void setUp() throws Exception {
        System.setProperty(ServerConstants.DERBY_HOME, "conf");
        store = new SecurityTokenStore();
    }

    public void testAddToken() {
        try {
            String tokenId = "token" + System.currentTimeMillis();
            Token token = getTestToken(tokenId);
            String secret = "The sky is falling!";
            token.setSecret(secret.getBytes());
            store.add(token);
            Token retToken = store.getToken(tokenId);
            assertNotNull(retToken);
            assertEquals(secret, new String(retToken.getSecret()));

        } catch (TrustException e) {
            e.printStackTrace();
            fail("Unexpected Exception :" + e);
        }
    }

    public void testUpdateToken() {
        String tokenId = "token" + System.currentTimeMillis();
        try {
            Token testToken = getTestToken(tokenId);
            String secret = "The sky is falling!";
            testToken.setSecret(secret.getBytes());
            store.add(testToken);
            Token retToken = store.getToken(tokenId);
            assertNotNull(retToken);
            assertEquals(secret, new String(retToken.getSecret()));
            String newSecret = "New secret. Old secret was false!";
            retToken.setSecret(newSecret.getBytes());
            store.update(retToken);
            retToken = store.getToken(tokenId);
            assertNotNull(retToken);
            assertEquals(newSecret, new String(retToken.getSecret()));
        } catch (TrustException e) {
            e.printStackTrace();
            fail("Unexpected Exception :" + e);
        }
    }

    public void testLoadTokensFromDatabase() {
        String tokenId = "token" + System.currentTimeMillis();
        try {
            Token testToken1 = getTestToken(tokenId);
            String secret = "The sky is falling!";
            testToken1.setSecret(secret.getBytes());
            store.add(testToken1);

            Token testToken2 = getTestToken(tokenId + 1);
            testToken2.setSecret(secret.getBytes());
            store.add(testToken2);

            SecurityTokenStore store2 = new SecurityTokenStore();
            String[] tokenIdentifiers = store2.getTokenIdentifiers();
            assertTrue(tokenIdentifiers.length >= 2);
        } catch (TrustException e) {
            e.printStackTrace();
            fail("Unexpected Exception :" + e);
        }
    }

    public void testAddTokenWithProperties() {
        try {
            String tokenId = "token" + System.currentTimeMillis();
            Token token = getTestToken(tokenId);
            String secret = "The sky is falling!";
            token.setSecret(secret.getBytes());

            Properties props = new Properties();
            props.setProperty("A", "AVal");
            props.setProperty("B", "BVal");
            props.setProperty("C", "CVal");
            token.setProperties(props);

            store.add(token);
            Token retToken = store.getToken(tokenId);
            assertNotNull(retToken);
            assertEquals(secret, new String(retToken.getSecret()));
            assertEquals(3, retToken.getProperties().size());

            SecurityTokenStore store2 = new SecurityTokenStore();
            Token retToken2 = store2.getToken(tokenId);
            assertNotNull(retToken2);
            assertEquals(secret, new String(retToken2.getSecret()));
            assertEquals(3, retToken2.getProperties().size());

        } catch (TrustException e) {
            e.printStackTrace();
            fail("Unexpected Exception :" + e);
        }
    }

    public void _testTokenExpiry() {
        try {
            // Add a new Token that will expire in 1 sec
            String tokenId = "testToken" + System.currentTimeMillis();
            store.add(getTestToken(tokenId, new Date(System.currentTimeMillis() + 1000)));
            Token[] validTokens = store.getValidTokens();
            assertEquals(1, validTokens.length);
            assertEquals(tokenId, validTokens[0].getId());

            // Simulate a delay, so that at the end of the delay, we can check token expiry
            try {
                Thread.sleep(1500);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            assertTrue(store.getExpiredTokens().length > 0);   // Token should have expired now

            // Renew all expired tokens
            Token[] expiredTokens = store.getExpiredTokens();
            for (int i = 0; i < expiredTokens.length; i++) {
                Token token = expiredTokens[i];
                token.setState(Token.RENEWED);
                token.setExpires(new Date(System.currentTimeMillis() + 5000));
                store.update(token);
            }

            //Renew all valid tokens
            validTokens = store.getValidTokens();
            for (int i = 0; i < validTokens.length; i++) {
                Token token = validTokens[i];
                token.setState(Token.RENEWED);
                token.setExpires(new Date(System.currentTimeMillis() + 5000));
                store.update(token);
            }

            // All tokens should be valid now
            assertEquals(0, store.getExpiredTokens().length);
            assertEquals(store.getTokenIdentifiers().length, store.getValidTokens().length);

            // Verify that the DB has got updated
            SecurityTokenStore store2 = new SecurityTokenStore();
            assertEquals(0, store2.getExpiredTokens().length);
            assertEquals(store2.getTokenIdentifiers().length, store2.getValidTokens().length);
        } catch (TrustException e) {
            e.printStackTrace();
            fail("Unexpected Exception :" + e);
        }
    }

    private Token getTestToken(String tokenId, Date expiry) throws TrustException {
        OMFactory factory = DOOMAbstractFactory.getOMFactory();
        OMElement tokenEle = factory.createOMElement("testToken",
                                                     "http://www.wso2.org/products/wsas",
                                                     "wso2wsas");
        Token token = new Token(tokenId, tokenEle, new Date(), expiry);
        token.setAttachedReference(tokenEle);
        token.setPreviousToken(tokenEle);
        token.setState(Token.ISSUED);
        token.setSecret("Top secret!".getBytes());
        return token;
    }

    private Token getTestToken(String tokenId) throws TrustException {
        return getTestToken(tokenId, new Date());
    }
}
