/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence;

import junit.framework.TestCase;
import org.hibernate.Session;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.persistence.dao.ServiceGroupDAO;
import org.wso2.wsas.persistence.dataobject.*;
import org.wso2.wsas.persistence.exception.*;
import org.wso2.wsas.util.HibernateConfig;
import org.wso2.wsas.util.HibernateConfigFactory;

import java.util.List;

/**
 *
 */
public class PersistenceTest extends TestCase {
    private HibernateConfig hbConfig = HibernateConfigFactory.getDefaultConfig("wso2wsas.test");
    private PersistenceManager pm;

    protected void setUp() throws Exception {
        System.setProperty(ServerConstants.DERBY_HOME, "conf");
        pm = new PersistenceManager();
    }

    public void testAddAdminAccount() {
        ServiceUserRoleDO role = new ServiceUserRoleDO();
        role.setRole(ServerConstants.ADMIN_ROLE + System.currentTimeMillis());
        try {
            pm.addUserRole(role);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception");
        }

        ServiceUserDO admin = new ServiceUserDO();
//        admin.setServiceUserRole(role);
        admin.setUsername("admin1-" + System.currentTimeMillis());
        admin.setPassword("wso2wsas");
        try {
            pm.addUser(admin);
        } catch (ServiceUserAlreadyExistsException e) {
            fail("Unexpected Exception");
        }

        Session session = hbConfig.currentSession();
        List result = session.createQuery("from ServiceUserDO").list();
        validateAdmin(result);
        hbConfig.closeSession();
    }

    public void testAddModules() {
        try {
            pm.addModule(getModule("sandesha"));
            pm.addModule(getModule("security"));
        } catch (ModuleAlreadyExistsException e) {
            fail("Unexpected Exception");
        }

        Session session = hbConfig.currentSession();
        List result =
                session.createQuery("from ModuleDO as mod " +
                                    "where lower(mod.moduleIdentifierDO.name) " +
                                    "like 'sandesha%'").list();
        validateSandeshaModule(result);

        result =
                session.createQuery("from ModuleDO as mod " +
                                    "where lower(mod.moduleIdentifierDO.name) " +
                                    "like 'security%'").list();
        validateSecurityModule(result);
        hbConfig.closeSession();
    }

    public void testModifyAdmin() {
        /*Session session = hbConfig.currentSession();
        List result = session.createQuery("from AdminDO where username like 'admin%'").list();
        for (int i = 0; i < result.size(); i++) {
            AdminDO admin = (AdminDO) result.get(i);
            admin.setPassword(admin.getPassword() + "-NEW");
            try {
                pm.updateAdmin(admin);
            } catch (AdminNotFoundException e) {
                fail("Unexpected Exception");
            }
        }
        hbConfig.closeSession();*/
    }

    public void testDeleteAdmin() {
        /*Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        List result = session.createQuery("from AdminDO where username='admin'").list();
        for (int i = 0; i < result.size(); i++) {
            AdminDO admin = (AdminDO) result.get(i);
            session.delete(admin);
        }
        tx.commit();
        hbConfig.closeSession();*/
    }

    public void testAddUserRole() {
        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        serviceUserRole.setRole("testAddUserRole-Role" + System.currentTimeMillis());
        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }
    }

    public void testAddService() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO sid = new ServiceIdentifierDO();
        String serviceId = "service" + System.currentTimeMillis();
        sid.setServiceId(serviceId);
        String version = "0.95";
        sid.setVersion(version);
        service.setServiceIdentifierDO(sid);
        service.setIsUTAuthEnabled(true);
        ServiceGroupDO sgDO = new ServiceGroupDO();
        sgDO.setName("test-sg-" + System.currentTimeMillis());
        new ServiceGroupDAO(hbConfig).createServiceGroup(sgDO);
        service.setServiceGroup(sgDO);

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        ServiceDO resultService = pm.getService(serviceId, version);
        assertNotNull(resultService);
        assertEquals(serviceId, resultService.getServiceIdentifierDO().getServiceId());
        assertEquals(version, resultService.getServiceIdentifierDO().getVersion());
    }

    public void _testAssignUserToService() { // fails
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO sid = new ServiceIdentifierDO();
        String serviceId = "service" + System.currentTimeMillis();
        sid.setServiceId(serviceId);
        String version = "0.95";
        sid.setVersion(version);
        service.setServiceIdentifierDO(sid);
        service.setIsUTAuthEnabled(true);

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "Role" + timestamp;
        serviceUserRole.setRole(role);
        ServiceUserDO serviceUser = new ServiceUserDO();
//        serviceUser.setServiceUserRole(serviceUserRole);
        String username = "username" + timestamp;
        serviceUser.setUsername(username);
        serviceUser.setPassword("password");

        serviceUser.addService(service);
        try {
            pm.addUser(serviceUser);
        } catch (ServiceUserAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        ServiceUserDO user = pm.getUser(username);
        assertNotNull(user);
    }

    public void testAddUser() {
        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "testAddUser-Role" + timestamp;
        serviceUserRole.setRole(role);

        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }
        ServiceUserDO serviceUser = new ServiceUserDO();
//        serviceUser.setServiceUserRole(serviceUserRole);
        String username = "username" + timestamp;
        serviceUser.setUsername(username);
        serviceUser.setPassword("password");

        try {
            pm.addUser(serviceUser);
        } catch (ServiceUserAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        assertNotNull(pm.getUser(username));

        /* Set users = pm.getUsersWithRole(role);
      assertNotNull(users);
      assertFalse(users.isEmpty());
      assertEquals(1, users.size()); */

        /* for (Iterator iter = users.iterator(); iter.hasNext();) {
            ServiceUserDO serviceUserDO = (ServiceUserDO) iter.next();
            assertEquals(username, serviceUserDO.getUsername());
            assertEquals(role, serviceUserDO.getServiceUserRole().getRole());
            assertEquals("password", serviceUserDO.getPassword());
        }*/
    }

    public void testUpdateUser() {
        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "testUpdateUser-Role" + timestamp;
        serviceUserRole.setRole(role);
        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }
        ServiceUserDO serviceUser = new ServiceUserDO();
//        serviceUser.setServiceUserRole(serviceUserRole);
        String username = "testUpdateUser-username" + timestamp;
        serviceUser.setUsername(username);
        serviceUser.setPassword("password");

        try {
            pm.addUser(serviceUser);
        } catch (ServiceUserAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        ServiceUserDO user = pm.getUser(username);
        assertNotNull(user);
        user.setPassword("newPassword");

        try {
            pm.updateUser(user);
        } catch (ServiceUserNotFoundException e) {
            fail("Unexpected Exception" + e);
        }
    }

    public void testGetUsersWithRole() {
        /*ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "Role" + timestamp;
        serviceUserRole.setRole(role);

        long random = System.currentTimeMillis();
        for (int i = 0; i < 10; i++) {
            random++;
            ServiceUserDO serviceUser = new ServiceUserDO();
            serviceUser.setServiceUserRole(serviceUserRole);
            String username = "username" + random;
            serviceUser.setUsername(username);
            serviceUser.setPassword("password" + (random * 2));

            try {
                pm.addUser(serviceUser);
            } catch (ServiceUserAlreadyExistsException e) {
                fail("Unexpected Exception " + e);
            }
        }

        Set usersWithRole = pm.getUsersWithRole(role);
        assertEquals(10, usersWithRole.size());*/
    }

    public void testAddingSameUserTwice() {
        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "testAddingSameUserTwice-Role" + timestamp;
        serviceUserRole.setRole(role);
        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        // Add first time
        {
            ServiceUserDO serviceUser = new ServiceUserDO();
//            serviceUser.setServiceUserRole(serviceUserRole);
            serviceUser.setUsername("testAddingSameUserTwice-username" + timestamp);
            serviceUser.setPassword("password");
            try {
                pm.addUser(serviceUser);
            } catch (ServiceUserAlreadyExistsException e) {
                fail("Unexpected Exception " + e);
            }
        }

        // Add second time
        {
            ServiceUserDO serviceUser = new ServiceUserDO();
//            serviceUser.setServiceUserRole(serviceUserRole);
            serviceUser.setUsername("testAddingSameUserTwice-username" + timestamp);
            serviceUser.setPassword("password");
            System.out.println("## THE FOLLOWING EXCEPTION IS AN ANTICIPATED ONE.\n" +
                               "## THIS UNIT TEST VERIFIES THAT THE SYSTEM THROWS AN EXCEPTION \n" +
                               "## IF THE SAME USER IS ADDED MORE THAN ONCE. PLEASE IGNORE THIS EXCEPTION.");
            try {
                pm.addUser(serviceUser);
                fail("Expected Exception did not occur");
            } catch (Exception e) {

                assertTrue(true);
            }
        }
    }

    public void testAssignUserToService() {
        String username = "AddUserToService-Username" + System.currentTimeMillis();
        String serviceId = "AddUserToService-sid" + System.currentTimeMillis();
        String version = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;

        // Add service
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO sid = new ServiceIdentifierDO();
        sid.setServiceId(serviceId);
        sid.setVersion(version);
        service.setServiceIdentifierDO(sid);
        service.setIsUTAuthEnabled(false);
        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        // Add user
        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        long timestamp = System.currentTimeMillis();
        String role = "AddUserToService-Role" + timestamp;
        serviceUserRole.setRole(role);

        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        ServiceUserDO serviceUser = new ServiceUserDO();
//        serviceUser.setServiceUserRole(serviceUserRole);
        serviceUser.setUsername(username);
        serviceUser.setPassword("password");
        try {
            pm.addUser(serviceUser);
        } catch (ServiceUserAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        assertNotNull(pm.getUser(username));

        try {
            pm.assignUserToService(username, serviceId, version);
        } catch (ServiceUserNotFoundException e) {
            fail("Unexpected Exception " + e);
        } catch (ServiceNotFoundException e) {
            fail("Unexpected Exception " + e);
        }

        assertEquals(1, pm.getUsersForService(serviceId, version).length);
    }

    public void testGetAllUsers() {
        /*List allUsers = pm.getUsers();
        for (Iterator iterator = allUsers.iterator(); iterator.hasNext();) {
            ServiceUserDO o = (ServiceUserDO) iterator.next();
        }*/
    }

    public void testAddServiceWithPolicies() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "PolicyService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        ServicePolicyDO policy1 = new ServicePolicyDO();
        policy1.setPolicy("Policy 1");
        policy1.setType(1);
        policy1.setUuid("P1-" + String.valueOf(System.currentTimeMillis()));
        service.addPolicy(policy1);

        ServicePolicyDO policy2 = new ServicePolicyDO();
        policy2.setPolicy("Policy 2");
        policy2.setType(1);
        policy2.setUuid("P2-" + String.valueOf(System.currentTimeMillis()));
        service.addPolicy(policy2);

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        service = pm.getService(serviceId, version);
        assertNotNull(service);
        assertEquals(2, pm.getServicePolicies(service).length);
    }

    public void testAddServiceWithEngagedModules() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        for (int i = 0; i < 10; i++) {
            ModuleDO module = new ModuleDO();
            ModuleIdentifierDO modID = new ModuleIdentifierDO();
            modID.setName("TEST-MOD" + System.currentTimeMillis() + i);
            modID.setVersion("Mod-V" + System.currentTimeMillis() + i);
            module.setModuleIdentifierDO(modID);

            try {
                pm.addModule(module);
            } catch (ModuleAlreadyExistsException e) {
                e.printStackTrace();
                fail("Unexpected Exception " + e);
            }
            service.addEngagedModule(module);
        }

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        service = pm.getService(serviceId, version);
        assertNotNull(service);

        assertEquals(10, pm.getEngagedModules(service).length);
    }

    public void testAddMessages() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestMessageService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        OperationDO operation = new OperationDO();
        operation.setName("testOp");
        service.addOperation(operation);
        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        assertNotNull(pm.getService(serviceId, version));

        MessageDO message = new MessageDO();
        message.setType(1);
        message.setSequence(System.currentTimeMillis());
        message.setServiceId(serviceId);
        message.setServiceVersion(version);
        message.setXml("<hello>Hi</hello>");
        message.setOperationName(operation.getName());

        pm.addMessage(message);

        MessageDO[] messages = pm.getMessages(message);
        assertEquals(1, messages.length);

        String xml = messages[0].getXml();
        assertEquals("<hello>Hi</hello>", xml);

        MessageDO msg1 = new MessageDO(serviceId, version, "testOp");
        MessageDO[] messages1 = pm.getMessages(msg1);
        assertEquals(1, messages1.length);
        xml = messages1[0].getXml();
        assertEquals("<hello>Hi</hello>", xml);
    }

    public void testAddMessagesWithNoOperation() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestMessageService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        assertNotNull(pm.getService(serviceId, version));

        MessageDO message = new MessageDO();
        message.setType(1);
        message.setSequence(System.currentTimeMillis());
        message.setServiceId(serviceId);
        message.setServiceVersion(version);
        message.setXml("<hello>Hi</hello>");
        message.setOperationName("testOp");

        pm.addMessage(message);

        MessageDO[] messages = pm.getMessages(message);
        assertEquals(1, messages.length);

        String xml = messages[0].getXml();
        assertEquals("<hello>Hi</hello>", xml);

        MessageDO msg1 = new MessageDO(serviceId, version, "testOp");
        MessageDO[] messages1 = pm.getMessages(msg1);
        assertEquals(1, messages1.length);
        xml = messages1[0].getXml();
        assertEquals("<hello>Hi</hello>", xml);
    }

    public void testAddServiceWithOperationModuleEngagement() {
        ModuleDO module = new ModuleDO();
        ModuleIdentifierDO modID = new ModuleIdentifierDO();
        String moduleName = "TEST-MOD" + System.currentTimeMillis();
        modID.setName(moduleName);
        String moduleVersion = "Mod-V" + System.currentTimeMillis();
        modID.setVersion(moduleVersion);
        module.setModuleIdentifierDO(modID);

        try {
            pm.addModule(module);
        } catch (ModuleAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        OperationDO op = new OperationDO();
        op.setName("op" + System.currentTimeMillis());
        service.addOperation(op);
        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        pm.addEngagedOperationToModule(moduleName, moduleVersion, pm.getServiceOperations(service)[0]);

        service = pm.getService(serviceId, version);
        assertNotNull(service);
        OperationDO[] operations = pm.getServiceOperations(service);
        assertEquals(1, operations.length);
        assertEquals(0, pm.getEngagedModules(service).length);

        for (int i = 0; i < operations.length; i++) {
            op = operations[i];
            assertEquals(1, pm.getEngagedModules(op).length);
        }
    }

    public void testAddModuleWithParameters() {
        ModuleDO module = new ModuleDO();
        ModuleIdentifierDO modID = new ModuleIdentifierDO();
        String name = "Mod";
        String version = String.valueOf(System.currentTimeMillis());
        modID.setName(name);
        modID.setVersion(version);
        module.setModuleIdentifierDO(modID);

        ModuleParameterDO param = new ModuleParameterDO();
        param.setName("Param1");
        param.setValue("Val1");
        module.addParameter(param);

        try {
            pm.addModule(module);
        } catch (ModuleAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        module = pm.getModule(name, version);
        assertNotNull(module);

        ModuleParameterDO[] params = pm.getModuleParameters(module);
        for (int i = 0; i < params.length; i++) {
            param = params[i];
            assertEquals("Param1", param.getName());
            assertEquals("Val1", param.getValue());
        }
    }

    public void testAddServiceWithParameters() {
        String name = "Service";
        String version = String.valueOf(System.currentTimeMillis());
        ServiceIdentifierDO sid = new ServiceIdentifierDO();
        sid.setServiceId(name);
        sid.setVersion(version);
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        service.setServiceIdentifierDO(sid);

        OperationDO op = new OperationDO();
        op.setName("myOp");
        service.addOperation(op);

        ServiceParameterDO param = new ServiceParameterDO();
        param.setName("Param1");
        param.setValue("Val1");
        service.addParameter(param);

        OperationParameterDO param2 = new OperationParameterDO();
        param2.setName("Param2");
        param2.setValue("Val2");
        op.addParameter(param2);

        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            fail("Unexpected Exception " + e);
        }

        service = pm.getService(name, version);
        assertNotNull(service);

        ServiceParameterDO[] params = pm.getServiceParameters(service);
        for (int i = 0; i < params.length; i++) {
            param = params[i];
            assertEquals("Param1", param.getName());
            assertEquals("Val1", param.getValue());
        }

        op = pm.getServiceOperations(service)[0];
        assertEquals("myOp", op.getName());
        param2 = pm.getOperationParameters(op)[0];
        assertEquals("Param2", param2.getName());
        assertEquals("Val2", param2.getValue());
    }

    private void setServiceGroup(ServiceDO service) {
        ServiceGroupDO sgDO = new ServiceGroupDO();
        sgDO.setName("test-sg" + System.currentTimeMillis());
        new ServiceGroupDAO(hbConfig).createServiceGroup(sgDO);
        service.setServiceGroup(sgDO);
    }

    public void testGetMaxMessageSeqNum() {
        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestMessageService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        OperationDO operation = new OperationDO();
        String opName = "testOperation";
        operation.setName(opName);
        service.addOperation(operation);
        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        assertNotNull(pm.getService(serviceId, version));

        for (int i = 0; i < 10; i++) {
            MessageDO message = new MessageDO();
            message.setType(i);
            message.setSequence(i);
            message.setServiceId(serviceId);
            message.setServiceVersion(version);
            message.setXml("<hello>Hi</hello>");
            message.setOperationName(operation.getName());

            pm.addMessage(message);
            assertEquals(i, pm.getMaxMessageSequence(serviceId, version, opName));
        }
    }

    public void testDeleteService() {
        ModuleDO module = new ModuleDO();
        ModuleIdentifierDO modID = new ModuleIdentifierDO();
        String moduleName = "TEST-MOD" + System.currentTimeMillis();
        modID.setName(moduleName);
        String moduleVersion = "Mod-V" + System.currentTimeMillis();
        modID.setVersion(moduleVersion);
        module.setModuleIdentifierDO(modID);

        try {
            pm.addModule(module);
        } catch (ModuleAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        ServiceDO service = new ServiceDO();
        setServiceGroup(service);
        ServiceIdentifierDO id = new ServiceIdentifierDO();
        String serviceId = "TestService" + System.currentTimeMillis();
        id.setServiceId(serviceId);
        String version = "V" + System.currentTimeMillis();
        id.setVersion(version);
        service.setServiceIdentifierDO(id);

        OperationDO op = new OperationDO();
        op.setName("op" + System.currentTimeMillis());
        service.addOperation(op);

        ServicePolicyDO policy1 = new ServicePolicyDO();
        policy1.setPolicy("Policy 1");
        policy1.setType(1);
        policy1.setUuid("P1-" + String.valueOf(System.currentTimeMillis()));
        try {
            pm.addService(service);
        } catch (ServiceAlreadyExistsException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }
        pm.addEngagedOperationToModule(moduleName, moduleVersion, op);
        pm.addPolicyToService(serviceId, version, policy1);

        service = pm.getService(serviceId, version);
        assertNotNull(service);

        OperationDO[] operations = pm.getServiceOperations(serviceId, version);
        assertEquals(1, operations.length);
        assertEquals(0, pm.getEngagedModules(serviceId, version).length);

        for (int i = 0; i < operations.length; i++) {
            op = operations[i];
            assertEquals(1, pm.getEngagedModules(op).length);
        }

        try {
            pm.deleteService(serviceId, version);
        } catch (ServiceNotFoundException e) {
            e.printStackTrace();
            fail("Unexpected Exception " + e);
        }

        assertNull(pm.getService(serviceId, version));
    }

    private void validateAdmin(List list) {
        for (int i = 0; i < list.size(); i++) {
            ServiceUserDO admin = (ServiceUserDO) list.get(i);
            assertNotNull(admin.getPassword());
//            assertTrue(admin.getUsername().indexOf("admin") == 0);
        }
    }

    private void validateSandeshaModule(List list) {
        for (int i = 0; i < list.size(); i++) {
            ModuleDO module = (ModuleDO) list.get(i);
            assertEquals("sandesha", module.getModuleIdentifierDO().getName());
        }
    }

    private void validateSecurityModule(List list) {
        for (int i = 0; i < list.size(); i++) {
            ModuleDO module = (ModuleDO) list.get(i);
            assertEquals("security", module.getModuleIdentifierDO().getName());
        }
    }

    private ModuleDO getModule(String moduleName) {
        ModuleIdentifierDO modID = new ModuleIdentifierDO();
        modID.setName(moduleName);
        modID.setVersion(String.valueOf(System.currentTimeMillis()));

        ModuleDO module = new ModuleDO();
        addModuleParameter("db.url",
                           "jdbc:derby:data/" + moduleName.toUpperCase() + "_DB;create=true",
                           module);
        addModuleParameter("db.driver", "org.apache.derby.jdbc.EmbeddedDriver", module);
        addModuleParameter("db.sql.dialect", "org.hibernate.dialect.DerbyDialect", module);
        addModuleParameter("db.username", "username", module);
        addModuleParameter("db.password", "password", module);

        module.setModuleIdentifierDO(modID);
        return module;
    }

    private void addModuleParameter(String name, String value, ModuleDO module) {
        ModuleParameterDO param = new ModuleParameterDO();
        param.setName(name);
        param.setValue(value);
        module.addParameter(param);
    }
}
