/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.axis2.Constants;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.wso2.adminui.AdminUIConstants;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerConstants;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.Map;

/**
 * A collection of useful utility methods
 */
public class WsasUtils {

    public static boolean isAdminConsoleEnabled() {
        boolean enableAdminConsole = false;
        String enableAdminConsoleProp =
                ServerConfiguration.getInstance().getFirstProperty("Management.EnableConsole");
        if (enableAdminConsoleProp != null) {
            enableAdminConsole = Boolean.valueOf(enableAdminConsoleProp).booleanValue();
        }
        return enableAdminConsole;
    }

    public static void setServletContextProperties(ServletContext servletContext,
                                                   Map generatedPages,
                                                   boolean isAdminConsoleEnabled,
                                                   int httpPort,
                                                   int httpsPort,
                                                   String servicePath)
            throws ServletException {
        servletContext.setAttribute(AdminUIConstants.GENERATED_PAGES, generatedPages);
        servletContext.setAttribute(AdminUIConstants.ENABLE_CONSOLE,
                                    Boolean.valueOf(isAdminConsoleEnabled));
        servletContext.setAttribute(AdminUIConstants.HTTP_PORT, httpPort + "");
        servletContext.setAttribute(AdminUIConstants.HTTPS_PORT, httpsPort + "");
        servletContext.setAttribute(AdminUIConstants.SERVICE_CONTEXT_PATH, servicePath);
    }

    public static String getAxis2Xml() {
        String axis2XML = ServerConfiguration.getInstance().
                getFirstProperty("Axis2Config.ConfigurationFile");
        if (axis2XML == null) {
            axis2XML = System.getProperty(Constants.AXIS2_CONF);
        }
        return axis2XML;
    }

    public static String getWsasServerXml() {
        String serverXML = System.getProperty(ServerConstants.WSO2WSAS_SERVER_XML);
        if (serverXML == null) {
            serverXML =
                    System.getProperty(ServerConstants.WSO2WSAS_HOME) + File.separator + "conf" +
                    File.separator + "server.xml";
        }
        return serverXML;
    }

    /**
     * Check whther the specified Strin corresponds to a URL
     *
     * @param location The String to be checked
     * @return true - if <code>location</code> is a URL, false - otherwise
     */
    public static boolean isURL(String location) {
        try {
            new URL(location);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    /**
     * Get the last updated time of the service artifact corresponding to AxisServiceGroup
     * serviceGroup
     *
     * @param serviceGroup The service group whose last updated time we need to get
     * @return The last updated time of the service artifact corresponding to AxisServiceGroup
     *         serviceGroup. Will return null if a Axis2 URL repository is in use.
     */
    public static Long lastUpdatedTime(AxisServiceGroup serviceGroup) {
        String axis2Repo = ServerConfiguration.getInstance().
                getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        if (isURL(axis2Repo)) { // We do not support retrieving the timestamps of URLs
            return null;
        }

        Long lastUpdated = null;
        String fileName = "";
        for (Iterator serviceIter = serviceGroup.getServices(); serviceIter.hasNext();) {
            AxisService axisService = (AxisService) serviceIter.next();
            URL fn = axisService.getFileName();
            if (fn != null) {
                fileName = fn.getPath();
            }
            if ((fileName != null) && (fileName.trim().length() != 0)) {
                File file = new File(fileName);
                lastUpdated = new Long(file.lastModified());
                break;
            }
        }
        return lastUpdated;
    }

    /**
     * Get the port of the command listener. A server socket is opened on this port to listen
     * to incoming commands
     *
     * @return The command listener port
     */
    public static int getCommandListenerPort() {
        int serverPort = -1;
        ServerConfiguration serverConfig = ServerConfiguration.getInstance();
        String port = serverConfig.getFirstProperty("Ports.CommandListener");
        if (port == null) {
            port = serverConfig.getFirstProperty("CommandListener.Port");
        }
        if (port != null) {
            serverPort = Integer.parseInt(port);
        }
        return serverPort;
    }
}
