/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.axiom.soap.SOAPEnvelope;
import org.apache.axis2.Constants;
import org.apache.axis2.context.MessageContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.tracer.TracerConstants;
import org.wso2.tracer.TracerUtils;
import org.wso2.tracer.module.TracePersister;
import org.wso2.utils.xml.XMLPrettyPrinter;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.MessageDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 *
 */
public class TracePersisterImpl implements TracePersister {
    private static final String TRACING_MAP = "local_wso2tracer.map";
    private static final String REQUEST_NUMBER = "local_wso2tracer.request.number";

    private static Log log = LogFactory.getLog(TracePersisterImpl.class);
    private MessageContext msgContext;

    private static Log TRACE_LOGGER = LogFactory.getLog("trace.messages");

    private PersistenceManager pm = new PersistenceManager();

    public void setMsgContext(MessageContext msgContext) {
        this.msgContext = msgContext;
    }

    public boolean isTracingEnabled() {
        String prop = pm.getConfigurationProperty(TracerConstants.TRACING_STATUS);
        return prop != null && prop.equalsIgnoreCase("ON");
    }

    public long saveMessage(String operationName,
                            String serviceName,
                            int flow,
                            SOAPEnvelope env) {
        MessageDO msgDO = new MessageDO();
        msgDO.setServiceId(serviceName);
        msgDO.setOperationName(operationName);

        String xml;

        try {
            xml = TracerUtils.getPrettyString(env, msgContext);
        } catch (Exception e) {
            String msg = " The received SOAP Message could not be serialized";
            log.warn(msg, e);
            xml = msg + " : " + e;
        }

        msgDO.setXml(xml);
        if (TRACE_LOGGER.isTraceEnabled()) {
            TRACE_LOGGER.trace(xml);
        }

        long msgSequence = getMessageSequence(serviceName + "." + operationName,
                                              serviceName,
                                              operationName);
        msgDO.setType(flow);
        msgDO.setSequence(msgSequence);
        pm.addMessage(msgDO);

        return msgSequence;
    }

    public void saveTraceStatus(String onOff) {
        pm.updateConfigurationProperty(TracerConstants.TRACING_STATUS, onOff);
    }

    public String[] getMessages(String serviceId,
                                String operationName,
                                long messageSequence) {
        MessageDO msg = new MessageDO();
        msg.setServiceId(serviceId);
        msg.setOperationName(operationName);
        msg.setSequence(messageSequence);

        MessageDO[] messages = pm.getMessages(msg);
        String[] resp = new String[2];

        if ((messages != null) && (messages.length != 0)) {
            for (int i = 0; i < messages.length; i++) {
                MessageDO message = messages[i];

                if (message.getType() == MessageContext.IN_FLOW ||
                    message.getType() == MessageContext.IN_FAULT_FLOW) {
                    resp[0] = message.getXml();
                } else if (message.getType() == MessageContext.OUT_FLOW ||
                           message.getType() == MessageContext.OUT_FAULT_FLOW) {
                    resp[1] = message.getXml();
                }
            }
        }

        return resp;
    }

    private long getMessageSequence(String key, String serviceName, String operationName) {
        long msgSequence = 1;

        // check whether this is a continuation of an existing MEP
        Object requestNumber = msgContext.getOperationContext().getProperty(REQUEST_NUMBER);
        if ((requestNumber != null) && requestNumber instanceof Long) {
            msgSequence = ((Long) requestNumber).intValue();
        } else {
            // Need to have a counter for each and operation
            Map monitoringHandlerMap =
                    (Map) msgContext.getConfigurationContext().getProperty(TRACING_MAP);

            if (monitoringHandlerMap == null) {
                monitoringHandlerMap = new HashMap();
                msgContext.getConfigurationContext().setProperty(TRACING_MAP,
                                                                 monitoringHandlerMap);
            }

            Object counterInt = monitoringHandlerMap.get(key);

            if (counterInt == null) {
                PersistenceManager pm = new PersistenceManager();
                msgSequence = pm.getMaxMessageSequence(serviceName,
                                                       ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                                       //TODO: replace this with service version when it is available
                                                       operationName) + 1;
            } else if (counterInt instanceof Long) {
                msgSequence = ((Long) counterInt).intValue() + 1;
            }

            monitoringHandlerMap.put(serviceName + "." + operationName,
                                     new Long(msgSequence));
            msgContext.getOperationContext().setProperty(REQUEST_NUMBER,
                                                         new Long(msgSequence));
        }

        return msgSequence;
    }
}
