/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.ServerException;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

/**
 * ServerController listens to incoming commands on a designated socket
 * and executes those commands.
 */
public class ServerController implements Runnable {

    // use the local loopback address so that only localhost can send the messages
    public static final String LOCAL_HOST_IP = "127.0.0.1";
    public static final String SHUTDOWN_COMMAND = "shutdownserver";
    public static final String RESTART_COMMAND = "restartserver";
    public static ServerSocket serverSocket;

    private Controllable controllable;
    private int port;
    private static Log log = LogFactory.getLog(ServerController.class);

    public ServerController(Controllable controllable) {
        this.controllable = controllable;
    }

    public ServerController() {
    }

    public void run() {
        listenForCommands();
    }

    private void listenForCommands() {
        try {
            serverSocket =
                    new ServerSocket(port, 1, InetAddress.getByName(LOCAL_HOST_IP));
        } catch (IOException e) {
            log.error("Cannot listen on port " + port + ". May be it is in use.", e);
            System.exit(1);
        }
        while (true) {
            try {

                // wait here until some one talks to server
                Socket socket = serverSocket.accept();
                InputStream inputStream = socket.getInputStream();
                StringBuffer stringBuffer = new StringBuffer();
                int ch;
                while ((ch = inputStream.read()) != -1) {
                    stringBuffer.append((char) ch);
                }
                String command = stringBuffer.toString();
                log.debug("Received Command ==> " + command);
                if (command.equals(SHUTDOWN_COMMAND)) {
                    try {
                        // close the socket and exit from the jvm
                        inputStream.close();
                        socket.close();
                        serverSocket.close();
                        controllable.shutdownGracefully();
                    } catch (Exception e) {
                        log.error("Exception while shutting down server", e);
                    }
                } else if (command.equals(RESTART_COMMAND)) {
                    try {
                        log.info("Restarting WSO2 WSAS...");
                        controllable.restart();
                    } catch (ServerException e) {
                        log.error("Exception while restarting the server", e);
                    } finally {
                        // close the socket and exit from the jvm
                        inputStream.close();
                        socket.close();
                        serverSocket.close();
                    }
                } else {
                    log.error("Invalid command : " + command);
                }
            } catch (IOException e) {

                // if we cannot start the server then we cannot run the server
                log.error("Cannot listen on port " + port + ". May be it is in use.", e);
                System.exit(1);
            }
        }
    }

    /**
     * send the shutdown command to server listen port
     */

    public void shutDownServer() {
        try {
            Socket socket = new Socket(LOCAL_HOST_IP, port);
            OutputStream outStream = socket.getOutputStream();
            for (int i = 0; i < SHUTDOWN_COMMAND.length(); i++) {
                outStream.write(SHUTDOWN_COMMAND.charAt(i));
            }
            outStream.flush();
            outStream.close();
            socket.close();
            System.out.println(" Shutdown signal posted to WSO2 WSAS");
            System.out.println(" Server shutdown may take some time. " +
                               "Please check the server's log file for completion.");
        } catch (IOException e) {
            System.err.println(" Could not connect to WSO2 WSAS controller at " +
                               LOCAL_HOST_IP + ":" + port);
            e.printStackTrace();
            System.exit(1);
        }
    }

    public void restartServer() {
        try {
            Socket socket = new Socket(LOCAL_HOST_IP, port);
            OutputStream outStream = socket.getOutputStream();
            for (int i = 0; i < RESTART_COMMAND.length(); i++) {
                outStream.write(RESTART_COMMAND.charAt(i));
            }
            outStream.flush();
            outStream.close();
            socket.close();
            System.out.println(" Restart signal posted to WSO2 WSAS");
            System.out.println(" Server restart may take some time. " +
                               "Please check the server's log file for completion.");
        } catch (IOException e) {
            System.err.println(" Could not connect to WSO2 WSAS controller at " +
                               LOCAL_HOST_IP + ":" + port);
            e.printStackTrace();
            System.exit(1);
        }
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }
}
