/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.wso2.utils.InputReader;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerConfigurationException;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.ServerConstants;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * A tool for updating the password of a user
 */
public class PasswordUpdater {
    public static final String DB_URL = "--db-url";
    public static final String DB_DRIVER = "--db-driver";
    public static final String DB_USERNAME = "--db-username";
    public static final String DB_PASSWORD = "--db-password";
    public static final String USERNAME = "--username";
    public static final String NEW_PASSWORD = "--new-password";


    public static void main(String[] args) {
        new PasswordUpdater().run(args);
    }

    private void run(String[] args) {
        String wso2wsasHome = System.getProperty(ServerConstants.WSO2WSAS_HOME);
        if (wso2wsasHome == null) {
            wso2wsasHome = new File(".").getAbsolutePath();
            System.setProperty(ServerConstants.WSO2WSAS_HOME, wso2wsasHome);
        }

        if (args.length == 0) {
            printUsage();
            System.exit(0);
        }

        String dbURL = getParam(DB_URL, args);

        if (dbURL == null || dbURL.indexOf("jdbc:") != 0) {
            System.err.println(" Invalid database DB_URL : " + dbURL);
            printUsage();
            System.exit(0);
        }

        // ------- DB Connection params
        String dbDriver = getParam(DB_DRIVER, args);
        if (dbDriver == null) {
            dbDriver = "org.apache.derby.jdbc.EmbeddedDriver";
        }
        String dbUsername = getParam(DB_USERNAME, args);
        if (dbUsername == null) {
            dbUsername = "wso2wsas";
        }
        String dbPassword = getParam(DB_PASSWORD, args);
        if (dbPassword == null) {
            dbPassword = "wso2wsas";
        }

        // ------------ Load the DB Driver
        try {
            Class.forName(dbDriver);
        } catch (ClassNotFoundException e) {
            System.err.println(" Database driver [" + dbDriver + "] not found in classpath.");
            System.exit(1);
        }

        // Connect to the database
        Connection conn = null;
        try {
            conn = DriverManager.getConnection(dbURL, dbUsername, dbPassword);
        } catch (Exception e) {
            System.err.println("Cannot connect to database. \n" +
                               "Please make sure that the JDBC URL is correct and that you have \n" +
                               "stopped WSO2 WSAS before running this script. Root cause is : \n" + e);
            System.exit(1);
        }

        // --------- Capture the service username and password
        String username = getParam(USERNAME, args);
        while (username == null || username.trim().length() == 0) {
            System.out.print("Username: ");
            try {
                username = InputReader.readInput();
            } catch (IOException e) {
                System.err.println(" Could not read username : " + e);
                System.exit(1);
            }
        }

        String password = getParam(NEW_PASSWORD, args);
        if (password == null || password.trim().length() == 0) {
            String passwordRepeat = null;
            while (password == null || password.trim().length() == 0) {
                try {
                    password = InputReader.readPassword("New password: ");
                } catch (IOException e) {
                    System.err.println("Unable to read password : " + e);
                    System.exit(1);
                }
            }
            while (passwordRepeat == null || passwordRepeat.trim().length() == 0) {
                try {
                    passwordRepeat = InputReader.readPassword("Re-enter new password: ");
                } catch (IOException e) {
                    System.err.println("Unable to read re-entered password : " + e);
                    System.exit(1);
                }
            }
            if (!password.equals(passwordRepeat)) {
                System.err.println(" Password and re-entered password do not match");
                System.exit(1);
            }
        }

        // ----------- Update the DB
        PreparedStatement statement = null;
        try {
            ServerConfiguration config = ServerConfiguration.getInstance();
            config.init(Thread.currentThread().getContextClassLoader().getResourceAsStream("server.xml"));
            password = new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                      config.getFirstProperty("Security.KeyStore.Password"),
                                      config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                      config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                      config.getFirstProperty("Security.KeyStore.Type")).
                    encryptAndBase64Encode(password.getBytes());

            String sql = "update service_user_t set c_password=? where c_username=?";
            statement = conn.prepareStatement(sql);
            statement.setString(1, password);
            statement.setString(2, username.trim());
            statement.execute();
            if (statement.getUpdateCount() == 0) {
                System.err.println(" User with name '" + username + "' does not exist.");
            } else {
                System.out.println(" Password of user '" + username + "' updated successfully");
            }
        } catch (SQLException e) {
            System.err.println(" Could not update password : " + e);
        } catch (CryptoException e) {
            e.printStackTrace();
            System.err.println(" Password encyption failed : " + e);
        } catch (ServerConfigurationException e) {
            e.printStackTrace();
            System.err.println(" Server error : " + e);
        } finally {
            if (statement != null) {
                try {
                    statement.close();
                } catch (SQLException e) {
                    System.err.println(" Could not close statement : " + e);
                }
            }
            if (conn != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                    System.err.println(" Could not close connection : " + e);
                }
            }
        }
    }

    /**
     * This will check the given parameter in the array and will return, if available
     *
     * @param param
     * @param args
     * @return the parameter
     */
    private String getParam(String param, String[] args) {
        if (param == null || "".equals(param)) {
            return null;
        }
        for (int i = 0; i < args.length; i = i + 2) {
            String arg = args[i];
            if (param.equalsIgnoreCase(arg) && (args.length >= (i + 1))) {
                return args[i + 1];
            }
        }
        return null;
    }

    private void printUsage() {
        System.out.println("Usage: chpasswd --db-url DB_URL [OPTIONS]\n");
        System.out.println("         " + DB_URL + " : The JDBC database URL. " +
                           "e.g. jdbc:derby:/home/wso2wsas/database/WSO2WSAS_DB\n");
        System.out.println("Options");
        System.out.println("         " + DB_DRIVER + "    : The database driver class. " +
                           "e.g. org.apache.derby.jdbc.EmbeddedDriver");
        System.out.println("         " + DB_USERNAME + "  : The database username");
        System.out.println("         " + DB_PASSWORD + "  : The database password");
        System.out.println("         " + USERNAME + "     : The username of the user whose " +
                           "password is to be changed. If this is not given, " +
                           "you will be prompted for this field later.");
        System.out.println("         " + NEW_PASSWORD + " : The new password of the user " +
                           "whose password is to be changed. If this is not given, " +
                           "you will be prompted for this field later.");
    }
}
