/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.axis2.AxisFault;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Appender;
import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.net.SyslogAppender;
import org.springframework.util.Log4jConfigurer;
import org.wso2.utils.CircularBuffer;
import org.wso2.utils.MemoryAppender;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.AppenderDO;
import org.wso2.wsas.persistence.dataobject.LoggerDO;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 *
 */
public class LoggingUtil {
    private static final Log log = LogFactory.getLog(LoggingUtil.class);

    private static PersistenceManager pm = new PersistenceManager();

    public static void setSystemLoggingParameters(String logLevel, String logPattern) {
        System.setProperty(ServerConstants.Logging.SYSTEM_LOG_LEVEL, logLevel);
        System.setProperty(ServerConstants.Logging.SYSTEM_LOG_PATTERN, logPattern);
    }

    public static String getSystemLogLevel() {
        return System.getProperty(ServerConstants.Logging.SYSTEM_LOG_LEVEL);
    }

    public static String getSystemLogPattern() {
        return System.getProperty(ServerConstants.Logging.SYSTEM_LOG_PATTERN);
    }

    public static void loadDefaultConfiguration() {

        // parse the logging file and get the information
        loadDefaultGlobalLogSettings();

        // get all the appenders travelling through the all the loggers
        // currently in log4j there is no way to get all the appenders directly
        Set appenderSet = new HashSet();
        Logger rootLogger = LogManager.getRootLogger();
        Enumeration appenders = rootLogger.getAllAppenders();

        addAppendersToSet(appenders, appenderSet);

        // get the loggers and save them
        Enumeration loggers = LogManager.getCurrentLoggers();
        Logger logger;
        LoggerDO loggerDO;
        while (loggers.hasMoreElements()) {
            logger = (Logger) loggers.nextElement();
            loggerDO = toLoggerDO(logger);
            if (loggerDO != null) {
                pm.addLogger(loggerDO);
            }
            addAppendersToSet(logger.getAllAppenders(), appenderSet);
        }

        // save appenders to database
        Appender appender;
        AppenderDO appenderDO;
        for (Iterator iter = appenderSet.iterator(); iter.hasNext();) {
            appender = (Appender) iter.next();
            appenderDO = toAppenderDO(appender);
            pm.addAppender(appenderDO);
        }
    }

    public static void restoreDefaults() throws AxisFault {
        pm.removeAllLoggersAndAppenders();
        pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_IS_LOADED, "false");
        LogManager.resetConfiguration();

        try {
            Log4jConfigurer.initLogging("classpath:log4j.properties");
        } catch (FileNotFoundException e) {
            String msg = "Cannot restore default logging configuration." +
                         " log4j.properties file not found in the classpath";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }
    }

    public static void loadCustomConfiguration() {
        // set the appender details

        // we have not provided a facility to add or remove appenders so all the
        // initial appender set should present in the system.
        // and all the initall logger should present in the system
        Set appenderSet = new HashSet();
        Logger rootLogger = LogManager.getRootLogger();

        // set the root logger level
        String rootLoggerLevel =
                pm.getConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_LEVEL);
        rootLogger.setLevel(Level.toLevel(rootLoggerLevel));
        String systemLogPattern =
                pm.getConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_PATTERN);
        setSystemLoggingParameters(rootLoggerLevel, systemLogPattern);

        addAppendersToSet(rootLogger.getAllAppenders(), appenderSet);

        // go through other loggers and put them to appender set
        // get the loggers and save them
        // and set their details according to data base settings
        Enumeration loggers = LogManager.getCurrentLoggers();
        Logger logger;
        LoggerDO loggerDO;
        while (loggers.hasMoreElements()) {
            logger = (Logger) loggers.nextElement();
            // we ignore all class level defined loggers
            addAppendersToSet(logger.getAllAppenders(), appenderSet);
            loggerDO = pm.getLoggerDO(logger.getName());
            if (loggerDO != null) {
                logger.setLevel(Level.toLevel(loggerDO.getLogLevel()));
                logger.setAdditivity(loggerDO.getAdditivity());
            }
        }

        // update the appender data according to data stored in database
        Appender appender;

        for (Iterator iter = appenderSet.iterator(); iter.hasNext();) {
            appender = (Appender) iter.next();
            AppenderDO appenderDO = pm.getAppenderDO(appender.getName());
            if (appenderDO != null) {
                if ((appender.getLayout() != null) &&
                    (appender.getLayout() instanceof PatternLayout)) {
                    ((PatternLayout) appender.getLayout()).
                            setConversionPattern(appenderDO.getPattern());
                }
                if (appender instanceof FileAppender) {
                    FileAppender fileAppender = ((FileAppender) appender);
                    fileAppender.setFile(appenderDO.getLogFileName());
                    fileAppender.activateOptions();
                }

                if (appender instanceof MemoryAppender) {
                    MemoryAppender memoryAppender = (MemoryAppender) appender;
                    memoryAppender.setCircularBuffer(new CircularBuffer(200));
                    memoryAppender.activateOptions();
                }

                if (appender instanceof SyslogAppender) {
                    SyslogAppender syslogAppender = (SyslogAppender) appender;
                    syslogAppender.setSyslogHost(appenderDO.getSysLogHost());
                    syslogAppender.setFacility(appenderDO.getFacility());
                }

                if (appender instanceof AppenderSkeleton) {
                    AppenderSkeleton appenderSkeleton = (AppenderSkeleton) appender;
                    appenderSkeleton.setThreshold(Level.toLevel(appenderDO.getThreshold()));
                    appenderSkeleton.activateOptions();
                }
            }
        }
    }

    private static void loadDefaultGlobalLogSettings() {

        // put the values to configuration table
        pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_IS_LOADED,
                                       "true");

        // root logger debug level
        String systemLogLevel = Logger.getRootLogger().getLevel().toString();
        pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_LEVEL, systemLogLevel);

        // set the default pattern as follows
        String systemLogPattern = "%5p [%d] %x %m%n";
        pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_PATTERN,
                                       systemLogPattern);

        setSystemLoggingParameters(systemLogLevel, systemLogPattern);
    }

    private static void addAppendersToSet(Enumeration appenders, Set appenderSet) {
        while (appenders.hasMoreElements()) {
            Appender appender = (Appender) appenders.nextElement();
            appenderSet.add(appender);
        }
    }

    /**
     * when getting loggers from getCurrentLoggers method it will return all the loggers in the
     * system. this includes loggers we have initialized using getLogger methods. but these loggers
     * does not have a log level set they get the log level from the parent loggers. but to store in
     * database we use only loggers with a log level
     *
     * @param logger The Log4J Logger to be converted
     * @return loggerDO The corresponding LoggerDO
     */
    public static LoggerDO toLoggerDO(Logger logger) {
        LoggerDO loggerDO = new LoggerDO();
        loggerDO.setName(logger.getName());
        loggerDO.setLogLevel(logger.getEffectiveLevel().toString());
        loggerDO.setAdditivity(logger.getAdditivity());
        return loggerDO;
    }

    /**
     * Coverts the provided Log4J appender into an AppenderDO
     *
     * @param appender The Log4J Appender to be coverted
     * @return appenderDO The corresponding AppenderDO
     */
    private static AppenderDO toAppenderDO(Appender appender) {
        AppenderDO appenderDO = new AppenderDO();
        appenderDO.setName(appender.getName());
        if (appender.requiresLayout()) {
            Layout layout = appender.getLayout();
            if (layout instanceof PatternLayout) {
                appenderDO.setPattern(((PatternLayout) layout).getConversionPattern());
            }
        }
        if (appender instanceof FileAppender) {
            FileAppender fileAppender = (FileAppender) appender;
            String fileName = fileAppender.getFile();
            File logFile = new File(fileName);
            if (!logFile.isAbsolute()) {
                if (fileName.startsWith(".")) {
                    fileName = fileName.substring(1);
                }
                fileName = (System.getProperty(ServerConstants.WSO2WSAS_HOME) + "/" +
                            fileName).replace('\\', '/');
                fileAppender.setFile(fileName);
                fileAppender.activateOptions();
            }
            appenderDO.setLogFileName(fileName);
            appenderDO.setIsFileAppender(true);
        } else {
            appenderDO.setIsFileAppender(false);
        }

        if (appender instanceof MemoryAppender) {
            MemoryAppender memoryAppender = (MemoryAppender) appender;
            memoryAppender.
                    setCircularBuffer(new CircularBuffer(ServerConstants.
                            Logging.MEMORY_APPENDER_BUFFER_SZ));
            memoryAppender.activateOptions();
        }

        // normally all the appenders inherit from AppenderSkelton
        if (appender instanceof AppenderSkeleton) {
            AppenderSkeleton appenderSkeleton = (AppenderSkeleton) appender;
            if (appenderSkeleton.getThreshold() != null) {
                appenderDO.setThreshold(appenderSkeleton.getThreshold().toString());
            } else {
                appenderDO.setThreshold("DEBUG");
            }
        }

        if (appender instanceof SyslogAppender) {
            SyslogAppender syslogAppender = (SyslogAppender) appender;

            // if user has not set these properties system automatically assign default values
            appenderDO.setSysLogHost(syslogAppender.getSyslogHost());
            appenderDO.setFacility(syslogAppender.getFacility());
            appenderDO.setIsSysLogAppender(true);
        } else {
            appenderDO.setIsSysLogAppender(false);
        }
        return appenderDO;
    }
}
