/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.axis2.AxisFault;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.exception.KeyStoreAlreadyExistsException;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.security.KeyStore;
import java.security.cert.Certificate;
import java.util.Date;
import java.util.Enumeration;

/**
 *
 */
public final class KeyStoreUtil {

    private static PersistenceManager pm = new PersistenceManager();
    private static Log log = LogFactory.getLog(KeyStoreUtil.class);

    private KeyStoreUtil() {
    }

    public synchronized static boolean keyStoreExists(String keyStoreLocation) {
        return pm.getKeyStore(getKeyStoreName(keyStoreLocation)) != null;
    }

    public synchronized static void persistKeyStore(String keyStoreLocation,
                                                    String keyStorePassword,
                                                    String keyStoreType,
                                                    String privateKeyAlias,
                                                    String privateKeyPassword,
                                                    String provider,
                                                    boolean isPrimaryKeyStore)
            throws KeyStoreAlreadyExistsException, ServerException {

        String ksName = getKeyStoreName(keyStoreLocation);
        if (pm.getKeyStore(ksName) == null) {
            KeyStoreDO keyStoreDO = new KeyStoreDO();
            keyStoreDO.setKeyStoreName(ksName);
            keyStoreDO.setIsPrimaryKeyStore(isPrimaryKeyStore);
            if (!new File(keyStoreLocation).isAbsolute()) {
                keyStoreLocation = System.getProperty(ServerConstants.WSO2WSAS_HOME) +
                                   File.separator + keyStoreLocation;
            }
            keyStoreDO.setFilePath(keyStoreLocation);
            keyStoreDO.setKeyStoreType(keyStoreType);
            keyStoreDO.setProvider(provider);
            keyStoreDO.setLastUpdatedTime(new Date());
            keyStoreDO.setPrivateKeyAlias(privateKeyAlias);
            try {
                ServerConfiguration config = ServerConfiguration.getInstance();
                CryptoUtil cryptoUtil =
                        new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                       config.getFirstProperty("Security.KeyStore.Password"),
                                       config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                       config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                       config.getFirstProperty("Security.KeyStore.Type"));
                keyStoreDO.setPrivateKeyPassword(cryptoUtil.
                        encryptAndBase64Encode(privateKeyPassword.getBytes()));
                keyStoreDO.setStorePassword(cryptoUtil.
                        encryptAndBase64Encode(keyStorePassword.getBytes()));
            } catch (CryptoException e) {
                throw new ServerException("Cannot encrypt WSO2 WSAS " +
                                          "store and/or private key passwords", e);
            }
            pm.addKeyStore(keyStoreDO);
        } else {
            throw new KeyStoreAlreadyExistsException("Keystore with name " + ksName +
                                                     " already exists");
        }
    }

    private static String getKeyStoreName(String keyStoreLocation) {
        String ksName = keyStoreLocation;
        if (keyStoreLocation.indexOf("/") != -1) {
            int index = keyStoreLocation.lastIndexOf("/");
            if (index != -1) {
                ksName = keyStoreLocation.substring(index + 1);
            }
        } else {
            int index = keyStoreLocation.lastIndexOf("\\");
            if (index != -1) {
                ksName = keyStoreLocation.substring(index + 1);
            }
        }
        return ksName;
    }

    public static Certificate getCertificate(String keyStoreName,
                                             String alias) throws AxisFault {
        try {
            KeyStore keyStore = getKeyStore(keyStoreName);
            Enumeration enumeration = keyStore.aliases();
            while (enumeration.hasMoreElements()) {
                String itemAlias = (String) enumeration.nextElement();
                if (itemAlias.equals(alias)) {
                    return keyStore.getCertificate(alias);
                }
            }
            return null;
        } catch (Exception e) {
            String msg = "Could not read certificates from keystore file. ";
            log.error(msg, e);
            throw new AxisFault(msg + e.getMessage());
        }
    }

    public static KeyStore getKeyStore(String keyStoreName) throws Exception {
        KeyStoreDO keyStoreDO = pm.getKeyStore(keyStoreName);
        KeyStore keyStore = KeyStore.getInstance(keyStoreDO.getKeyStoreType());
        String filePath = keyStoreDO.getFilePath();
        if (!new File(filePath).isAbsolute()) {
            filePath = System.getProperty(ServerConstants.WSO2WSAS_HOME) +
                       File.separator + filePath;
        }
        FileInputStream ksIn = new FileInputStream(filePath);
        BufferedInputStream ksbufin = new BufferedInputStream(ksIn);
        try {
            String storePassword = keyStoreDO.getStorePassword();
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil =
                    new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                   config.getFirstProperty("Security.KeyStore.Password"),
                                   config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                   config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                   config.getFirstProperty("Security.KeyStore.Type"));
            keyStore.load(ksbufin,
                          new String(cryptoUtil.base64DecodeAndDecrypt(storePassword)).toCharArray());
        } finally {
            ksIn.close();
            ksbufin.close();
        }

        return keyStore;
    }
}
