/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.apache.commons.collections.BidiMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.TimerTask;

/**
 *
 */
public class HouseKeepingTask extends TimerTask {
    private static Log log = LogFactory.getLog(HouseKeepingTask.class);

    private String workDir;
    private BidiMap fileResourceMap;
    private FileManipulator fileMan;
    private int fileTimeoutMillis;

    public HouseKeepingTask(String workDir, BidiMap fileResourceMap) {
        this.workDir = workDir;
        this.fileResourceMap = fileResourceMap;
        this.fileMan = new FileManipulator();
        fileTimeoutMillis =
                Integer.parseInt(ServerConfiguration.getInstance().
                        getFirstProperty("HouseKeeping.MaxTempFileLifetime")) * 60 * 1000;
    }

    public void run() {
        log.debug("Starting house-keeping task...");
        File workDir = new File(this.workDir);
        if (workDir.exists()) {
            Collection deletedFiles = new ArrayList();
            clean(workDir, deletedFiles);
            log.debug("Clearing filemap cache...");
            if (fileResourceMap != null) {
                for (Iterator iterator = deletedFiles.iterator(); iterator.hasNext();) {
                    fileResourceMap.removeValue(iterator.next());
                }
            }
        }
        log.debug("House-keeping complete.");
    }

    private void clean(File file, Collection deletedFiles) {
        File[] children = file.listFiles();
        if (children != null) {
            for (int i = 0; i < children.length; i++) {
                clean(children[i], deletedFiles);
            }
            if (file.listFiles() == null || file.listFiles().length == 0) { // all children deleted?
                String absPath = file.getAbsolutePath();
                if (log.isDebugEnabled()) {
                    log.debug("Deleting directory " + absPath + "...");
                }
                deletedFiles.add(absPath);
                fileMan.deleteDir(file);
            }
        } else {
            if (System.currentTimeMillis() -
                file.lastModified() >= fileTimeoutMillis) {
                String absPath = file.getAbsolutePath();
                if (file.isDirectory()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Deleting directory " + absPath + "...");
                    }
                    deletedFiles.add(absPath);
                    fileMan.deleteDir(file);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("Deleting file " + absPath + "...");
                    }
                    deletedFiles.add(absPath);
                    file.delete();
                }
            }
        }
    }

    public void setWorkDir(String workDir) {
        this.workDir = workDir;
    }

    public void setFileResourceMap(BidiMap fileResourceMap) {
        this.fileResourceMap = fileResourceMap;
    }
}
