/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.util;

import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;

/**
 * A utility for managing Hibernate Configuration related stuff.
 * <p/>
 * We should maintain only one instance of HibernateConfig within WSO2 WSAS admin. Each of the
 * modules/services deployed should also maintain there own single instance
 */
public class HibernateConfig {
    private final SessionFactory SESSION_FACTORY;
    private final ThreadLocal SESSION = new ThreadLocal();

    /**
     * Constructor for initializing this Object providing the DB Connection Identifier
     * (<code>dbConnIdentifier</code>), and the the Hibernate Data Object classes
     * (<code>dataObjects</code>), along with other DB parameters.
     *
     * @param dbConnIdentifier The database connection identifier. Can be a DB URL or a Datasource
     *                         JNDI name. If it is a Datsource JNDI name, it should start with
     *                         <i>java:comp</i>. This is a <b>required parameter</b> <br/>
     * @param dbDriver         The fully qualified DB driver class name. This is a <b>required
     *                         parameter</b> <br/>
     * @param sqlDialect       The hibernate SQL dialect for the specifed RDBMS. This is a
     *                         <b>required parameter</b><br/>
     * @param username         The username to connect to the DB<br/>
     * @param password         The password corresponding to the <code>username</code><br/>
     * @param dataObjects      Array of Hibernate Data Object Classes. This is a <b>required
     *                         parameter</b><br/>
     */
    public HibernateConfig(String dbConnIdentifier,
                           String dbDriver,
                           String sqlDialect,
                           String username,
                           String password,
                           Class[] dataObjects) {

        if (!validateRequiredParams(new Object[]{dbConnIdentifier, dbDriver,
                                                 sqlDialect, dataObjects})) {
            throw new IllegalArgumentException("One or more of the required parameters " +
                                               "were not provided");
        }
        try {
            Configuration cfg = new Configuration();
            for (int i = 0; i < dataObjects.length; i++) {
                cfg.addClass(dataObjects[i]);
            }

            commonInit(cfg, sqlDialect, dbDriver, dbConnIdentifier, username, password);
            SESSION_FACTORY = cfg.buildSessionFactory();
        } catch (Throwable ex) {
            System.err.println("Initial SessionFactory creation failed." + ex);
            throw new ExceptionInInitializerError(ex);
        }
    }

    /**
     * Constructor for initializing this Object providing the DB Connection Identifier
     * (<code>dbConnIdentifier</code>), and the the Hibernate hbm XML filenames
     * (<code>hbmXMLFilenames</code>), along with other DB parameters.
     *
     * @param dbConnIdentifier The database connection identifier. Can be a DB URL or a Datasource
     *                         JNDI name. If it is a Datsource JNDI name, it should start with
     *                         <i>java:comp</i>. This is a <b>required parameter</b> <br/>
     * @param dbDriver         The fully qualified DB driver class name. This is a <b>required
     *                         parameter</b> <br/>
     * @param sqlDialect       The hibernate SQL dialect for the specifed RDBMS. This is a
     *                         <b>required parameter</b><br/>
     * @param username         The username to connect to the DB<br/>
     * @param password         The password corresponding to the <code>username</code><br/>
     * @param hbmXMLFilenames  Array of Hibernate hbm XML Filenames. This is a <b>required
     *                         parameter</b><br/>
     */
    public HibernateConfig(String dbConnIdentifier,
                           String dbDriver,
                           String sqlDialect,
                           String username,
                           String password,
                           String[] hbmXMLFilenames) {

        if (!validateRequiredParams(new Object[]{dbConnIdentifier, dbDriver,
                                                 sqlDialect, hbmXMLFilenames})) {
            throw new IllegalArgumentException("One or more of the required parameters " +
                                               "were not provided");
        }
        try {
            Configuration cfg = new Configuration();
            for (int i = 0; i < hbmXMLFilenames.length; i++) {
                cfg.addResource(hbmXMLFilenames[i]);
            }

            commonInit(cfg, sqlDialect, dbDriver, dbConnIdentifier, username, password);
            SESSION_FACTORY = cfg.buildSessionFactory();
        } catch (Throwable ex) {
            System.err.println("Initial SessionFactory creation failed." + ex);
            throw new ExceptionInInitializerError(ex);
        }
    }

    /**
     * Default Constructor. This should be package protected since we should only instantiate it
     * through it HibernateConfigFactory <br/>
     * <p/>
     * This will pickup hibernate.cfg.xml from the classpath and load the configuration from this
     * file.
     */
    HibernateConfig() {
        try {
            // Create the SessionFactory from hibernate.cfg.xml, which has to be in the classpath
            SESSION_FACTORY =
                    new Configuration().
                            configure("wso2wsas.hibernate.cfg.xml").buildSessionFactory();
        } catch (Throwable ex) {
            System.err.println("Initial SessionFactory creation failed." + ex);
            throw new ExceptionInInitializerError(ex);
        }
    }

    private void commonInit(Configuration cfg, String sqlDialect, String dbDriver,
                            String dbConnIdentifier, String username, String password) {

        cfg.setProperty("hibernate.dialect", sqlDialect);

        if (dbConnIdentifier.indexOf("java:comp") == 0) { // Is it a datasource JNDI?
            cfg.setProperty("hibernate.connection.datasource", dbConnIdentifier);
        } else { // it is a db URL
            cfg.setProperty("hibernate.connection.url", dbConnIdentifier);
            if (username != null && username.trim().length() != 0) {
                cfg.setProperty("hibernate.connection.username",
                                username);
            }
            if (password != null && password.trim().length() != 0) {
                cfg.setProperty("hibernate.connection.password",
                                password);
            }

//            cfg.setProperty("hibernate.hbm2ddl.auto", "update");
            cfg.setProperty("hibernate.connection.driver_class", dbDriver);
        }
    }

    private boolean validateRequiredParams(Object[] params) {
        if (params == null) {
            return false;
        }
        for (int i = 0; i < params.length; i++) {
            Object param = params[i];
            if (param == null) {
                return false;
            }
            if (param instanceof String && ((String) param).trim().length() == 0) {
                return false;
            }
            if (param instanceof Object[]) {
                if (!validateRequiredParams((Object[]) param)) {
                    return false;
                }
            }
        }
        return true;
    }

    public Session currentSession() throws HibernateException {
        Session session = (Session) SESSION.get();
        if (session == null) {
            session = SESSION_FACTORY.openSession();
            SESSION.set(session);
        }
        return session;
    }

    public void closeSession() throws HibernateException {
        Session session = (Session) SESSION.get();
        if (session != null) {
            session.close();
        }
        SESSION.set(null);
    }
}
