/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.transport.util;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.AxisService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.TransportDO;
import org.wso2.wsas.transport.HttpGetRequestProcessor;
import org.wso2.wsas.util.KeyStoreUtil;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 *
 */
public class CertProcessor implements HttpGetRequestProcessor {
    private static Log log = LogFactory.getLog(CertProcessor.class);
    private static PersistenceManager pm = new PersistenceManager();

    public void process(HttpServletRequest request,
                        HttpServletResponse response,
                        ConfigurationContext configurationContext) throws Exception {
        String requestURI = request.getRequestURI();
        String contextPath = configurationContext.getServiceContextPath();
        String serviceName =
                requestURI.substring(requestURI.indexOf(contextPath) + contextPath.length() + 1);

        AxisService axisService =
                configurationContext.getAxisConfiguration().getServiceForActivation(serviceName);
        OutputStream outputStream = response.getOutputStream();
        if (!axisService.isActive()) {
            response.setContentType("text/html");
            outputStream.write(("<h4>Service " + serviceName +
                                " is inactive. Cannot retrieve certificate.</h4>").getBytes());
            outputStream.flush();
        } else {
            ServiceDO serviceDO = pm.getService(serviceName, null);
            KeyStoreDO privateKeyStore;
            if ((privateKeyStore = serviceDO.getPrivateKeyStore()) == null) {
                Set set = serviceDO.getTransports();
                boolean httpsEnabled = false;
                for (Iterator iterator = set.iterator(); iterator.hasNext();) {
                    TransportDO transportDO = (TransportDO) iterator.next();
                    String transportProtocol = transportDO.getTransportProtocol();
                    if (transportProtocol.equals("https")) {
                        httpsEnabled = true;
                        break;
                    }
                }
                if (httpsEnabled || serviceDO.getIsExposedOnAllTransports()) {
                    KeyStoreDO[] keyStoreDOs = pm.getKeyStores();
                    List list = Arrays.asList(keyStoreDOs);
                    for (Iterator iterator = list.iterator(); iterator.hasNext();) {
                        KeyStoreDO keyStoreDO = (KeyStoreDO) iterator.next();
                        if (keyStoreDO.getIsPrimaryKeyStore()) {
                            Certificate cert =
                                    KeyStoreUtil.getCertificate(keyStoreDO.getKeyStoreName(),
                                                                keyStoreDO.getPrivateKeyAlias());
                            serializeCert(cert, response, outputStream, serviceName);
                            break;
                        }
                    }
                } else {
                    response.setContentType("text/html");
                    outputStream.write(("<h4>Service " + serviceName +
                                        " does not have a private key.</h4>").getBytes());
                    outputStream.flush();
                }
            } else {
                // pump out the certificate
                Certificate certificate =
                        KeyStoreUtil.getCertificate(privateKeyStore.getKeyStoreName(),
                                                    privateKeyStore.getPrivateKeyAlias());
                serializeCert(certificate, response, outputStream, serviceName);
            }
        }
    }

    /**
     * Pump out the certificate
     *
     * @param certificate  cert
     * @param response     response
     * @param outputStream out stream
     * @param serviceName  service name
     * @throws AxisFault will be thrown
     */
    private void serializeCert(Certificate certificate, HttpServletResponse response,
                               OutputStream outputStream, String serviceName) throws AxisFault {
        try {
            response.setContentType("application/octet-stream");
            response.setHeader("Content-Disposition", "filename=" + serviceName + ".cert");
            outputStream.write(certificate.getEncoded());
        } catch (CertificateEncodingException e) {
            String msg = "Could not get encoded format of certificate";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        } catch (IOException e) {
            String msg = "Faliour when serializing to stream";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        } finally {
            try {
                outputStream.flush();
            } catch (IOException e) {
                String msg = "Faliour when serializing to stream";
                log.error(msg, e);
            }
        }
    }
}
