/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.transport.fileupload;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.wsas.ServerConstants;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.namespace.QName;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public class FileUploadExecutorManager {
    private static Log log = LogFactory.getLog(FileUploadExecutorManager.class);
    private ConfigurationContext configurationContext;
    private Map executorMap = new HashMap();

    public FileUploadExecutorManager(ConfigurationContext configurationContext)
            throws ServerException {
        this.configurationContext = configurationContext;
        this.executorMapFilter();
    }

    public boolean execute(HttpServletRequest request,
                           HttpServletResponse response) throws
                                                         IOException,
                                                         ServerException {
        String requestURI = request.getRequestURI();

        //for generic Executor to work
        boolean executed = false;
        for (Iterator iterator = executorMap.keySet().iterator(); iterator.hasNext();) {
            String key = (String) iterator.next();
            if (requestURI.indexOf(key) >= 0) {
                FileUploadExecutor obj = (FileUploadExecutor) executorMap.get(key);
                obj.execute(request, response);
                executed = true;
                break;
            }
        }
        if (!executed) {
            // this is the place '*'  handles if present
            Object obj = executorMap.get("*");
            if (obj == null) {
                log.warn("Reached 'All' section but Could not find the Implementation Class");
                return false;
            }
            AnyFileUploadExecutor executor = (AnyFileUploadExecutor) obj;
            executor.execute(request, response);
        }
        return true;
    }

    private void executorMapFilter() throws ServerException {
        ServerConfiguration serverConfiguration = ServerConfiguration.getInstance();
        OMElement documentElement = serverConfiguration.getDocumentElement();
        OMElement fileUploadConfigElement =
                documentElement.getFirstChildWithName(
                        new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "FileUploadConfig"));
        for (Iterator iterator = fileUploadConfigElement.getChildElements(); iterator.hasNext();) {
            OMElement mapppingElement = (OMElement) iterator.next();
            if (mapppingElement.getLocalName().equalsIgnoreCase("Mapping")) {
                OMElement actionsElement =
                        mapppingElement.getFirstChildWithName(
                                new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Actions"));

                if (actionsElement == null) {
                    String msg = "The mandatory FileUploadConfig/Actions entry " +
                                 "does not exist or is empty in the WSO2WSAS_HOME/conf/server.xml " +
                                 "file. Please fix this error in the  server.xml file and restart.";
                    log.error(msg);
                    throw new ServerException(msg);
                }
                Iterator actionElementIterator =
                        actionsElement.getChildrenWithName(
                                new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Action"));

                if (!actionElementIterator.hasNext()) {
                    String msg = "A FileUploadConfig/Mapping entry in the " +
                                 "WSO2WSAS_HOME/conf/server.xml should have at least on Action " +
                                 "defined. Please fix this error in the server.xml file and " +
                                 "restart.";
                    log.error(msg);
                    throw new ServerException(msg);
                }

                OMElement classElement = mapppingElement.getFirstChildWithName(
                        new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Class"));

                if (classElement == null || classElement.getText() == null) {
                    String msg = "The mandatory FileUploadConfig/Mapping/Class entry " +
                                 "does not exist or is empty in the WSO2WSAS_HOME/conf/server.xml " +
                                 "file. Please fix this error in the  server.xml file and restart.";
                    log.error(msg);
                    throw new ServerException(msg);
                }

                FileUploadExecutor object;
                String className = classElement.getText().trim();
                try {
                    Class clazz = Class.forName(className);
                    Constructor constructor =
                            clazz.getConstructor(new Class[]{ConfigurationContext.class});
                    object = (FileUploadExecutor) constructor
                            .newInstance(new Object[]{configurationContext});

                } catch (Exception e) {
                    String msg = "Error occurred while trying to instantiate the " + className +
                                 " class specified as a FileUploadConfig/Mapping/class element in " +
                                 "the WSO2WSAS_HOME/conf/server.xml file. Please fix this error in " +
                                 "the server.xml file and restart.";
                    log.error(msg, e);
                    throw new ServerException(msg, e);
                }

                while (actionElementIterator.hasNext()) {
                    OMElement actionElement = (OMElement) actionElementIterator.next();
                    if (actionElement.getText() == null) {
                        String msg = "A FileUploadConfig/Mapping/Actions/Action element in the " +
                                     "WSO2WSAS_HOME/conf/server.xml file is empty. Please include " +
                                     "the correct value in this file and restart.";
                        log.error(msg);
                        throw new ServerException(msg);
                    }
                    executorMap.put(actionElement.getText().trim(), object);
                }
            }
        }
    }
}
