/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.transport.fileupload;

import org.apache.axis2.context.ConfigurationContext;
import org.apache.commons.collections.bidimap.TreeBidiMap;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.servlet.ServletRequestContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.utils.WSO2Constants;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.util.WsasUtils;

import javax.activation.DataHandler;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
/*
 * 
 */

public abstract class AbstractFileUploadExecutor implements FileUploadExecutor {

    protected static final Log log = LogFactory.getLog(AbstractFileUploadExecutor.class);

    protected ConfigurationContext configurationContext;

    public AbstractFileUploadExecutor(ConfigurationContext configurationContext) {
        this.configurationContext = configurationContext;
    }

    protected void checkServiceFileExtensionValidity(String fileExtension,
                                                     String[] allowedExtensions) throws Exception {
        boolean isExtensionValid = false;
        StringBuffer allowedExtensionsStr = new StringBuffer();
        for (int i = 0; i < allowedExtensions.length; i++) {
            allowedExtensionsStr.append(allowedExtensions[i]).append(",");
            if (fileExtension.endsWith(allowedExtensions[i])) {
                isExtensionValid = true;
                break;
            }
        }
        if (!isExtensionValid) {
            throw new Exception(" Illegal file type." +
                                " Allowed file extensions are " + allowedExtensionsStr);
        }
    }

    protected File uploadFile(HttpServletRequest request,
                              String repoDir,
                              HttpServletResponse response,
                              String extension) throws IOException {

        response.setContentType("text/html; charset=utf-8");
        ServletRequestContext servletRequestContext = new ServletRequestContext(request);
        boolean isMultipart =
                ServletFileUpload.isMultipartContent(servletRequestContext);
        File uploadedFile = null;
        if (isMultipart) {
            try {
                // Create a new file upload handler
                List items = parseRequest(servletRequestContext);

                // Process the uploaded items
                for (Iterator iter = items.iterator(); iter.hasNext();) {
                    FileItem item = (FileItem) iter.next();
                    if (!item.isFormField()) {
                        String fileName = item.getName();
                        String fileExtension = fileName;
                        fileExtension = fileExtension.toLowerCase();
                        if (extension != null && !fileExtension.endsWith(extension)) {
                            throw new Exception(" Illegal file type. Only " +
                                                extension + " files can be uploaded");

                        }
                        String fileNameOnly = getFileName(fileName);
                        uploadedFile = new File(repoDir, fileNameOnly);
                        item.write(uploadedFile);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
                log.error("File upload failed", e);
                response.getWriter().write("<script language=\"javascript\">" +
                                           "top.wso2.wsf.Util.alertWarning('File upload FAILED. " +
                                           e.getMessage() + "');" +
                                           "</script>");
            }
        }
        return uploadedFile;
    }

    protected String getFileName(String fileName) {
        String fileNameOnly;
        if (fileName.indexOf("\\") < 0) {
            fileNameOnly = fileName.substring(fileName.lastIndexOf("/") + 1,
                                              fileName.length());
        } else {
            fileNameOnly = fileName.substring(fileName.lastIndexOf("\\") + 1,
                                              fileName.length());
        }
        return fileNameOnly;
    }

    /**
     * This is the common method that can be used for Fileupload.
     * extraStoreDirUUID is the name of the javascript that's going to
     * execute on the client side at the secound run.
     *
     * @param request
     * @param response
     * @param extraStoreDirUUID
     * @return Status true/fase.
     * @throws ServerException
     * @throws IOException
     */

    protected boolean executeCommon(HttpServletRequest request, HttpServletResponse response,
                                    String extraStoreDirUUID)
            throws ServerException, IOException {
        if (extraStoreDirUUID == null ||
            (extraStoreDirUUID.equals("") && extraStoreDirUUID.length() == 0)) {
            throw new ServerException("Specify extraStoreDirUUID");
        }
        extraStoreDirUUID = extraStoreDirUUID.trim();
        ServletRequestContext servletRequestContext = new ServletRequestContext(request);
        boolean isMultipart =
                ServletFileUpload.isMultipartContent(servletRequestContext);
        PrintWriter out = response.getWriter();
        response.setContentType("text/html; charset=utf-8");
        try {
            if (isMultipart) {

                List items = parseRequest(servletRequestContext);
                boolean multiItems = false;
                if (items.size() > 1) {
                    multiItems = true;
                }
                // Process the uploaded items
                List uuidList = new ArrayList();
                for (Iterator iter = items.iterator(); iter.hasNext();) {
                    FileItem item = (FileItem) iter.next();
                    if (!item.isFormField()) {
                        String uuid = String.valueOf(System.currentTimeMillis() + Math.random());
                        uuidList.add(uuid);
                        String extraFileLocation =
                                configurationContext.getProperty(ServerConstants.WORK_DIR) +
                                File.separator + "extra" +
                                File.separator + uuid + File.separator;

                        File dirs = new File(extraFileLocation);
                        if (!dirs.exists()) {
                            dirs.mkdirs();
                        }
                        File uploadedFile = new File(extraFileLocation,
                                                     getFileName(item.getName()));
                        String fileName = item.getName();
                        if ((fileName == null || fileName.length() == 0) && multiItems) {
                            continue;
                        }
                        item.write(uploadedFile);
                        Map fileResourceMap =
                                (Map) configurationContext
                                        .getProperty(ServerConstants.FILE_RESOURCE_MAP);
                        if (fileResourceMap == null) {
                            fileResourceMap = new TreeBidiMap();
                            configurationContext.setProperty(ServerConstants.FILE_RESOURCE_MAP,
                                                             fileResourceMap);
                        }
                        fileResourceMap.put(uuid, uploadedFile.getAbsolutePath());
                    }
                }
                // call the javascript which will in turn call the relevant web service
                int size = uuidList.size();
                if (size == 1) {
                    out.write("<script language=\"javascript\">" +
                              "top." + extraStoreDirUUID + "('" + uuidList.get(0) + "');" +
                              "</script>");
                } else {
                    String s = "var uObj = new Object();";
                    for (int i = 0; i < size; i++) {
                        s += "uObj[" + i + "]=\"" + uuidList.get(i) + "\";\n";
                    }
                    out.write("<script language=\"javascript\">" +
                              s +
                              "top." + extraStoreDirUUID + "(uObj);" +
                              "</script>");
                }

                out.flush();
            }
        } catch (Exception e) {
            log.error("File upload FAILED", e);
            out.write("<script language=\"javascript\">" +
                      "top.wso2.wsf.Util.alertWarning('File upload FAILED. File may be non-existent or invalid.');" +
                      "</script>");
        } finally {
            out.close();
        }
        return true;
    }

    protected List parseRequest(ServletRequestContext requestContext) throws FileUploadException {
        // Create a factory for disk-based file items
        FileItemFactory factory = new DiskFileItemFactory();
        // Create a new file upload handler
        ServletFileUpload upload = new ServletFileUpload(factory);
        // Parse the request
        return upload.parseRequest(requestContext);
    }


    /**
     * This is a helper method that will be used upload main entity (ex: wsdd, jar, class etc) and
     * its resources to a given deployer.
     *
     * @param request
     * @param response
     * @param uploadDirName
     * @param extensions
     * @param utilityString
     * @return boolean
     * @throws IOException
     */
    protected boolean uploadArtifacts(HttpServletRequest request,
                                      HttpServletResponse response,
                                      String uploadDirName,
                                      String[] extensions,
                                      String utilityString)
            throws IOException {
        String axis2Repo = ServerConfiguration.getInstance().
                getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        PrintWriter out = response.getWriter();
        if (WsasUtils.isURL(axis2Repo)) {
            out.write("<script language=\"javascript\">" +
                      "top.wso2.wsf.Util.alertWarning('You are not permitted to upload jars to URL repository " +
                      axis2Repo + "');" +
                      "</script>");
            out.flush();
            return false;
        }

        String tmpDir = (String) configurationContext.getProperty(WSO2Constants.WORK_DIR);
        String uuid = String.valueOf(System.currentTimeMillis() + Math.random());
        tmpDir = tmpDir + File.separator + "artifacts" + File.separator + uuid + File.separator;
        new File(tmpDir).mkdirs();

        response.setContentType("text/html; charset=utf-8");

        ServletRequestContext servletRequestContext = new ServletRequestContext(request);
        boolean isMultipart =
                ServletFileUpload.isMultipartContent(servletRequestContext);
        if (isMultipart) {
            try {
                // Create a new file upload handler
                List items = parseRequest(servletRequestContext);
                // Process the uploaded items
                for (Iterator iter = items.iterator(); iter.hasNext();) {
                    FileItem item = (FileItem) iter.next();
                    if (!item.isFormField()) {
                        String fileName = item.getName();
                        String fileExtension = fileName;
                        fileExtension = fileExtension.toLowerCase();

                        String fileNameOnly = getFileName(fileName);
                        File uploadedFile;

                        String fieldName = item.getFieldName();

                        if (fieldName != null && fieldName.equals("jarResource")) {
                            if (fileExtension.endsWith(".jar")) {
                                File servicesDir =
                                        new File(tmpDir + File.separator + uploadDirName, "lib");
                                if (!servicesDir.exists()) {
                                    servicesDir.mkdirs();
                                }
                                uploadedFile = new File(servicesDir, fileNameOnly);
                                item.write(uploadedFile);
                            }
                        } else {
                            File servicesDir = new File(tmpDir, uploadDirName);
                            if (!servicesDir.exists()) {
                                servicesDir.mkdirs();
                            }
                            uploadedFile = new File(servicesDir, fileNameOnly);
                            item.write(uploadedFile);
                        }
                    }
                }

                //First lets filter for jar resources
                String repo = configurationContext.getAxisConfiguration().getRepository().getPath();

                //Writting the artifacts to the proper location
                FileManipulator fm = new FileManipulator();
                String parent = repo + File.separator + uploadDirName;
                File mainDir = new File(tmpDir + File.separator + uploadDirName);
                File libDir = new File(mainDir, "lib");
                File[] resourceLibFile = fm.getMatchingFiles(libDir.getAbsolutePath(), null, "jar");


                for (int j = 0; j < resourceLibFile.length; j++) {
                    File dst = new File(parent, "lib");
                    File src = resourceLibFile[j];
                    String[] files = libDir.list();
                    for (int i = 0; i < files.length; i++) {
                        copyFile(src, new File(dst, files[i]));
                    }
                }

                for (int i = 0; i < extensions.length; i++) {
                    File[] mainFiles =
                            fm.getMatchingFiles(mainDir.getAbsolutePath(), null, extensions[i]);
                    for (int j = 0; j < mainFiles.length; j++) {
                        File dst = new File(parent);
                        String[] files = mainDir.list();
                        for (int k = 0; k < files.length; k++) {
                            File f = new File(dst, files[k]);
                            if (!f.isDirectory()) {
                                copyFile(mainFiles[j], f);
                            }
                        }

                    }
                }
                out.write("<script language=\"javascript\">" +
                          "top.completeServiceFileUpload('Files have been uploaded " +
                          "successfully. This page will be auto refreshed shortly with " +
                          "the status of the created " + utilityString + " service ');" +
                          "</script>");
                return true;
            } catch (Exception e) {
                log.error("File upload failed", e);
                out.write("<script language=\"javascript\">" +
                          "top.wso2.wsf.Util.alertWarning('File upload FAILED. " +
                          e.getMessage() + "');" +
                          "</script>");
            }
        }
        return false;
    }

    private void copyFile(File src, File dst) throws IOException, FileNotFoundException {
        String dstAbsPath = dst.getAbsolutePath();
        String dstDir = dstAbsPath.substring(0, dstAbsPath.lastIndexOf(File.separator));
        File dir = new File(dstDir);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        DataHandler dh = new DataHandler(src.toURL());
        FileOutputStream out = new FileOutputStream(dst);
        dh.writeTo(out);
        out.flush();
    }
}
