/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.transport;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jaxen.JaxenException;
import org.jaxen.XPath;
import org.wso2.adminui.UIProcessor;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.util.WsasUtils;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Hashtable;
import java.util.Map;


/**
 * This is the bridge between WSO2WSAS and WSO2 Admin UI Framework
 */
public class UIInitializerServlet extends HttpServlet {

    private static final long serialVersionUID = -7939917535315904295L;

    private static Log log = LogFactory.getLog(UIInitializerServlet.class);

    public void init(ServletConfig servletConfig) throws ServletException {
        // assumes only exploded case
        ServletContext servletContext = servletConfig.getServletContext();
        String resourceBase = servletContext.getRealPath(".");
        try {
            Map files = new Hashtable();

            UIProcessor.createPages(resourceBase, "ui-extensions-config.xml", files);
            Axis2XMLInfo axis2XMLInfo = getAxis2XMLInfo();
            WsasUtils.setServletContextProperties(servletContext,
                                                  files,
                                                  WsasUtils.isAdminConsoleEnabled(),
                                                  axis2XMLInfo.getHttpPort(),
                                                  axis2XMLInfo.getHttpsPort(),
                                                  axis2XMLInfo.getServicePath());
        } catch (Exception e) {
            String msg = "Exception occurred while initializing the UI";
            log.error(msg, e);
            throw new ServletException(msg, e);
        }
    }

    private Axis2XMLInfo getAxis2XMLInfo() throws IOException, XMLStreamException,
                                                  JaxenException {

        ServerConfiguration serverConfiguration = ServerConfiguration.getInstance();

        String axis2XML = serverConfiguration.getFirstProperty("Axis2Config.ConfigurationFile");

        InputStream axis2XMLInStream;
        boolean isURl = isURL(axis2XML);
        if (isURl) {
            axis2XMLInStream = new URL(axis2XML).openStream();
        } else {
            axis2XMLInStream = new FileInputStream(new File(axis2XML));
        }

        XMLStreamReader xmlStreamReader =
                XMLInputFactory.newInstance().createXMLStreamReader(axis2XMLInStream);
        StAXOMBuilder builder = new StAXOMBuilder(xmlStreamReader);

        String servicePath = "services";
        XPath xp = new AXIOMXPath("/axisconfig/parameter[@name='servicePath']");
        OMElement documentElement = builder.getDocumentElement();
        OMElement servicePathEle = (OMElement) xp.selectSingleNode(documentElement);
        if (servicePathEle != null) {
            String text = servicePathEle.getText();
            if (text != null) {
                servicePath = text;
            }
        }

        xp = new AXIOMXPath("/axisconfig/transportReceiver[@name='http']/parameter[@name='port']");
        OMElement httpPortEle = (OMElement) xp.selectSingleNode(documentElement);

        xp = new AXIOMXPath("/axisconfig/transportReceiver[@name='https']/parameter[@name='port']");
        OMElement httpsPortEle = (OMElement) xp.selectSingleNode(documentElement);

        //For proxying
        xp = new AXIOMXPath(
                "/axisconfig/transportReceiver[@name='http']/parameter[@name='proxyPort']");
        Object obj;
        if ((obj = xp.selectSingleNode(documentElement)) != null) {
            httpPortEle = (OMElement) obj;
        }
        xp = new AXIOMXPath(
                "/axisconfig/transportReceiver[@name='https']/parameter[@name='proxyPort']");
        if ((obj = xp.selectSingleNode(documentElement)) != null) {
            httpsPortEle = (OMElement) obj;
        }
        return new Axis2XMLInfo(servicePath, Integer.parseInt(httpsPortEle.getText()),
                                Integer.parseInt(httpPortEle.getText()));

    }

    private boolean isURL(String axis2XMLLocation) {
        try {
            new URL(axis2XMLLocation);
            return true;
        } catch (MalformedURLException e) {
            return false;
        }
    }

    private static class Axis2XMLInfo {
        private String servicePath;
        private int httpsPort;
        private int httpPort;

        public Axis2XMLInfo(String servicePath, int httpsPort, int httpPort) {
            this.servicePath = servicePath;
            this.httpsPort = httpsPort;
            this.httpPort = httpPort;
        }

        public String getServicePath() {
            return servicePath;
        }

        public int getHttpsPort() {
            return httpsPort;
        }

        public int getHttpPort() {
            return httpPort;
        }
    }
}