/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.serverinfo;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMText;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jaxen.JaxenException;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.XPath;
import org.wso2.wsas.transport.ServerPropertyKeys;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;
import java.io.FileInputStream;
import java.util.List;
import java.util.Properties;

/**
 *
 */
public class GeronimoServerInfo implements ServerInfo {

    public static final String GERONIMO_CONFIG_NAMESPACE =
            "http://geronimo.apache.org/xml/ns/attributes-1.1";

    private int httpPort = 8080;
    private int httpsPort = 8443;

    private Properties props;
    private static Log log = LogFactory.getLog(GeronimoServerInfo.class);


    public int getHttpPort() {
        return this.httpPort;
    }

    public int getHttpsPort() {
        return this.httpsPort;
    }

    public void setProperties(Properties props) {
        this.props = props;
    }

    public void process() {

        String configXml = props.getProperty(ServerPropertyKeys.GERONIMO_CONFIG_XML);
        String geronimoVersion = props.getProperty(ServerPropertyKeys.APPSERVER_VERSION);
        try {

            XMLStreamReader parser = XMLInputFactory.newInstance()
                    .createXMLStreamReader(new FileInputStream(configXml));
            StAXOMBuilder staxOMBuilder = new StAXOMBuilder(parser);
            OMElement docElement = staxOMBuilder.getDocumentElement();

            OMElement omElement =
                    getOMElement("//ns:module[@name='geronimo/tomcat/" + geronimoVersion + "/car']", docElement);

            if (omElement != null) {
                //i.e this is a tomcat server
                if (log.isDebugEnabled()) {
                    log.debug("Going to process a Geronimo-Tomcat container");
                }

                // default web container is always TomcatWebConnector
                OMElement defaultWebConnector =
                        getOMElement("//ns:gbean[@name='TomcatWebConnector']", omElement);
                if (defaultWebConnector != null) {
                    httpPort = Integer.parseInt(
                            getNodeString("//ns:gbean[@name='TomcatWebConnector']/ns:attribute[@name='port']/text()",
                                          omElement));


                } else {
                    // i.e default tomcat connector is not there we have to find another node
                    // then we have to find the gbean under tomcat module
                    // the find the port number of that gbean
                    List gbeanList = getNodes("//ns:module[@name='geronimo/tomcat/" + geronimoVersion + "/car']/" +
                                              "ns:gbean[ns:attribute[@name='protocol' and " +
                                              "text()='HTTP']]/ns:attribute[@name='port']/text()",
                                              omElement);
                    if (gbeanList.size() > 0) {
                        OMText value = (OMText) gbeanList.get(0);
                        httpsPort = Integer.parseInt(value.getText());
                    } else {
                        log.error("Could not find the http port number for Geronimo-Tomcat container");
                    }
                }

                log.info("Http port for Geromimo Tomcat server ==> " + httpPort);

                OMElement defaultSSLConnector = getOMElement("//ns:gbean[@name='TomcatWebSSLConnector']", omElement);
                if (defaultSSLConnector != null) {
                    httpsPort = Integer.parseInt(
                            getNodeString("//ns:gbean[@name='TomcatWebSSLConnector']/ns:attribute[@name='port']/text()",
                                          omElement));
                } else {
                    List gbeanList = getNodes("//ns:module[@name='geronimo/tomcat/" + geronimoVersion + "/car']/" +
                                              "ns:gbean[ns:attribute[@name='protocol' and " +
                                              "text()='HTTPS']]/ns:attribute[@name='port']/text()",
                                              omElement);
                    if (gbeanList.size() > 0) {
                        OMText value = (OMText) gbeanList.get(0);
                        httpsPort = Integer.parseInt(value.getText());
                    } else {
                        log.error("Could not find the https port number for Geronimo-Tomcat container");
                    }
                }

                log.info("Https port for Geronimo-Tomcat server ==> " + httpsPort);
            } else {
                // i.e this is a jetty server
                omElement = getOMElement("//ns:module[@name='geronimo/jetty/" + geronimoVersion + "car']", docElement);
                if (omElement != null) {
                    OMElement defaultWebConnector = getOMElement("//ns:gbean[@name='JettyWebConnector']",
                                                                 omElement);
                    if (defaultWebConnector != null) {
                        httpPort = Integer.parseInt(
                                getNodeString("//ns:gbean[@name='JettyWebConnector']/ns:attribute[@name='port']/text()",
                                              omElement));

                    } else {
                        // i.e default tomcat connector is not there we have to find another node
                        List gbeanList = getNodes("//ns:module[@name='geronimo/jetty/" + geronimoVersion + "car']/" +
                                                  "ns:gbean[ns:attribute[@name='protocol' and " +
                                                  "text()='HTTPS']]/ns:attribute[@name='port']/text()",
                                                  omElement);
                        if (gbeanList.size() > 0) {
                            OMText value = (OMText) gbeanList.get(0);
                            httpPort = Integer.parseInt(value.getText());
                        } else {
                            log.error("Could not find the http port number for Geronimo-Jetty container");
                        }
                    }

                    log.info("Http port for Geromimo Jetty server ==> " + httpPort);

                    OMElement defaultSSLConnector = getOMElement("//ns:gbean[@name='JettySSLConnector']",
                                                                 omElement);
                    if (defaultSSLConnector != null) {
                        httpsPort = Integer.parseInt(
                                getNodeString("//ns:gbean[@name='JettySSLConnector']/ns:attribute[@name='port']/text()",
                                              omElement));
                    } else {
                        List gbeanList = getNodes("//ns:module[@name='geronimo/jetty/" + geronimoVersion + "car']/" +
                                                  "ns:gbean[ns:attribute[@name='protocol' " +
                                                  "and text()='HTTPS']]/ns:attribute[@name='port']/text()",
                                                  omElement);
                        if (gbeanList.size() > 0) {
                            OMText value = (OMText) gbeanList.get(0);
                            httpsPort = Integer.parseInt(value.getText());
                        } else {
                            log.error("Could not find the http port number for Geronimo-Jetty container");
                        }
                    }

                    if (log.isDebugEnabled()) {
                        log.debug("Https port for Geromimo-Jetty server ==> " + httpsPort);
                    }
                } else {
                    log.warn("could not find the http and https ports for Geronimo server");
                }

            }

        } catch (Exception e) {
            e.printStackTrace();
            String msg = "Cannot read HTTP and HTTPS ports from server.xml";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        }

    }

    private List getNodes(String xpathString, OMElement parentElement) throws JaxenException {
        XPath xpath = getXPath(xpathString);
        return xpath.selectNodes(parentElement);
    }

    private String getNodeString(String xpathString,
                                 OMElement parentElement) throws JaxenException {
        XPath xpath = getXPath(xpathString);
        return xpath.stringValueOf(parentElement);
    }

    private OMElement getOMElement(String xpathString,
                                   OMElement parentElement) throws JaxenException {
        XPath xpath = getXPath(xpathString);
        return (OMElement) xpath.selectSingleNode(parentElement);
    }

    private XPath getXPath(String xpathString) throws JaxenException {
        SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();
        nsCtx.addNamespace("ns", GERONIMO_CONFIG_NAMESPACE);
        XPath xpath = new AXIOMXPath(xpathString);
        xpath.setNamespaceContext(nsCtx);
        return xpath;
    }

}
