/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.security.pox;

import org.apache.axiom.om.impl.dom.jaxp.DocumentBuilderFactoryImpl;
import org.apache.axiom.om.util.Base64;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.HandlerDescription;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.Handler;
import org.apache.axis2.transport.http.HTTPConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.rampart.util.Axis2Util;
import org.apache.ws.security.WSConstants;
import org.apache.ws.security.WSSecurityException;
import org.apache.ws.security.message.WSSecHeader;
import org.apache.ws.security.message.WSSecTimestamp;
import org.apache.ws.security.message.WSSecUsernameToken;
import org.w3c.dom.Document;
import org.wso2.utils.NetworkUtils;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerManager;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Map;

/**
 * Handler to convert the HTTP basic auth information into
 * <code>wsse:UsernameToken</code>
 */
public class POXSecurityHandler implements Handler {

    private static Log log = LogFactory.getLog(POXSecurityHandler.class);

    private HandlerDescription description;

    private PersistenceManager pm = new PersistenceManager();

    /**
     * @see org.apache.axis2.engine.Handler#cleanup()
     */
    public void cleanup() {
    }

    /**
     * @see org.apache.axis2.engine.Handler#init(org.apache.axis2.description.HandlerDescription)
     */
    public void init(HandlerDescription description) {
        this.description = description;
    }

    /**
     * @see org.apache.axis2.engine.Handler#invoke(org.apache.axis2.context.MessageContext)
     */
    public InvocationResponse invoke(MessageContext msgCtx) throws AxisFault {

        //If this is not a REST/POX message simply return
        if (!msgCtx.isDoingREST() || !msgCtx.getIncomingTransportName().equals("https")) { // This handler will interecept REST requests only arriving via HTTPS
            return InvocationResponse.CONTINUE;
        }

        //Then check whether UT auth is enabled on the service
        AxisService service = msgCtx.getAxisService();

        if (service == null) {
            log.info("Service not dispatched");
            return InvocationResponse.CONTINUE;
        }

        ServiceDO serviceDO =
                pm.getService(service.getName(), ServiceIdentifierDO.EMPTY_SERVICE_VERSION);

        if (serviceDO != null) {
            if (!serviceDO.getIsUTAuthEnabled()) {
                //TODO: We specifically have to check for UT auth
                return InvocationResponse.CONTINUE;
            }
        } else {
            return InvocationResponse.CONTINUE;
        }

        HttpServletRequest request =
                (HttpServletRequest) msgCtx.getProperty(HTTPConstants.MC_HTTP_SERVLETREQUEST);
        String url = request.getRequestURL().toString();
        int index = url.indexOf("://");
        String protocol = url.substring(0, index);
        if (!protocol.equalsIgnoreCase("https")) {
            HttpServletResponse response =
                    (HttpServletResponse) msgCtx.getProperty(HTTPConstants.MC_HTTP_SERVLETRESPONSE);
            try {
                String requestURI = request.getRequestURI();
                response.sendRedirect("https://" +
                                      NetworkUtils.getLocalHostname() + ":" +
                                      ServerManager.getInstance().getHttpsPort() + requestURI);
                return InvocationResponse.ABORT;
            } catch (IOException e) {
                throw AxisFault.makeFault(e);
            }
        }

        try {

            // Set the DOM impl to DOOM
            DocumentBuilderFactoryImpl.setDOOMRequired(true);
            Map map = (Map) msgCtx.getProperty(MessageContext.TRANSPORT_HEADERS);
            String tmp = (String) map.get("Authorization");
            String username = null;
            String password = null;

            if (tmp != null) {
                tmp = tmp.trim();
            }
            if (tmp != null && tmp.startsWith("Basic ")) {
                tmp = new String(Base64.decode(tmp.substring(6)));
                int i = tmp.indexOf(':');
                if (i == -1) {
                    username = tmp;
                } else {
                    username = tmp.substring(0, i);
                }

                if (i != -1) {
                    password = tmp.substring(i + 1);
                    if (password != null && password.equals("")) {
                        password = null;
                    }
                }
            }

            //If the user information is not available
            //Throw an error
            if (username == null || password == null ||
                password.trim().length() == 0 || username.trim().length() == 0) {
                msgCtx.setProperty(org.apache.axis2.Constants.HTTP_RESPONSE_STATE,
                                   String.valueOf(HttpServletResponse.SC_UNAUTHORIZED));
                msgCtx.setProperty(org.apache.axis2.Constants.HTTP_BASIC_AUTH_REALM,
                                   ServerConfiguration.getInstance().getFirstProperty("Name"));
                throw new AxisFault("Authentication Failure");
            }

            Document doc = Axis2Util.getDocumentFromSOAPEnvelope(msgCtx.getEnvelope(),
                                                                 false);

            WSSecHeader secHeader = new WSSecHeader();
            secHeader.insertSecurityHeader(doc);

            WSSecUsernameToken utBuilder = new WSSecUsernameToken();
            utBuilder.setPasswordType(WSConstants.PASSWORD_TEXT);
            utBuilder.setUserInfo(username, password);
            utBuilder.build(doc, secHeader);

            WSSecTimestamp tsBuilder = new WSSecTimestamp();
            tsBuilder.build(doc, secHeader);

            /**
             * Set the new SOAPEnvelope
             */
            msgCtx.setEnvelope(Axis2Util.getSOAPEnvelopeFromDOMDocument(doc, false));

        } catch (WSSecurityException wssEx) {
            throw new AxisFault("WSDoAllReceiver: Error in converting to Document", wssEx);
        } finally {
            DocumentBuilderFactoryImpl.setDOOMRequired(false);
        }
        return InvocationResponse.CONTINUE;
    }

    public void flowComplete(MessageContext msgContext) {
        //To change body of implemented methods use File | Settings | File Templates.
    }

    /**
     * @see org.apache.axis2.engine.Handler#getHandlerDesc()
     */
    public HandlerDescription getHandlerDesc() {
        return this.description;
    }

    /**
     * @see org.apache.axis2.engine.Handler#getName()
     */
    public String getName() {
        return "REST/POX Security handler";
    }

    /**
     * @see org.apache.axis2.engine.Handler#getParameter(java.lang.String)
     */
    public Parameter getParameter(String name) {
        return this.description.getParameter(name);
    }
}
