/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.security;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ws.security.WSPasswordCallback;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.Set;

/**
 * The password callback handler to be used to enable UsernameToken
 * authentication for services.
 */
public class ServicePasswordCallbackHandler implements CallbackHandler {
    private static final Log log = LogFactory.getLog(ServicePasswordCallbackHandler.class);

    private String serviceId;
    private String serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;

    public ServicePasswordCallbackHandler(String serviceId, String serviceVersion) {
        this.serviceId = serviceId;
        this.serviceVersion = serviceVersion;
    }

    public ServicePasswordCallbackHandler(String serviceId) {
        this.serviceId = serviceId;
    }

    public void handle(Callback[] callbacks) throws IOException,
                                                    UnsupportedCallbackException {
        for (int i = 0; i < callbacks.length; i++) {
            if (callbacks[i] instanceof WSPasswordCallback) {
                WSPasswordCallback passwordCallback = (WSPasswordCallback) callbacks[i];

                String username = passwordCallback.getIdentifer();
                String storedPasswd = getPassword(username, this.serviceId, this.serviceVersion);

                switch (passwordCallback.getUsage()) {

                    case WSPasswordCallback.SIGNATURE:
                    case WSPasswordCallback.DECRYPT:
                        passwordCallback.setPassword(this.getPrivateKeyPassword(username));
                        break;

                    case WSPasswordCallback.USERNAME_TOKEN_UNKNOWN:

                        String receivedPasswd = passwordCallback.getPassword();
                        boolean authneticated = receivedPasswd != null &&
                                                receivedPasswd.equals(storedPasswd);
                        if (!authneticated) {
                            throw new UnsupportedCallbackException(callbacks[i],
                                                                   "check failed");
                        }

                        break;
                    default:

                        /*
                         * When the password is null WS4J reports an error saying
                         * no password available for the user. But its better if we 
                         * simply report authentication failure
                         * Therefore setting the password to be the empty string in this
                         * situation.
                         */
                        if (storedPasswd == null) {
                            storedPasswd = "";
                        }
                        passwordCallback.setPassword(storedPasswd);
                        break;

                }


            } else {
                throw new UnsupportedCallbackException(callbacks[i],
                                                       "Unrecognized Callback");
            }
        }
    }


    private String getPrivateKeyPassword(String username) throws IOException {
        PersistenceManager pm = new PersistenceManager();
        KeyStoreDO ksDO = pm.getService(serviceId, serviceVersion).getPrivateKeyStore();
        if (username.equals(ksDO.getPrivateKeyAlias())) {
            try {
                ServerConfiguration config = ServerConfiguration.getInstance();
                CryptoUtil cryptoUtil =
                        new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                       config.getFirstProperty("Security.KeyStore.Password"),
                                       config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                       config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                       config.getFirstProperty("Security.KeyStore.Type"));
                return new String(cryptoUtil.base64DecodeAndDecrypt(ksDO.getPrivateKeyPassword()));
            } catch (CryptoException e) {
                throw new IOException(e.getMessage());
            }
        } else {
            return null;
        }
    }

    /**
     * @param username
     * @return The password corresponding to the username
     */
    private String getPassword(String username, String serviceId, String serviceVersion) {
        PersistenceManager pm = new PersistenceManager();
        if (serviceVersion == null || serviceVersion.trim().length() == 0) {
            serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        }

        ServiceUserDO serviceUser = pm.getUser(username);
        if (serviceUser == null) { // user does not exist?
            return null;
        }

        ServiceDO service = pm.getService(serviceId, serviceVersion);
        Set assignedRoles = service.getRoles();

        // first check whether this user has a role which can access the service
        boolean isValidRoleFound = false;
        for (Iterator iterator = assignedRoles.iterator(); iterator.hasNext();) {
            ServiceUserRoleDO role = (ServiceUserRoleDO) iterator.next();
            if (pm.userHasRole(username, role)) {
                isValidRoleFound = true;
                break;
            }
        }

        // user's doesn't have a single role which is authorized to access this service?
        if (!isValidRoleFound) {
            // check whether this user can access the service
            if (!service.getUsers().contains(serviceUser)) { // user not authorized to access this service?
                return null;
            }
        }

        // If we reached here, the user is authorized to access this service, or one or more
        // of his roles are authorized to access this service
        String password = null;
        try {
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil =
                    new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                   config.getFirstProperty("Security.KeyStore.Password"),
                                   config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                   config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                   config.getFirstProperty("Security.KeyStore.Type"));
            password = new String(cryptoUtil.base64DecodeAndDecrypt(serviceUser.getPassword()));
        } catch (CryptoException e) {
            log.error("Exception occurred while decrypting password", e);
        }
        return password;
    }
}
