/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.security;

import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.engine.ListenerManager;
import org.apache.commons.discovery.Resource;
import org.apache.commons.discovery.ResourceIterator;
import org.apache.commons.discovery.jdk.JDKHooks;
import org.apache.commons.discovery.resource.DiscoverResources;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ws.security.WSSecurityException;
import org.apache.ws.security.components.crypto.CredentialException;
import org.apache.ws.security.components.crypto.Crypto;
import org.apache.ws.security.components.crypto.X509NameTokenizer;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;

import javax.xml.namespace.QName;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

/**
 * ServerCrypto implementation to support a collection of keystores holding different trusted certs
 * and CA certs
 */
public class ServerCrypto implements Crypto {

    private static Log log = LogFactory.getLog(ServerCrypto.class);

    public final static String PROP_ID_KEY_STORE = "org.wso2.wsas.security.wso2wsas.crypto.keystore";

    public final static String PROP_ID_TRUST_STORES = "org.wso2.wsas.security.wso2wsas.crypto.truststores";

    public final static String PROP_ID_CERT_PROVIDER = "org.wso2.wsas.security.wso2wsas.crypto.cert.provider";

    public final static String PROP_ID_DEFAULT_ALIAS = "org.wso2.wsas.security.wso2wsas.crypto.alias";

    public final static String PROP_ID_XKMS_SERVICE_URL = "org.wso2.wsas.security.wso2wsas.crypto.xkms.url";

    public final static String PROP_ID_XKMS_SERVICE_PASS_PHRASE = "org.wso2.wsas.security.wso2wsas.crypto.xkms.pass";
    private static final String SKI_OID = "2.5.29.14";


    private Properties properties = null;
    private KeyStore keystore = null;
    private KeyStore cacerts = null;
    private ArrayList trustStores = null;
    private static CertificateFactory certFact;
    private PersistenceManager persistenceMgr;
    private Boolean useXkms;

    public ServerCrypto(Properties prop) throws CredentialException,
                                                IOException {
        this(prop, ServerCrypto.class.getClassLoader());
    }

    public ServerCrypto(Properties prop, ClassLoader loader)
            throws CredentialException, IOException {

        persistenceMgr = new PersistenceManager();

        this.properties = prop;

        // Get the primary keystore
        String ksId = this.properties.getProperty(PROP_ID_KEY_STORE);
        this.keystore = findKeyStore(ksId, loader);

        // Get other keystores if available
        String trustStoreIds = this.properties
                .getProperty(PROP_ID_TRUST_STORES);
        if (trustStoreIds != null && trustStoreIds.trim().length() != 0) {
            String[] ids = trustStoreIds.trim().split(",");
            this.trustStores = new ArrayList(ids.length);
            for (int i = 0; i < ids.length; i++) {
                this.trustStores.add(i, findKeyStore(ids[i], loader));
            }
        }

        /**
         * Load cacerts
         */
        String cacertsPath = System.getProperty("java.home") + "/lib/security/cacerts";
        InputStream cacertsIs = new FileInputStream(cacertsPath);
        try {
            String cacertsPasswd = properties.getProperty("org.wso2.wsas.security.wso2wsas.crypto.cacerts.password", "changeit");
            cacerts = KeyStore.getInstance(KeyStore.getDefaultType());
            cacerts.load(cacertsIs, cacertsPasswd.toCharArray());

        } catch (GeneralSecurityException e) {
            log.error("", e);
            throw new CredentialException(3, "secError00", e);
        } finally {
            cacertsIs.close();
        }

    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#loadCertificate(java.io.InputStream)
     */
    public X509Certificate loadCertificate(InputStream in)
            throws WSSecurityException {
        X509Certificate cert;
        try {
            cert = (X509Certificate) getCertificateFactory()
                    .generateCertificate(in);
        } catch (CertificateException e) {
            throw new WSSecurityException(
                    WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                    "parseError");
        }
        return cert;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getX509Certificates(byte[],boolean)
     */
    public X509Certificate[] getX509Certificates(byte[] data, boolean reverse)
            throws WSSecurityException {
        InputStream in = new ByteArrayInputStream(data);
        CertPath path;
        try {
            path = getCertificateFactory().generateCertPath(in);
        } catch (CertificateException e) {
            throw new WSSecurityException(
                    WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                    "parseError");
        }
        List l = path.getCertificates();
        X509Certificate[] certs = new X509Certificate[l.size()];
        Iterator iterator = l.iterator();
        for (int i = 0; i < l.size(); i++) {
            certs[(reverse) ? (l.size() - 1 - i) : i] = (X509Certificate) iterator
                    .next();
        }
        return certs;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getCertificateData(boolean,
     *      java.security.cert.X509Certificate[])
     */
    public byte[] getCertificateData(boolean reverse, X509Certificate[] certs)
            throws WSSecurityException {
        Vector list = new Vector();
        for (int i = 0; i < certs.length; i++) {
            if (reverse) {
                list.insertElementAt(certs[i], 0);
            } else {
                list.add(certs[i]);
            }
        }
        try {
            CertPath path = getCertificateFactory().generateCertPath(list);
            return path.getEncoded();
        } catch (CertificateEncodingException e) {
            throw new WSSecurityException(
                    WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                    "encodeError");
        } catch (CertificateException e) {
            throw new WSSecurityException(
                    WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                    "parseError");
        }
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getPrivateKey(java.lang.String,
     *      java.lang.String)
     */
    public PrivateKey getPrivateKey(String alias, String password)
            throws Exception {
        if (alias == null) {
            throw new Exception("alias is null");
        }
        boolean b = keystore.isKeyEntry(alias);

        if (!b && useXKMS()) {

            PrivateKey privateKey = XKMSCryptoClient.getPrivateKey(alias,
                                                                   properties.getProperty(PROP_ID_XKMS_SERVICE_URL),
                                                                   properties.getProperty(PROP_ID_XKMS_SERVICE_PASS_PHRASE));

            if (privateKey == null) {
                log.error("Cannot find key for alias" + alias);
                throw new Exception("Cannot find key for alias: " + alias);
            }

            return privateKey;

        } else {

            if (!b) {
                log.error("Cannot find key for alias: " + alias);
                throw new Exception("Cannot find key for alias: " + alias);
            }
            Key keyTmp = keystore.getKey(alias, password.toCharArray());
            if (!(keyTmp instanceof PrivateKey)) {
                throw new Exception("Key is not a private key, alias: " + alias);
            }
            return (PrivateKey) keyTmp;
        }
    }

    /**
     * This first looks into the primary keystore and then looks at the other trust stores
     *
     * @see org.apache.ws.security.components.crypto.Crypto#getCertificates(String)
     */
    public X509Certificate[] getCertificates(String alias) throws WSSecurityException {
        Certificate[] certs = new Certificate[0];
        Certificate cert = null;
        try {
            if (this.keystore != null) {
                // There's a chance that there can only be a set of trust stores
                certs = keystore.getCertificateChain(alias);
                if (certs == null || certs.length == 0) {
                    // no cert chain, so lets check if getCertificate gives us a
                    // result.
                    cert = keystore.getCertificate(alias);
                }
            }

            if (certs == null && cert == null && this.trustStores != null) {
                // Now look into the trust stores
                Iterator trustStoreIter = this.trustStores.iterator();
                while (trustStoreIter.hasNext()) {
                    KeyStore store = (KeyStore) trustStoreIter.next();
                    certs = store.getCertificateChain(alias);
                    if (certs != null) {
                        break; // found the certs
                    } else {
                        cert = store.getCertificate(alias);
                    }
                }
            }

            if (certs == null && cert == null && this.cacerts != null) {
                // There's a chance that there can only be a set of ca store
                certs = cacerts.getCertificateChain(alias);
                if (certs == null || certs.length == 0) {
                    // no cert chain, so lets check if getCertificate gives us a
                    // result.
                    cert = cacerts.getCertificate(alias);
                }
            }

            if (cert != null) {
                certs = new Certificate[]{cert};
            } else if (certs == null) {

                if (useXKMS()) {
                    return XKMSCryptoClient.getCertificates(alias, properties
                            .getProperty(PROP_ID_XKMS_SERVICE_URL));
                }

                // At this pont we don't have certs or a cert
                // return null;
            }
        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE, "keystore");
        }

        X509Certificate[] x509certs = new X509Certificate[0];
        if (certs != null) {
            x509certs = new X509Certificate[certs.length];
            for (int i = 0; i < certs.length; i++) {
                x509certs[i] = (X509Certificate) certs[i];
            }
        }
        return x509certs;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasForX509Cert(java.security.cert.Certificate)
     */
    public String getAliasForX509Cert(Certificate cert)
            throws WSSecurityException {
        try {
            String alias = null;

            if (this.keystore != null) {
                alias = keystore.getCertificateAlias(cert);

                // Use brute force search
                if (alias == null) {
                    alias = findAliasForCert(this.keystore, cert);
                }
            }

            // Check the trust stores
            if (alias == null && this.trustStores != null) {
                for (Iterator trustStoreIter = this.trustStores.iterator(); trustStoreIter
                        .hasNext();) {
                    KeyStore store = (KeyStore) trustStoreIter.next();
                    alias = store.getCertificateAlias(cert);
                    if (alias != null) {
                        break;
                    }
                }
            }

            // Use brute force search on the trust stores
            if (alias == null && this.trustStores != null) {
                for (Iterator trustStoreIter = this.trustStores.iterator(); trustStoreIter
                        .hasNext();) {
                    KeyStore store = (KeyStore) trustStoreIter.next();
                    alias = this.findAliasForCert(store, cert);
                    if (alias != null) {
                        break;
                    }
                }
            }

            if (alias == null && this.cacerts != null) {
                alias = cacerts.getCertificateAlias(cert);

                // Use brute force search
                if (alias == null) {
                    alias = findAliasForCert(this.cacerts, cert);
                }
            }


            if (alias != null) {
                return alias;
            }

        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "keystore");
        }

        if (useXKMS()) {
            return XKMSCryptoClient.getAliasForX509Certificate(
                    (X509Certificate) cert, properties
                    .getProperty(PROP_ID_XKMS_SERVICE_URL));
        }

        return null;
    }

    private String findAliasForCert(KeyStore ks, Certificate cert)
            throws KeyStoreException {
        Enumeration e = ks.aliases();
        while (e.hasMoreElements()) {
            String alias = (String) e.nextElement();
            X509Certificate cert2 = (X509Certificate) ks.getCertificate(alias);
            if (cert2.equals(cert)) {
                return alias;
            }
        }
        return null;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasForX509Cert(java.lang.String)
     */
    public String getAliasForX509Cert(String issuer) throws WSSecurityException {
        return getAliasForX509Cert(issuer, null, false);
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasForX509Cert(java.lang.String,
     *      java.math.BigInteger)
     */
    public String getAliasForX509Cert(String issuer, BigInteger serialNumber)
            throws WSSecurityException {
        return getAliasForX509Cert(issuer, serialNumber, true);
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasForX509Cert(byte[])
     */
    public String getAliasForX509Cert(byte[] skiBytes)
            throws WSSecurityException {
        try {

            Certificate cert;
            for (Enumeration e = keystore.aliases(); e.hasMoreElements();) {
                String alias = (String) e.nextElement();
                Certificate[] certs = this.getCertificates(alias);
                if (certs == null || certs.length == 0) {
                    return null;
                } else {
                    cert = certs[0];
                }
                if (!(cert instanceof X509Certificate)) {
                    continue;
                }
                byte[] data = getSKIBytesFromCert((X509Certificate) cert);
                if (data.length != skiBytes.length) {
                    continue;
                }
                if (Arrays.equals(data, skiBytes)) {
                    return alias;
                }
            }

        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "keystore");
        }

        if (useXKMS()) {
            return XKMSCryptoClient.getAliasForX509Certificate(skiBytes,
                                                               properties.getProperty(PROP_ID_XKMS_SERVICE_URL));
        }

        return null;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getDefaultX509Alias()
     */
    public String getDefaultX509Alias() {
        return this.properties.getProperty(PROP_ID_DEFAULT_ALIAS);
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getSKIBytesFromCert(java.security.cert.X509Certificate)
     */
    public byte[] getSKIBytesFromCert(X509Certificate cert)
            throws WSSecurityException {
        /*
         * Gets the DER-encoded OCTET string for the extension value (extnValue)
         * identified by the passed-in oid String. The oid string is represented
         * by a set of positive whole numbers separated by periods.
         */
        byte[] derEncodedValue = cert.getExtensionValue(SKI_OID);

        if (cert.getVersion() < 3 || derEncodedValue == null) {
            PublicKey key = cert.getPublicKey();
            if (!(key instanceof RSAPublicKey)) {
                throw new WSSecurityException(1, "noSKIHandling",
                                              new Object[]{"Support for RSA key only"});
            }
            byte[] encoded = key.getEncoded();
            // remove 22-byte algorithm ID and header
            byte[] value = new byte[encoded.length - 22];
            System.arraycopy(encoded, 22, value, 0, value.length);
            MessageDigest sha;
            try {
                sha = MessageDigest.getInstance("SHA-1");
            } catch (NoSuchAlgorithmException ex) {
                throw new WSSecurityException(1, "noSKIHandling",
                                              new Object[]{"Wrong certificate version (<3) and no "
                                                           + "SHA1 message digest availabe"});
            }
            sha.reset();
            sha.update(value);
            return sha.digest();
        }

        /**
         * Strip away first four bytes from the DerValue (tag and length of
         * ExtensionValue OCTET STRING and KeyIdentifier OCTET STRING)
         */
        byte abyte0[] = new byte[derEncodedValue.length - 4];

        System.arraycopy(derEncodedValue, 4, abyte0, 0, abyte0.length);
        return abyte0;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasForX509CertThumb(byte[])
     */
    public String getAliasForX509CertThumb(byte[] thumb)
            throws WSSecurityException {
        Certificate cert;
        MessageDigest sha;
        try {
            sha = MessageDigest.getInstance("SHA-1");
        } catch (NoSuchAlgorithmException e1) {
            throw new WSSecurityException(0, "noSHA1availabe");
        }
        try {
            for (Enumeration e = keystore.aliases(); e.hasMoreElements();) {
                String alias = (String) e.nextElement();
                Certificate[] certs = this.getCertificates(alias);
                if (certs == null || certs.length == 0) {
                    return null;
                } else {
                    cert = certs[0];
                }
                if (!(cert instanceof X509Certificate)) {
                    continue;
                }
                sha.reset();
                try {
                    sha.update(cert.getEncoded());
                } catch (CertificateEncodingException e1) {
                    throw new WSSecurityException(
                            WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                            "encodeError");
                }
                byte[] data = sha.digest();

                if (Arrays.equals(data, thumb)) {
                    return alias;
                }
            }
        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "keystore");
        }
        return null;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getKeyStore()
     */
    public KeyStore getKeyStore() {
        return this.keystore;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getCertificateFactory()
     */
    public CertificateFactory getCertificateFactory()
            throws WSSecurityException {
        if (certFact == null) {
            try {
                String provider = properties.getProperty(PROP_ID_CERT_PROVIDER);
                if (provider == null || provider.length() == 0) {
                    certFact = CertificateFactory.getInstance("X.509");
                } else {
                    certFact = CertificateFactory
                            .getInstance("X.509", provider);
                }
            } catch (CertificateException e) {
                throw new WSSecurityException(
                        WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                        "unsupportedCertType");
            } catch (NoSuchProviderException e) {
                throw new WSSecurityException(
                        WSSecurityException.SECURITY_TOKEN_UNAVAILABLE,
                        "noSecProvider");
            }
        }
        return certFact;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#validateCertPath(java.security.cert.X509Certificate[])
     */
    public boolean validateCertPath(X509Certificate[] certs) throws WSSecurityException {

        boolean result;
        if (useXKMS()) {
            result = XKMSCryptoClient.validateCertPath(certs, properties
                    .getProperty(PROP_ID_XKMS_SERVICE_URL));
            if (result) {
                return true;
            }
        }

        result = this.validateCertPath(this.keystore, certs);

        if (!result) {
            Iterator trustStoreIter = this.trustStores.iterator();
            while (!result) {
                result = this.validateCertPath(
                        (KeyStore) trustStoreIter.next(), certs);
            }
        }

        if (!result && cacerts != null) {
            result = this.validateCertPath(this.cacerts, certs);
        }

        return result;
    }

    /**
     * @see org.apache.ws.security.components.crypto.Crypto#getAliasesForDN(java.lang.String)
     */
    public String[] getAliasesForDN(String subjectDN)
            throws WSSecurityException {

        // Store the aliases found
        Vector aliases = new Vector();
        Certificate cert;

        // The DN to search the keystore for
        Vector subjectRDN = splitAndTrim(subjectDN);

        // Look at every certificate in the keystore
        try {
            for (Enumeration e = keystore.aliases(); e.hasMoreElements();) {
                String alias = (String) e.nextElement();

                Certificate[] certs = this.getCertificates(alias);
                if (certs == null || certs.length == 0) {
                    return null;
                } else {
                    cert = certs[0];
                }
                if (cert instanceof X509Certificate) {
                    Vector foundRDN = splitAndTrim(((X509Certificate) cert)
                            .getSubjectDN().getName());

                    if (subjectRDN.equals(foundRDN)) {
                        aliases.add(alias);
                    }
                }
            }
        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "keystore");
        }

        if (aliases.isEmpty() && useXKMS()) {
            String[] xkmsAliases = XKMSCryptoClient.getAliasesForDN(subjectDN,
                                                                    properties.getProperty(PROP_ID_XKMS_SERVICE_URL));
            if (xkmsAliases != null) {
                for (int i = 0; i < xkmsAliases.length; i++) {
                    aliases.add(xkmsAliases[i]);
                }
            }

        }

        // Convert the vector into an array
        String[] result = new String[aliases.size()];
        for (int i = 0; i < aliases.size(); i++) {
            result[i] = (String) aliases.elementAt(i);
        }
        return result;

    }

    private KeyStore findKeyStore(String ksId, ClassLoader loader)
            throws CredentialException, IOException {

        String location = this.getKeyStoreLocation(ksId);
        InputStream is = null;

        /**
         * Look for the keystore in classpaths
         */
        DiscoverResources disc = new DiscoverResources();
        disc.addClassLoader(JDKHooks.getJDKHooks()
                .getThreadContextClassLoader());
        disc.addClassLoader(loader);
        ResourceIterator iterator = disc.findResources(location);
        if (iterator.hasNext()) {
            Resource resource = iterator.nextResource();
            is = resource.getResourceAsStream();
        }

        /**
         * If we don't find it, then look on the file system.
         */
        if (is == null) {
            try {
                is = new FileInputStream(location);
            } catch (Exception e) {
                throw new CredentialException(3, "proxyNotFound",
                                              new Object[]{location});
            }
        }

        /**
         * Load the keystore
         */
        try {
            return load(is, ksId);
        } finally {
            is.close();
        }
    }

    /**
     * Loads the the keystore from an <code>InputStream </code>. <p/>
     *
     * @param input <code>InputStream</code> to read from
     * @param ksId  Key store ID
     * @return KeyStore
     * @throws CredentialException If credential are wrong
     */
    private KeyStore load(InputStream input, String ksId)
            throws CredentialException {
        if (input == null) {
            throw new IllegalArgumentException("input stream cannot be null");
        }
        try {
            KeyStore tempStore;

            String provider = null;
            String keyStoreProvider = this.getKeyStoreProvider(ksId);

            if (keyStoreProvider != null) {
                provider = properties.getProperty(keyStoreProvider);
            }

            if (provider == null || provider.trim().length() == 0) {
                tempStore = KeyStore.getInstance(properties.getProperty(this
                        .getKeyStoreLocation(ksId), KeyStore.getDefaultType()));
            } else {
                tempStore = KeyStore.getInstance(properties.getProperty(this
                        .getKeyStoreType(ksId), KeyStore.getDefaultType()),
                                                 provider);
            }
            String password = this.getKeyStorePassword(ksId);
            tempStore.load(input,
                           (password == null || password.length() == 0) ? new char[0]
                           : password.toCharArray());
            return tempStore;
        } catch (IOException e) {
            e.printStackTrace();
            throw new CredentialException(3, "ioError00", e);
        } catch (GeneralSecurityException e) {
            e.printStackTrace();
            throw new CredentialException(3, "secError00", e);
        } catch (Exception e) {
            e.printStackTrace();
            throw new CredentialException(-1, "error00", e);
        }
    }

    private String getAliasForX509Cert(String issuer, BigInteger serialNumber,
                                       boolean useSerialNumber) throws WSSecurityException {
        Vector issuerRDN = splitAndTrim(issuer);
        X509Certificate x509cert;
        Vector certRDN;
        Certificate cert;
        try {
            for (Enumeration e = keystore.aliases(); e.hasMoreElements();) {
                String alias = (String) e.nextElement();
                Certificate[] certs = this.getCertificates(alias);

                if (certs == null || certs.length == 0) {
                    return null;
                } else {
                    cert = certs[0];
                }
                if (!(cert instanceof X509Certificate)) {
                    continue;
                }
                x509cert = (X509Certificate) cert;
                if (useSerialNumber
                    && x509cert.getSerialNumber().compareTo(serialNumber) == 0) {
                    certRDN = splitAndTrim(x509cert.getIssuerDN().getName());
                    if (certRDN.equals(issuerRDN)) {
                        return alias;
                    }
                }
            }
        } catch (KeyStoreException e) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "keystore");
        }
        return null;
    }

    private Vector splitAndTrim(String inString) {
        X509NameTokenizer nmTokens = new X509NameTokenizer(inString);
        Vector vr = new Vector();

        while (nmTokens.hasMoreTokens()) {
            vr.add(nmTokens.nextToken());
        }
        java.util.Collections.sort(vr);
        return vr;
    }

    private boolean validateCertPath(KeyStore ks, Certificate[] certs)
            throws WSSecurityException {

        try {

            // Generate cert path
            java.util.List certList = java.util.Arrays.asList(certs);
            CertPath path = this.getCertificateFactory().generateCertPath(
                    certList);

            // Use the certificates in the keystore as TrustAnchors
            PKIXParameters param = new PKIXParameters(ks);

            // Do not check a revocation list
            param.setRevocationEnabled(false);

            // Verify the trust path using the above settings
            String provider = properties
                    .getProperty("org.apache.ws.security.crypto.merlin.cert.provider");
            CertPathValidator certPathValidator;
            if (provider == null || provider.length() == 0) {
                certPathValidator = CertPathValidator.getInstance("PKIX");
            } else {
                certPathValidator = CertPathValidator.getInstance("PKIX",
                                                                  provider);
            }
            certPathValidator.validate(path, param);
        } catch (NoSuchProviderException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        } catch (NoSuchAlgorithmException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        } catch (CertificateException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        } catch (InvalidAlgorithmParameterException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        } catch (CertPathValidatorException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        } catch (KeyStoreException ex) {
            throw new WSSecurityException(WSSecurityException.FAILURE,
                                          "certpath", new Object[]{ex.getMessage()}, ex);
        }

        return true;
    }

    // ////////////////////////////////////////////////////////////////////////
    // //////////////////WSO2 WSAS DB access methods///////////////////////////
    // ////////////////////////////////////////////////////////////////////////

    /**
     * Returns the keystore password
     *
     * @param ksId keystore identifier
     * @return the Key Store password
     * @throws ServerException If password decryption fails
     */
    private String getKeyStorePassword(String ksId) throws ServerException {
        KeyStoreDO keyStore = persistenceMgr.getKeyStore(ksId);
        if (keyStore != null) {
            String storePassword = keyStore.getStorePassword();
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil = new CryptoUtil(new File(config
                    .getFirstProperty("Security.KeyStore.Location"))
                    .getAbsolutePath(), config
                    .getFirstProperty("Security.KeyStore.Password"), config
                    .getFirstProperty("Security.KeyStore.KeyAlias"), config
                    .getFirstProperty("Security.KeyStore.KeyPassword"), config
                    .getFirstProperty("Security.KeyStore.Type"));
            String passwd;
            try {
                passwd = new String(cryptoUtil
                        .base64DecodeAndDecrypt(storePassword));
            } catch (CryptoException e) {
                throw new ServerException(e);
            }
            return passwd;
        }
        return null;
    }

    /**
     * Returns the type of the keystore
     *
     * @param ksId keystore identifier
     * @return The Key Store Type
     */
    private String getKeyStoreType(String ksId) {
        KeyStoreDO keyStore = persistenceMgr.getKeyStore(ksId);
        if (keyStore != null) {
            return keyStore.getKeyStoreType();
        }
        return null;
    }

    /**
     * Get the provider name of the keystore
     *
     * @param ksId keystore identifier
     * @return The Key Store Provider
     */
    private String getKeyStoreProvider(String ksId) {
        KeyStoreDO keyStore = persistenceMgr.getKeyStore(ksId);
        if (keyStore != null) {
            return keyStore.getProvider();
        }
        return null;
    }

    /**
     * Obtains the keystore location from the WSO2 WSAS database
     *
     * @param ksId keystore identifier
     * @return path to access the keystore
     */
    private String getKeyStoreLocation(String ksId) {
        KeyStoreDO keyStore = persistenceMgr.getKeyStore(ksId);
        if (keyStore != null) {
            return keyStore.getFilePath();
        }
        return null;
    }

    private boolean useXKMS() {

        if (useXkms != null) {
            return useXkms.booleanValue();
        }

        AxisConfiguration axisConfiguration =
                ListenerManager.defaultConfigurationContext.getAxisConfiguration();
        Parameter parameter = axisConfiguration.getParameter("XKMSConfig");

        if (parameter == null) {
            useXkms = Boolean.FALSE;
            return useXkms.booleanValue();
        }

        OMElement parameterElement = parameter.getParameterElement();
        OMAttribute attribute = parameterElement.getAttribute(new QName("enabled"));

        if (attribute != null) {
            String value = attribute.getAttributeValue();
            useXkms = Boolean.valueOf(value);
        }

        OMElement urlElement = parameterElement
                .getFirstChildWithName(new QName("URL"));
        properties.setProperty(PROP_ID_XKMS_SERVICE_URL, urlElement.getText());

        OMElement passPhraseElement = parameterElement
                .getFirstChildWithName(new QName("PassPhrase"));
        properties.setProperty(PROP_ID_XKMS_SERVICE_PASS_PHRASE,
                               passPhraseElement.getText());

        return useXkms.booleanValue();
    }

}
