/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dataobject;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Data object representing a service in WSO2 WSAS
 */
public class ServiceDO extends AbstractDataObject {

    private boolean isUTAuthEnabled;  // Username Token Authentication Enabled
    private String documentation;
    private boolean isActive;  // Is service active
    private boolean isExposedOnAllTransports;  // Is service exposed on all available transports
    private ServiceIdentifierDO serviceIdentifierDO;
    private Set policies = new HashSet(); // one-many relationship between services & service policies
    private Set operations = new HashSet(); // one-many relationship between services & service operations
    private Set engagedModules = new HashSet(); // many-many relationship between services & modules (Service-Module Engagement)
    private ServiceGroupDO serviceGroup;
    private Set parameters = new HashSet(); // one-many relationship between Services and ServiceParameters
    private Set users = new HashSet(); // many-many relationship between services & service users
    private Set roles = new HashSet(); // many-to-many relationship between services & roles
    private Set transports = new HashSet(); // many-to-many relationship between services & transports
    private SecurityScenarioDO securityScenario; // 1-m relationship between Security Scenarios & Services

    private KeyStoreDO privateKeyStore; // 1-m relationship between KeyStores & Services
    private Set trustedCertStores = new HashSet(); // many-many relationship between trust stores & services

    /**
     * Constructs a new ServiceDO.
     */
    public ServiceDO() {
    }

    /**
     * Getter for property 'serviceIdentifierDO'.
     *
     * @return Value for property 'serviceIdentifierDO'.
     * @see #setServiceIdentifierDO
     */
    public ServiceIdentifierDO getServiceIdentifierDO() {
        return serviceIdentifierDO;
    }

    /**
     * Setter for property 'serviceIdentifierDO'.
     *
     * @param serviceIdentifierDO Value to set for property 'serviceIdentifierDO'.
     * @see #getServiceIdentifierDO
     */
    public void setServiceIdentifierDO(ServiceIdentifierDO serviceIdentifierDO) {
        this.serviceIdentifierDO = serviceIdentifierDO;
    }

    /**
     * Getter for property 'isUTAuthEnabled'.
     *
     * @return Value for property 'isUTAuthEnabled'.
     * @see #setIsUTAuthEnabled
     */
    public boolean getIsUTAuthEnabled() {
        return isUTAuthEnabled;
    }

    /**
     * Setter for property 'isUTAuthEnabled'.
     *
     * @param utAuthEnabled Value to set for property 'isUTAuthEnabled'.
     * @see #getIsUTAuthEnabled
     */
    public void setIsUTAuthEnabled(boolean utAuthEnabled) {
        this.isUTAuthEnabled = utAuthEnabled;
    }

    /**
     * Getter for property 'isExposedOnAllTransports'.
     *
     * @return Value for property 'isExposedOnAllTransports'.
     * @see #setIsExposedOnAllTransports
     */
    public boolean getIsExposedOnAllTransports() {
        return isExposedOnAllTransports;
    }

    /**
     * Setter for property 'isExposedOnAllTransports'.
     *
     * @param isExposedOnAllTransports Value to set for property 'isExposedOnAllTransports'.
     * @see #getIsExposedOnAllTransports
     */
    public void setIsExposedOnAllTransports(boolean isExposedOnAllTransports) {
        this.isExposedOnAllTransports = isExposedOnAllTransports;
    }

    /**
     * Getter for property 'isActive'.
     *
     * @return Value for property 'isActive'.
     * @see #setIsActive
     */
    public boolean getIsActive() {
        return isActive;
    }

    /**
     * Setter for property 'isActive'.
     *
     * @param isActive Value to set for property 'isActive'.
     * @see #getIsActive
     */
    public void setIsActive(boolean isActive) {
        this.isActive = isActive;
    }

    /**
     * Getter for property 'users'.
     *
     * @return Value for property 'users'.
     * @see #setUsers
     */
    public Set getUsers() {
        return users;
    }

    /**
     * Getter for property 'policies'.
     *
     * @return Value for property 'policies'.
     * @see #setPolicies
     */
    public Set getPolicies() {
        return policies;
    }

    /**
     * Setter for property 'policies'.
     *
     * @param policies Value to set for property 'policies'.
     * @see #getPolicies
     */
    public void setPolicies(Set policies) {
        this.policies = policies;
    }

    /**
     * Getter for property 'operations'.
     *
     * @return Value for property 'operations'.
     * @see #setOperations
     */
    public Set getOperations() {
        return operations;
    }

    /**
     * Method to get a single operationDO by specifying name.
     *
     * @param operationName
     * @return OperationDO[] array
     */

    public OperationDO[] getOperation(String operationName) {

        for (Iterator iter = operations.iterator(); iter.hasNext();) {
            OperationDO opDO = (OperationDO) iter.next();
            if (opDO.getName().equals(operationName)) {
                return new OperationDO[]{opDO};
            }
        }
        return new OperationDO[0];
    }

    /**
     * Setter for property 'operations'.
     *
     * @param operations Value to set for property 'operations'.
     * @see #getOperations
     */
    public void setOperations(Set operations) {
        this.operations = operations;
    }

    /**
     * Setter for property 'users'.
     *
     * @param users Value to set for property 'users'.
     * @see #getUsers
     */
    public void setUsers(Set users) {
        this.users = users;
    }

    /**
     * Getter for property 'parameters'.
     *
     * @return Value for property 'parameters'.
     * @see #setParameters
     */
    public Set getParameters() {
        return parameters;
    }

    /**
     * Retrun the ServiceParameterDO to the given para name
     *
     * @param paramName
     * @return ServiceParameterDO
     */
    public ServiceParameterDO getParameter(String paramName) {
        for (Iterator iterator = parameters.iterator(); iterator.hasNext();) {
            ServiceParameterDO serviceGroupParameterDO = (ServiceParameterDO) iterator.next();
            if (serviceGroupParameterDO.getName().equals(paramName)) {
                return serviceGroupParameterDO;
            }

        }
        return null;
    }

    /**
     * Setter for property 'parameters'.
     *
     * @param parameters Value to set for property 'parameters'.
     * @see #getParameters
     */
    public void setParameters(Set parameters) {
        this.parameters = parameters;
    }

    /**
     * Getter for property 'roles'.
     *
     * @return Value for property 'roles'.
     * @see #setRoles
     */
    public Set getRoles() {
        return roles;
    }

    /**
     * Setter for property 'roles'.
     *
     * @param roles Value to set for property 'roles'.
     * @see #getRoles
     */
    public void setRoles(Set roles) {
        this.roles = roles;
    }

    /**
     * Getter for property 'transports'.
     *
     * @return Value for property 'transports'.
     * @see #setTransports
     */
    public Set getTransports() {
        return transports;
    }

    /**
     * Setter for property 'transports'.
     *
     * @param transports Value to set for property 'transports'.
     * @see #getTransports
     */
    public void setTransports(Set transports) {
        this.transports = transports;
    }

    /**
     * Getter for property 'privateKeyStore'.
     *
     * @return Value for property 'privateKeyStore'.
     * @see #setPrivateKeyStore
     */
    public KeyStoreDO getPrivateKeyStore() {
        return privateKeyStore;
    }

    /**
     * Setter for property 'privateKeyStore'.
     *
     * @param privateKeyStore Value to set for property 'privateKeyStore'.
     * @see #getPrivateKeyStore
     */
    public void setPrivateKeyStore(KeyStoreDO privateKeyStore) {
        if (this.privateKeyStore != null && privateKeyStore == null) {
            this.privateKeyStore.getPkStoreServices().remove(this);
        }
        this.privateKeyStore = privateKeyStore;
    }

    /**
     * Getter for property 'trustStores'.
     *
     * @return Value for property 'trustStores'.
     * @see #setTrustedCertStores
     */
    public Set getTrustedCertStores() {
        return trustedCertStores;
    }

    /**
     * Setter for property 'trustStores'.
     *
     * @param trustedCertStores Value to set for property 'trustStores'.
     * @see #getTrustedCertStores
     */
    public void setTrustedCertStores(Set trustedCertStores) {
        this.trustedCertStores = trustedCertStores;
    }

    public void addRole(ServiceUserRoleDO role) {
        roles.add(role);
        role.getServices().add(this);
    }

    public void addTrustedCertStore(KeyStoreDO keyStore) {
        if (!trustedCertStores.contains(keyStore)) {
            this.trustedCertStores.add(keyStore);
            keyStore.getTrustStoreServices().add(this);
        }
    }

    public void removeAllTrustedCertStores() {
        for (Iterator iter = trustedCertStores.iterator(); iter.hasNext();) {
            KeyStoreDO keyStoreDO = (KeyStoreDO) iter.next();
            keyStoreDO.getTrustStoreServices().remove(this);
        }
        this.trustedCertStores.clear();
    }

    public void removeAllUsers() {
        for (Iterator iter = users.iterator(); iter.hasNext();) {
            ServiceUserDO userDO = (ServiceUserDO) iter.next();
            userDO.getServices().remove(this);
        }
        this.users.clear();
    }

    public void removeAllRoles() {
        for (Iterator iter = roles.iterator(); iter.hasNext();) {
            ServiceUserRoleDO roleDO = (ServiceUserRoleDO) iter.next();
            roleDO.getServices().remove(this);
        }
        this.roles.clear();
    }

    public void removeAllTransports() {
        for (Iterator iter = transports.iterator(); iter.hasNext();) {
            TransportDO transportDO = (TransportDO) iter.next();
            transportDO.getServices().remove(this);
        }
        this.transports.clear();
    }

    public void removeSecurityScenario() {
        if (securityScenario != null) {
            securityScenario.getServices().remove(this);
            securityScenario = null;
        }
    }

    public void removeAllRelationships() {
        removeAllRoles();
        removeAllTransports();
        removeAllTrustedCertStores();
        removeAllUsers();
        removeSecurityScenario();
    }

    public void addParameter(ServiceParameterDO param) {
        parameters.add(param);
        param.setService(this);
    }

    public void addUser(ServiceUserDO serviceUser) {
        users.add(serviceUser);
        serviceUser.getServices().add(this);
    }

    public void addPolicy(ServicePolicyDO policy) {
        policies.add(policy);
        policy.setService(this);
    }

    public void addOperation(OperationDO operation) {
        operations.add(operation);
        operation.setService(this);
    }

    public void addEngagedModule(ModuleDO module) {
        engagedModules.add(module);
        module.getEngagedServices().add(this);
    }

    public void addTransport(TransportDO transport) {
        if (transport != null) {
            transports.add(transport);
            transport.getServices().add(this);
        }
    }

    public void addTrustStore(KeyStoreDO trustStore) {
        trustedCertStores.add(trustStore);
    }

    /**
     * Getter for property 'engagedModules'.
     *
     * @return Value for property 'engagedModules'.
     * @see #setEngagedModules
     */
    public Set getEngagedModules() {
        return engagedModules;
    }

    /**
     * Setter for property 'engagedModules'.
     *
     * @param engagedModules Value to set for property 'engagedModules'.
     * @see #getEngagedModules
     */
    public void setEngagedModules(Set engagedModules) {
        this.engagedModules = engagedModules;
    }

    /**
     * Getter for property 'serviceGroup'.
     *
     * @return Value for property 'serviceGroup'.
     * @see #setServiceGroup
     */
    public ServiceGroupDO getServiceGroup() {
        return serviceGroup;
    }

    /**
     * Setter for property 'serviceGroup'.
     *
     * @param serviceGroup Value to set for property 'serviceGroup'.
     * @see #getServiceGroup
     */
    public void setServiceGroup(ServiceGroupDO serviceGroup) {
        this.serviceGroup = serviceGroup;
    }

    /**
     * Getter for property 'securityScenario'.
     *
     * @return Value for property 'securityScenario'.
     * @see #setSecurityScenario
     */
    public SecurityScenarioDO getSecurityScenario() {
        return securityScenario;
    }

    /**
     * Setter for property 'securityScenario'.
     *
     * @param securityScenario Value to set for property 'securityScenario'.
     * @see #getSecurityScenario
     */
    public void setSecurityScenario(SecurityScenarioDO securityScenario) {
        this.securityScenario = securityScenario;
    }

    /**
     * {@inheritDoc}
     */
    public int hashCode() {
        if (serviceIdentifierDO != null) {
            return serviceIdentifierDO.hashCode();
        }
        return 0;
    }

    /**
     * {@inheritDoc}
     */
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof ServiceDO) {
            ServiceDO serviceDO = (ServiceDO) obj;
            if (serviceIdentifierDO != null) {
                return serviceIdentifierDO.equals(serviceDO.getServiceIdentifierDO());
            }
        }
        return false;
    }

    public String getDocumentation() {
        return documentation;
    }

    public void setDocumentation(String documentation) {
        this.documentation = documentation;
    }
}
