/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dataobject;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.impl.dom.DOOMAbstractFactory;
import org.apache.axiom.om.util.Base64;
import org.apache.rahas.Token;
import org.apache.rahas.TrustException;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import java.io.ByteArrayInputStream;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;

/**
 *
 */
public class SecurityTokenDO extends AbstractDataObject {
    private String tokenId;
    private int state = -1;
    private String tokenStr;
    private String previousTokenStr;
    private String attachedReferenceStr;
    private String unattachedReferenceStr;
    private String secretStr;
    private Date created;
    private Date expires;
    private Set securityTokenProperties = new HashSet(); // One-many relationship between Tokens and Token Props

    public SecurityTokenDO() {
    }

    public SecurityTokenDO(String id, OMElement token, Date created, Date expires) {
        this.tokenId = id;
        this.created = created;
        this.expires = expires;
        this.tokenStr = token.toString();
    }

    public Token toToken() throws TrustException {
        Token token;
        try {
            token = new Token(tokenId, toOMElement(tokenStr), created, expires);
            token.setAttachedReference(toOMElement(attachedReferenceStr));
            token.setUnattachedReference(toOMElement(unattachedReferenceStr));
            token.setPreviousToken(toOMElement(previousTokenStr));
            token.setState(state);
            token.setSecret(Base64.decode(secretStr));

            // set the Token props
            Properties tokenProps = new Properties();
            for (Iterator iterator = securityTokenProperties.iterator(); iterator.hasNext();) {
                SecurityTokenPropertyDO prop = (SecurityTokenPropertyDO) iterator.next();
                tokenProps.setProperty(prop.getName(), prop.getValue());
            }
            token.setProperties(tokenProps);
        } catch (XMLStreamException e) {
            throw new TrustException("Could not construct token from TokenDO", e);
        }
        return token;
    }

    private OMElement toOMElement(String serializedEle) throws XMLStreamException {
        if (serializedEle == null) {
            return null;
        }
        XMLStreamReader reader =
                XMLInputFactory.newInstance().
                        createXMLStreamReader(new ByteArrayInputStream(serializedEle.getBytes()));

        StAXOMBuilder builder = new StAXOMBuilder(DOOMAbstractFactory.getOMFactory(),
                                                  reader);
        return builder.getDocumentElement();
    }

    public void update(Token token) {
        this.tokenStr = token.getToken().toString();
        this.previousTokenStr = (token.getPreviousToken() != null) ?
                                token.getPreviousToken().toString() : null;
        this.unattachedReferenceStr = (token.getUnattachedReference() != null) ?
                                      token.getUnattachedReference().toString() : null;
        this.attachedReferenceStr = (token.getAttachedReference() != null) ?
                                    token.getAttachedReference().toString() : null;
        this.secretStr = Base64.encode(token.getSecret());
        setExpires(token.getExpires());
        setState(token.getState());
        Properties tokenProps = token.getProperties();
        if (tokenProps != null) {
            for (Iterator iter = tokenProps.keySet().iterator();
                 iter.hasNext();) {
                String key = (String) iter.next();
                SecurityTokenPropertyDO prop =
                        new SecurityTokenPropertyDO(key, tokenProps.getProperty(key));
                addProperty(prop);
            }
        }
    }

    public String getTokenStr() {
        return tokenStr;
    }

    public void setTokenStr(String tokenStr) {
        this.tokenStr = tokenStr;
    }

    public String getSecretStr() {
        return secretStr;
    }

    public void setSecretStr(String secretStr) {
        this.secretStr = secretStr;
    }

    public String getPreviousTokenStr() {
        return previousTokenStr;
    }

    public void setPreviousTokenStr(String previousTokenStr) {
        this.previousTokenStr = previousTokenStr;
    }

    public String getAttachedReferenceStr() {
        return attachedReferenceStr;
    }

    public void setAttachedReferenceStr(String attachedReferenceStr) {
        this.attachedReferenceStr = attachedReferenceStr;
    }

    public String getUnattachedReferenceStr() {
        return unattachedReferenceStr;
    }

    public void setUnattachedReferenceStr(String unattachedReferenceStr) {
        this.unattachedReferenceStr = unattachedReferenceStr;
    }

    public String getTokenId() {
        return tokenId;
    }

    public void setTokenId(String tokenId) {
        this.tokenId = tokenId;
    }

    public int getState() {
        return state;
    }

    public void setState(int state) {
        this.state = state;
    }

    public Date getCreated() {
        return created;
    }

    public void setCreated(Date created) {
        this.created = created;
    }

    public Date getExpires() {
        return expires;
    }

    public void setExpires(Date expires) {
        this.expires = expires;
    }

    public Set getSecurityTokenProperties() {
        return securityTokenProperties;
    }

    public void setSecurityTokenProperties(Set securityTokenProperties) {
        this.securityTokenProperties = securityTokenProperties;
    }

    public void addProperty(SecurityTokenPropertyDO prop) {
        if (!securityTokenProperties.contains(prop)) {
            securityTokenProperties.add(prop);
            prop.setToken(this);
        } else {
            for (Iterator iter = securityTokenProperties.iterator(); iter.hasNext();) {
                SecurityTokenPropertyDO oldPropDO = (SecurityTokenPropertyDO) iter.next();
                if (oldPropDO.getName().equals(prop.getName())) {
                    oldPropDO.setValue(prop.getValue());
                }
            }
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final SecurityTokenDO that = (SecurityTokenDO) o;
        return tokenId.equals(that.tokenId);
    }

    public int hashCode() {
        return tokenId.hashCode();
    }
}
