/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Data access object managing persistence operations related to user roles
 */
public class UserRoleDAO extends BaseDAO {
    private static final Log log = LogFactory.getLog(UserRoleDAO.class);

    public UserRoleDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public ServiceUserRoleDO[] getUserRoles() {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserRoleDO[] roles = new ServiceUserRoleDO[0];
        try {
            Criteria roleCriteria = session.createCriteria(ServiceUserRoleDO.class);
            List list = roleCriteria.list();
            roles = (ServiceUserRoleDO[]) list.toArray(new ServiceUserRoleDO[list.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve user roles";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return roles;
    }

    public ServiceUserRoleDO getUserRole(String role) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserRoleDO serviceUserRoleDO = null;
        try {
            Criteria roleCriteria = session.createCriteria(ServiceUserRoleDO.class);
            roleCriteria.add(Expression.eq("role", role.trim()));
            serviceUserRoleDO = (ServiceUserRoleDO) roleCriteria.uniqueResult();
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve user role";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return serviceUserRoleDO;
    }

    public void deleteRole(String role) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserRoleDO serviceUserRoleDO;
        try {
            Criteria roleCriteria = session.createCriteria(ServiceUserRoleDO.class);
            roleCriteria.add(Expression.eq("role", role.trim()));
            serviceUserRoleDO = (ServiceUserRoleDO) roleCriteria.uniqueResult();
            if (serviceUserRoleDO == null) {
                tx.rollback();
                return;
            }
            Set users = serviceUserRoleDO.getUsers();
            Set services = serviceUserRoleDO.getServices();
            for (Iterator iterator = services.iterator(); iterator.hasNext();) {
                ServiceDO service = (ServiceDO) iterator.next();
                service.getRoles().remove(serviceUserRoleDO);
                session.update(service);

            }
            for (Iterator iterator = users.iterator(); iterator.hasNext();) {
                ServiceUserDO user = (ServiceUserDO) iterator.next();
                user.getRoles().remove(serviceUserRoleDO);
                session.update(user);

            }
            session.delete(serviceUserRoleDO);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to delete role";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }
}
