/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Data access object managing persistence operations related to users
 */
public class UserDAO extends BaseDAO {
    private static final Log log = LogFactory.getLog(UserDAO.class);

    public UserDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public ServiceUserDO[] getUsers() {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserDO[] users = new ServiceUserDO[0];
        try {
            Criteria userCriteria = session.createCriteria(ServiceUserDO.class);
            List list = userCriteria.list();
            users = (ServiceUserDO[]) list.toArray(new ServiceUserDO[list.size()]);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve all users";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return users;
    }

    public void deleteUser(String username) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            Criteria userCriteria = session.createCriteria(ServiceUserDO.class);
            userCriteria.add(Expression.eq("username", username));
            ServiceUserDO user = (ServiceUserDO) userCriteria.uniqueResult();
            if (user == null) {
                tx.rollback();
                return;
            }
            Set services = user.getServices();
            for (Iterator iterator = services.iterator(); iterator.hasNext();) {
                ServiceDO service = (ServiceDO) iterator.next();
                if (service.getUsers().size() == 1) {
                    // this is the only user assigned to this service
                    service.setIsUTAuthEnabled(false);
                }
                service.getUsers().remove(user);
                session.update(service);
            }

            session.delete(user);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            e.printStackTrace();
            tx.rollback();
            String msg = "Unable to deleteUser";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void deleteRoleFromUser(String username, String role) {

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            Criteria userCriteria = session.createCriteria(ServiceUserDO.class);
            userCriteria.add(Expression.eq("username", username));
            ServiceUserDO userDO = (ServiceUserDO) userCriteria.uniqueResult();
            if (userDO == null) {
                tx.rollback();
                return;
            }

            Criteria roleCriteria = session.createCriteria(ServiceUserRoleDO.class);
            roleCriteria.add(Expression.eq("role", role));
            ServiceUserRoleDO roleDO = (ServiceUserRoleDO) roleCriteria.uniqueResult();
            if (roleDO == null) {
                tx.rollback();
                return;
            }

            Set roles = userDO.getRoles();
            if (roles.contains(roleDO)) {
                userDO.getRoles().remove(roleDO);
                session.update(userDO);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to remove role from user";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public ServiceUserDO getUser(String username) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserDO user = null;
        try {
            Criteria userCriteria = session.createCriteria(ServiceUserDO.class);
            userCriteria.add(Expression.eq("username", username));
            user = (ServiceUserDO) userCriteria.uniqueResult();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve user";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return user;
    }

    public ServiceUserRoleDO[] getUserRoles(ServiceUserDO serviceUserDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserRoleDO[] serviceUserRoleDOs = null;
        try {

            Set roles = serviceUserDO.getRoles();
            serviceUserRoleDOs =
                    (ServiceUserRoleDO[]) roles.toArray(new ServiceUserRoleDO[roles.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get roles";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceUserDO);
            hbConfig.closeSession();
        }
        return serviceUserRoleDOs;
    }

    public void addRole(String userName, ServiceUserRoleDO role) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserDO userDO = null;
        try {
            userDO = getUser(userName, session);
            role = (ServiceUserRoleDO) session.load(ServiceUserRoleDO.class, role.getId());
            if (!userDO.getRoles().contains(role)) {
                userDO.addRole(role);
                session.update(userDO);
                session.update(role);
            }
            tx.commit();

        } catch (Exception e) {
            tx.rollback();
            String msg = "Unable to add role";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(userDO);
            hbConfig.closeSession();
        }
    }

    public ServiceUserDO getUser(String username, Session session) {
        Criteria userCriteria = session.createCriteria(ServiceUserDO.class);
        userCriteria.add(Expression.eq("username", username));
        return (ServiceUserDO) userCriteria.uniqueResult();
    }

    public ServiceUserRoleDO[] getRoles(String serviceUser) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceUserDO serviceUserDO = null;
        ServiceUserRoleDO[] serviceUserRoleDOs = null;
        try {
            serviceUserDO = getUser(serviceUser, session);
            if (serviceUserDO != null) {
                Set roles = serviceUserDO.getRoles();
                serviceUserRoleDOs =
                        (ServiceUserRoleDO[]) roles.toArray(new ServiceUserRoleDO[roles.size()]);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get roles";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceUserDO);
            hbConfig.closeSession();
        }
        return serviceUserRoleDOs;
    }

    public boolean userHasRole(String username, ServiceUserRoleDO role) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        boolean containsRole = false;
        try {
            containsRole = getUser(username, session).getRoles().contains(role);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get roles";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return containsRole;
    }
}
