/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupParameterDO;
import org.wso2.wsas.persistence.exception.ServiceGroupNotFoundException;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Date;
import java.util.Set;

/**
 * Data access object managing persistence operations related to axis service groups.
 */
public class ServiceGroupDAO extends BaseDAO {
    private static final Log log = LogFactory.getLog(ServiceGroupDAO.class);

    public ServiceGroupDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public Long createServiceGroup(ServiceGroupDO trasientInstance) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            trasientInstance.setLastUpdatedTime(new Date());
            session.persist(trasientInstance);
            session.flush();
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }

        return trasientInstance.getId();
    }

    public Long createServiceGroup(ServiceGroupDO trasientInstance, Session session) {
        trasientInstance.setLastUpdatedTime(new Date());
        session.persist(trasientInstance);
        session.flush();
        return trasientInstance.getId();
    }

    public ServiceGroupDO getServiceGroup(String serviceGroupId) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceGroupDO serviceGroupDO = null;
        try {
            Criteria criteria = session.createCriteria(ServiceGroupDO.class);
            criteria.add(Expression.eq("name", serviceGroupId.trim()));
            serviceGroupDO = (ServiceGroupDO) criteria.uniqueResult();
            session.flush();
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }
        return serviceGroupDO;
    }

    public ServiceGroupDO getServiceGroup(String serviceGroupId, Session session) {
        Criteria criteria = session.createCriteria(ServiceGroupDO.class);
        criteria.add(Expression.eq("name", serviceGroupId.trim()));
        return (ServiceGroupDO) criteria.uniqueResult();
    }

    public ServiceGroupParameterDO getParameter(ServiceGroupDO serviceGroup, String paramName) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceGroupParameterDO paramDO = null;
        try {
            Criteria criteria = session.createCriteria(ServiceGroupParameterDO.class, "param")
                    .createAlias("param.serviceGroup", "sg")
                    .add(Expression.eq("name", paramName.trim()))
                    .add(Expression.eq("sg.name", serviceGroup.getName()));
            paramDO = (ServiceGroupParameterDO) criteria.uniqueResult();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getParameter";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(paramDO);
            hbConfig.closeSession();
        }
        return paramDO;
    }

    public ServiceDO[] getServices(ServiceGroupDO serviceGroup) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO[] serviceDOs = null;
        try {
            serviceDOs = getServices(serviceGroup, session);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get service operations";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return serviceDOs;
    }

    public ServiceDO[] getServices(ServiceGroupDO serviceGroup, Session session) {
        ServiceGroupDO serviceGroupDO =
                (ServiceGroupDO) session.load(ServiceGroupDO.class, serviceGroup.getId());
        Set serviceSet = serviceGroupDO.getServices();
        return (ServiceDO[]) serviceSet.toArray(new ServiceDO[serviceSet.size()]);
    }

    public void deleteServiceGroup(String serviceGroupName) throws ServiceGroupNotFoundException {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceGroupDO serviceGroupDO = getServiceGroup(serviceGroupName, session);
        if (serviceGroupDO == null) {
            String msg = "Service group " + serviceGroupName + " not found";
            log.error(msg);
            throw new ServiceGroupNotFoundException(msg);
        }
        ServiceDO[] serviceDOs = getServices(serviceGroupDO, session);
        ServiceDAO serviceDAO = new ServiceDAO(hbConfig);
        try {
            for (int i = 0; i < serviceDOs.length; i++) {
                serviceDAO.deleteService(serviceDOs[i], session);
            }
            session.delete(serviceGroupDO);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to delete service group " + serviceGroupName;
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }
}