/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.axis2.AxisFault;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Parameter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.rampart.handler.WSSHandlerConstants;
import org.apache.ws.security.handler.WSHandlerConstants;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.hibernate.criterion.Restrictions;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ServiceParameterDO;
import org.wso2.wsas.persistence.dataobject.ServicePolicyDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.exception.ServicePolicyNotFoundException;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Date;
import java.util.Iterator;
import java.util.Set;

/**
 * Data access object managing persistence operations related to axis services
 */
public class ServiceDAO extends BaseDAO {
    private static final Log log = LogFactory.getLog(ServiceDAO.class);

    public ServiceDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public void updateService(ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            service.setLastUpdatedTime(new Date());
            session.merge(service);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to update service";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public ServiceDO getService(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        try {
            serviceDO = getService(serviceId, version, session);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return serviceDO;
    }

    public void addOperation(String serviceId, String version, OperationDO op) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            service.addOperation(op);
            session.update(service);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to add operation";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public OperationDO[] getOperations(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        OperationDO[] opDOs = null;
        try {
            serviceDO = getService(serviceId, version, session);
            Set opSet = serviceDO.getOperations();
            opDOs = (OperationDO[]) opSet.toArray(new OperationDO[opSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get service operations";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return opDOs;
    }

    public OperationDO[] getOperations(ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        OperationDO[] opDOs = null;
        try {
            serviceDO = (ServiceDO) session.load(ServiceDO.class, service.getId());
            Set opSet = serviceDO.getOperations();
            opDOs = (OperationDO[]) opSet.toArray(new OperationDO[opSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get service operations";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return opDOs;
    }

    public void addUser(String serviceId, String version, ServiceUserDO user) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            user = getUser(user.getUsername(), session);
            service.setIsUTAuthEnabled(true);
            if (!service.getUsers().contains(user)) {
                service.addUser(user);
//                session.update(service);
                session.update(user);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addUser";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public void addRole(String serviceId, String version, ServiceUserRoleDO role) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            role = getRole(role.getRole(), session);
            service.setIsUTAuthEnabled(true);
            if (!service.getRoles().contains(role)) {
                service.addRole(role);
                session.update(role);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addRole";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public ServiceUserRoleDO getRole(String roleName, Session session) {
        Criteria criteria = session.createCriteria(ServiceUserRoleDO.class);
        criteria.add(Expression.eq("role", roleName.trim()));
        return (ServiceUserRoleDO) criteria.uniqueResult();
    }

    private ServiceUserDO getUser(String username, Session session) {
        Criteria criteria = session.createCriteria(ServiceUserDO.class);
        criteria.add(Expression.eq("username", username.trim()));
        return (ServiceUserDO) criteria.uniqueResult();
    }

    public void addPolicy(String serviceId, String version, ServicePolicyDO policy) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            if (!service.getPolicies().contains(policy)) {
                service.addPolicy(policy);
                session.update(service);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addPolicy";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public void updatePolicy(ServicePolicyDO servicePolicy) throws ServicePolicyNotFoundException {

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            Criteria criteria = session.createCriteria(ServicePolicyDO.class, "policy")
                    .createAlias("policy.service", "s")
                    .add(Restrictions.eq("policy.uuid", servicePolicy.getUuid()))
                    .add(Restrictions.eq("s.serviceIdentifierDO.serviceId",
                                         servicePolicy.getService().
                                                 getServiceIdentifierDO().getServiceId().trim()))
                    .add(Restrictions.eq("s.serviceIdentifierDO.version",
                                         servicePolicy.getService().
                                                 getServiceIdentifierDO().getVersion().trim()));
            ServicePolicyDO origServicePolicy =
                    (ServicePolicyDO) criteria.uniqueResult();

            if (origServicePolicy == null) {
                session.save(servicePolicy);
            } else {
                origServicePolicy.setPolicy(servicePolicy.getPolicy());
                session.update(origServicePolicy);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to update service policy";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void addParameter(String serviceId, String version, ServiceParameterDO param) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            if (!service.getParameters().contains(param)) {
                service.addParameter(param);
                session.update(service);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to add parameter";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public ServiceUserDO[] getUsers(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServiceUserDO[] userDOs = null;
        try {
            serviceDO = getService(serviceId, version, session);
            Set userSet = serviceDO.getUsers();
            userDOs = (ServiceUserDO[]) userSet.toArray(new ServiceUserDO[userSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get service users";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return userDOs;
    }

    public ServiceUserRoleDO[] getRoles(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServiceUserRoleDO[] userRoleDOs = null;
        try {
            serviceDO = getService(serviceId, version, session);
            Set roleSet = serviceDO.getRoles();
            userRoleDOs =
                    (ServiceUserRoleDO[]) roleSet.toArray(new ServiceUserRoleDO[roleSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get roles";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return userRoleDOs;

    }

    public ServicePolicyDO[] getPolicies(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServicePolicyDO[] policyDOs = null;
        try {
            serviceDO = getService(serviceId, version, session);
            Set opSet = serviceDO.getPolicies();
            policyDOs = (ServicePolicyDO[]) opSet.toArray(new ServicePolicyDO[opSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get policies";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return policyDOs;
    }

    public ServicePolicyDO[] getPolicies(ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServicePolicyDO[] policyDOs = null;
        try {
            serviceDO = (ServiceDO) session.load(ServiceDO.class, service.getId());
            Set opSet = serviceDO.getPolicies();
            policyDOs = (ServicePolicyDO[]) opSet.toArray(new ServicePolicyDO[opSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get policies";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return policyDOs;
    }

    public ServiceParameterDO[] getParameters(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServiceParameterDO[] paramDOs = null;
        try {
            serviceDO = getService(serviceId, version, session);
            Set opSet = serviceDO.getParameters();
            paramDOs = (ServiceParameterDO[]) opSet.toArray(new ServiceParameterDO[opSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get parameters";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return paramDOs;
    }

    public ServiceParameterDO[] getParameters(ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServiceParameterDO[] paramDOs = null;
        try {
            serviceDO = (ServiceDO) session.load(ServiceDO.class, service.getId());
            Set paramSet = serviceDO.getParameters();
            paramDOs = (ServiceParameterDO[]) paramSet.
                    toArray(new ServiceParameterDO[paramSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get parameters";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return paramDOs;
    }

    public ServiceParameterDO getParameter(ServiceDO service, String paramName) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ServiceParameterDO paramDO = null;

        try {
            ServiceIdentifierDO sid = service.getServiceIdentifierDO();
            Criteria criteria =
                    session.createCriteria(ServiceParameterDO.class, "param")
                            .createAlias("param.service", "s")
                            .add(Expression.eq("name", paramName.trim()))
                            .add(Expression.eq("s.serviceIdentifierDO.serviceId",
                                               sid.getServiceId().trim()))
                            .add(Expression.eq("s.serviceIdentifierDO.version",
                                               sid.getVersion().trim()));
            paramDO = (ServiceParameterDO) criteria.uniqueResult();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to get parameter";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return paramDO;
    }

    public void deleteService(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            deleteService(serviceId, version, session);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to deleteService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void deleteService(String serviceId, String version, Session session) {
        ServiceDO serviceDO = getService(serviceId, version, session);

        KeyStoreDO privateKeyStore = serviceDO.getPrivateKeyStore();
        if (privateKeyStore != null) {
            privateKeyStore.getPkStoreServices().remove(serviceDO);
            serviceDO.setPrivateKeyStore(null);
            session.update(privateKeyStore);
        }

        for (Iterator iter = serviceDO.getTrustedCertStores().iterator(); iter.hasNext();) {
            KeyStoreDO keyStoreDO = (KeyStoreDO) iter.next();
            keyStoreDO.getTrustStoreServices().remove(serviceDO);
        }
        serviceDO.removeAllRelationships();
        session.update(serviceDO);
        session.delete(serviceDO);
    }

    public void deleteService(ServiceDO serviceDO, Session session) {
        KeyStoreDO privateKeyStore = serviceDO.getPrivateKeyStore();
        if (privateKeyStore != null) {
            privateKeyStore.getPkStoreServices().remove(serviceDO);
            serviceDO.setPrivateKeyStore(null);
            session.update(privateKeyStore);
        }

        for (Iterator iter = serviceDO.getTrustedCertStores().iterator(); iter.hasNext();) {
            KeyStoreDO keyStoreDO = (KeyStoreDO) iter.next();
            keyStoreDO.getTrustStoreServices().remove(serviceDO);
        }
        serviceDO.removeAllRelationships();
        session.update(serviceDO);
        session.delete(serviceDO);
    }

    public ModuleDO[] getEngagedModules(String serviceId, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO;
        ModuleDO[] modules = new ModuleDO[0];
        try {
            serviceDO = getService(serviceId, version, session);
            Set engagedModules = serviceDO.getEngagedModules();
            modules = (ModuleDO[]) engagedModules.toArray(new ModuleDO[engagedModules.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to deleteService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return modules;
    }

    public ModuleDO[] getEngagedModules(ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO;
        ModuleDO[] modules = new ModuleDO[0];
        try {
            serviceDO = (ServiceDO) session.load(ServiceDO.class, service.getId());
            Set engagedModules = serviceDO.getEngagedModules();
            modules = (ModuleDO[]) engagedModules.toArray(new ModuleDO[engagedModules.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to deleteService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return modules;
    }

    private ServiceDO getService(String serviceId, String version, Session session) {
        Criteria criteria = session.createCriteria(ServiceDO.class);
        criteria.add(Expression.eq("serviceIdentifierDO.serviceId", serviceId.trim()));
        criteria.add(Expression.eq("serviceIdentifierDO.version", version.trim()));
        return (ServiceDO) criteria.uniqueResult();
    }

    public void removeServiceUser(String serviceId, String version,
                                  ServiceUserDO user,
                                  AxisService axisService) {

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            user = getUser(user.getUsername(), session);

            if (service.getUsers().contains(user)) {
                user.getServices().remove(service);
                service.getUsers().remove(user);
                session.update(user);
            }
            checkUTAuthenticationDisablingNeeded(service, axisService);

            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to remove user from service";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    public void removeServiceRole(String serviceId,
                                  String version,
                                  ServiceUserRoleDO role,
                                  AxisService axisService) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = getService(serviceId, version, session);
            role = getRole(role.getRole(), session);
            if (service.getRoles().contains(role)) {
                role.getServices().remove(service);
                service.getRoles().remove(role);
                session.update(role);
            }

            checkUTAuthenticationDisablingNeeded(service, axisService);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to remove role from service";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }

    private void checkUTAuthenticationDisablingNeeded(ServiceDO service,
                                                      AxisService axisService) throws AxisFault {

        if (service.getUsers().isEmpty() && service.getRoles().isEmpty()) {
            service.setIsUTAuthEnabled(false);
            axisService.removeParameter(new Parameter(WSSHandlerConstants.INFLOW_SECURITY, null));
            axisService.removeParameter(new Parameter(WSHandlerConstants.PW_CALLBACK_REF, null));
        }
    }

    public void removeAllTrustedCertStores(ServiceDO serviceDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service = null;
        try {
            service = (ServiceDO) session.load(ServiceDO.class, serviceDO.getId());
            for (Iterator iter = service.getTrustedCertStores().iterator(); iter.hasNext();) {
                KeyStoreDO keyStoreDO = (KeyStoreDO) iter.next();
                keyStoreDO.getTrustStoreServices().remove(service);
                session.update(keyStoreDO);
            }
            service.removeAllTrustedCertStores();
            session.update(service);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to remove all TrustedCertStores from service";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(service);
            hbConfig.closeSession();
        }
    }
}
