/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.OperationParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Date;
import java.util.Iterator;
import java.util.Set;

/**
 * Data access object managing persistence operations related to axis operations
 */
public class OperationDAO extends BaseDAO {
    private static final Log log = LogFactory.getLog(OperationDAO.class);

    public OperationDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public ServiceIdentifierDO getServiceIdentifier(OperationDO op) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO service;
        ServiceIdentifierDO serviceId = null;
        try {
            op = (OperationDO) session.load(OperationDO.class, op.getId());
            service = op.getService();
            serviceId = service.getServiceIdentifierDO();
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }

        return serviceId;
    }

    public void updateOperation(OperationDO operation) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            operation.setLastUpdatedTime(new Date());
            session.merge(operation);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to update service";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public ModuleDO[] getEngagedModules(OperationDO op) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO[] modules = new ModuleDO[0];
        try {
            op = (OperationDO) session.load(OperationDO.class, op.getId());
            Set engagedModules = op.getEngagedModules();
            modules = (ModuleDO[]) engagedModules.toArray(new ModuleDO[engagedModules.size()]);
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }

        return modules;
    }

    public OperationParameterDO[] getParameters(OperationDO op) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        OperationParameterDO[] params = new OperationParameterDO[0];
        try {
            op = (OperationDO) session.load(OperationDO.class, op.getId());
            Set paramSet = op.getParameters();
            params =
                    (OperationParameterDO[]) paramSet.toArray(new OperationParameterDO[paramSet.size()]);
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }

        return params;
    }

    public OperationDO getOperation(String serviceId, String serviceVersion, String opName) {
        String query = "from OperationDO as op where op.service.serviceIdentifierDO.serviceId='" +
                       serviceId.trim() + "' and op.service.serviceIdentifierDO.version='" +
                       serviceVersion.trim() + "' and op.name='" + opName.trim() + "'";
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        OperationDO op = null;
        try {
            op = (OperationDO) session.createQuery(query).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }

        return op;
    }

    public OperationParameterDO getOperationParameter(String serviceId,
                                                      String serviceVersion,
                                                      String operationId,
                                                      String paramName) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        OperationDO opDO = getOperation(serviceId, serviceVersion, operationId, session);
        OperationParameterDO opParamDO = null;
        try {
            for (Iterator iter = opDO.getParameters().iterator(); iter.hasNext();) {
                opParamDO = (OperationParameterDO) iter.next();
                if (opParamDO.getName().equals(paramName)) {
                    break;
                }
            }
        } catch (Exception e) {
            tx.rollback();
        } finally {
            hbConfig.closeSession();
        }
        return opParamDO;
    }

    public OperationDO getOperation(String serviceId,
                                    String serviceVersion,
                                    String opName,
                                    Session session) {
        String query = "from OperationDO as op where op.service.serviceIdentifierDO.serviceId='" +
                       serviceId.trim() + "' and op.service.serviceIdentifierDO.version='" +
                       serviceVersion.trim() + "' and op.name='" + opName.trim() + "'";
        return (OperationDO) session.createQuery(query).uniqueResult();
    }
}
