/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.ModuleParameterDO;
import org.wso2.wsas.persistence.dataobject.ModulePolicyDO;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupDO;
import org.wso2.wsas.persistence.exception.ModulePolicyNotFoundException;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Data access object managing persistence operations related to Axis2 modules
 */
public class ModuleDAO extends BaseDAO {

    private static final Log log = LogFactory.getLog(ModuleDAO.class);

    public ModuleDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public ModuleDO getModule(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getModule";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return moduleDO;
    }

    public void deleteModule(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set engagedOperations = moduleDO.getEngagedOperations();
            for (Iterator iterator = engagedOperations.iterator(); iterator.hasNext();) {
                OperationDO op = (OperationDO) iterator.next();
                op.getEngagedModules().remove(moduleDO);
                session.saveOrUpdate(op);
            }
            engagedOperations.clear();

            Set engagedServiceGroups = moduleDO.getEngagedServiceGroups();
            for (Iterator iterator = engagedServiceGroups.iterator(); iterator.hasNext();) {
                ServiceGroupDO sg = (ServiceGroupDO) iterator.next();
                sg.getEngagedModules().remove(moduleDO);
                session.saveOrUpdate(sg);
            }
            engagedServiceGroups.clear();

            Set engagedServices = moduleDO.getEngagedServices();
            for (Iterator iterator = engagedServices.iterator(); iterator.hasNext();) {
                ServiceDO service = (ServiceDO) iterator.next();
                service.getEngagedModules().remove(moduleDO);
                session.saveOrUpdate(service);
            }
            engagedServices.clear();

            session.delete(moduleDO);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to deleteModule";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public void addEngagedService(String moduleName, String version, ServiceDO service) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            service = (ServiceDO) session.load(ServiceDO.class, service.getId());
            if (!moduleDO.getEngagedServices().contains(service)) {
                moduleDO.addService(service);
                session.update(moduleDO);
                session.update(service);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addEngagedService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public void removeEngagedService(String moduleName, String version, ServiceDO serviceDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            if (moduleDO != null) {
                serviceDO = (ServiceDO) session.load(ServiceDO.class, serviceDO.getId());
                if (moduleDO.getEngagedServices().contains(serviceDO)) {
                    serviceDO.getEngagedModules().remove(moduleDO);
                    session.update(moduleDO);
                    session.update(serviceDO);
                }
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to removeEngagedService";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public void addEngagedServiceGroup(String moduleName,
                                       String version,
                                       ServiceGroupDO serviceGroup) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            serviceGroup =
                    (ServiceGroupDO) session.load(ServiceGroupDO.class, serviceGroup.getId());
            serviceGroup.addModule(moduleDO);
            session.update(moduleDO);
            session.update(serviceGroup);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addEngagedServiceGroup";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public void removeEngagedServiceGroup(String moduleName,
                                          String version,
                                          ServiceGroupDO serviceGrp) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            if (moduleDO != null) {
                serviceGrp =
                        (ServiceGroupDO) session.load(ServiceGroupDO.class, serviceGrp.getId());
                if (moduleDO.getEngagedServiceGroups().contains(serviceGrp)) {
                    serviceGrp.getEngagedModules().remove(moduleDO);
                    session.update(moduleDO);
                    session.update(serviceGrp);
                }
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to removeEngagedService group";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public ServiceDO[] getEngagedServices(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        ServiceDO[] serviceDOs = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set engServiceSet = moduleDO.getEngagedServices();
            serviceDOs = (ServiceDO[]) engServiceSet.toArray(new ServiceDO[engServiceSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getEngagedServices";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
        return serviceDOs;
    }

    public ServiceGroupDO[] getEngagedServiceGroups(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        ServiceGroupDO[] serviceGroupDOs = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set engServiceGroupSet = moduleDO.getEngagedServiceGroups();
            serviceGroupDOs =
                    (ServiceGroupDO[]) engServiceGroupSet.
                            toArray(new ServiceGroupDO[engServiceGroupSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getEngagedServices";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
        return serviceGroupDOs;
    }

    public void addEngagedOperation(String moduleName, String version, OperationDO operation) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        try {
            moduleDO = getModule(moduleName, version, session);
            operation = (OperationDO) session.load(OperationDO.class, operation.getId());
            if (!moduleDO.getEngagedOperations().contains(operation)) {
                moduleDO.addOperation(operation);
                session.update(moduleDO);
                session.update(operation);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addEngagedOperation";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void removeEngagedOperation(String moduleName, String version, OperationDO operation) {
        if (operation == null) { // sometimes there are dynamically added operations, which may not be on the DB
            return;
        }
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            if (moduleDO != null) {
                operation = (OperationDO) session.load(OperationDO.class, operation.getId());
                if (moduleDO.getEngagedOperations().contains(operation)) {
                    operation.getEngagedModules().remove(moduleDO);
                    //                moduleDO.getEngagedOperations().remove(operation);
                    session.update(moduleDO);
                    session.update(operation);
                }
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to removeEngagedOperation";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(moduleDO);
            hbConfig.closeSession();
        }
    }

    public void addPolicy(String moduleName, String version, ModulePolicyDO policy) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        try {
            moduleDO = getModule(moduleName, version, session);
            policy.setModule(moduleDO);
            session.save(policy);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addPolicy";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public OperationDO[] getEngagedOperations(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        OperationDO[] opDOs = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set engOpSet = moduleDO.getEngagedOperations();
            opDOs = (OperationDO[]) engOpSet.toArray(new OperationDO[engOpSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getEngagedOperations";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return opDOs;
    }

    public ModulePolicyDO[] getPolicies(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        ModulePolicyDO[] policies = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set policySet = moduleDO.getPolicies();
            policies = (ModulePolicyDO[]) policySet.toArray(new ModulePolicyDO[policySet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getPolicies";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return policies;
    }

    public ModulePolicyDO[] getPolicies(ModuleDO module) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        ModulePolicyDO[] policies = null;
        try {
            moduleDO = (ModuleDO) session.load(ModuleDO.class, module.getId());
            Set policySet = moduleDO.getPolicies();
            policies = (ModulePolicyDO[]) policySet.toArray(new ModulePolicyDO[policySet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getPolicies";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return policies;
    }

    public void updatePolicy(ModulePolicyDO modulePolicy) throws ModulePolicyNotFoundException {

        String query =
                "from ModulePolicyDO as policy where uuid='" + modulePolicy.getUuid() + "' and " +
                "policy.module.moduleIdentifierDO.name='" +
                modulePolicy.getModule().getModuleIdentifierDO().getName() + "' and " +
                "policy.module.moduleIdentifierDO.version='" +
                modulePolicy.getModule().getModuleIdentifierDO().getVersion() + "'";
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            ModulePolicyDO origModulePolicy =
                    (ModulePolicyDO) session.createQuery(query).uniqueResult();

            if (origModulePolicy == null) {
                throw new ModulePolicyNotFoundException("Module Policy with UUID " +
                                                        modulePolicy.getUuid() +
                                                        " not found");
            }

            origModulePolicy.setPolicy(modulePolicy.getPolicy());
            session.update(origModulePolicy);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to updatePolicy";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void addParameter(String moduleName, String version, ModuleParameterDO param) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        try {
            moduleDO = getModule(moduleName, version, session);
            param.setModule(moduleDO);
            session.save(param);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to addParameter";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public ModuleParameterDO[] getParameters(String moduleName, String version) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        ModuleParameterDO[] params = null;
        try {
            moduleDO = getModule(moduleName, version, session);
            Set paramSet = moduleDO.getParameters();
            params = (ModuleParameterDO[]) paramSet.toArray(new ModuleParameterDO[paramSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getParameters";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return params;
    }

    public ModuleParameterDO[] getParameters(ModuleDO module) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO moduleDO;
        ModuleParameterDO[] params = null;
        try {
            moduleDO = (ModuleDO) session.load(ModuleDO.class, module.getId());
            Set paramSet = moduleDO.getParameters();
            params = (ModuleParameterDO[]) paramSet.toArray(new ModuleParameterDO[paramSet.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getParameters";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return params;
    }

    public ModuleParameterDO getParameter(ModuleDO module, String paramName) {
        String query = "from ModuleParameterDO as param where param.name='" + paramName +
                       "' and param.module.moduleIdentifierDO.name='" +
                       module.getModuleIdentifierDO().getName() +
                       "' and param.module.moduleIdentifierDO.version='" +
                       module.getModuleIdentifierDO().getVersion() + "'";

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceDO serviceDO = null;
        ModuleParameterDO paramDO = null;
        try {
            paramDO = (ModuleParameterDO) session.createQuery(query).uniqueResult();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getParameter";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            session.evict(serviceDO);
            hbConfig.closeSession();
        }
        return paramDO;
    }

    public ModuleDO getModule(String moduleName, String version, Session session) {
        String query = version != null && version.trim().length() != 0 ?
                       "from ModuleDO as module where module.moduleIdentifierDO.name='" +
                       moduleName.trim() + "' and module.moduleIdentifierDO.version='" +
                       version.trim() + "'"
                       :
                       "from ModuleDO as module where module.moduleIdentifierDO.name='" +
                       moduleName.trim() + "' order by module.moduleIdentifierDO.version";
//        String query = "from ModuleDO as module where module.moduleIdentifierDO.name='" +
//                       moduleName.trim() + "' and module.moduleIdentifierDO.version='" +
//                       version.trim() + "'";
        List list = session.createQuery(query).list();
        if (!list.isEmpty()) {
            return (ModuleDO) list.get(0);
        }
        return null;
    }

    public ModuleDO[] getAllModules() {
        String query = "from ModuleDO";
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ModuleDO[] modules = new ModuleDO[0];
        try {
            List list = session.createQuery(query).list();
            modules = (ModuleDO[]) list.toArray(new ModuleDO[list.size()]);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to getAllModules";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return modules;
    }
}
