/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.wso2.wsas.persistence.dataobject.EJBProviderDO;
import org.wso2.wsas.util.HibernateConfig;

import java.util.List;

/**
 * Data access object managing persistence operations related to EJB configuration
 */
public class EJBConfigurationDAO extends BaseDAO {

    private static final Log log = LogFactory.getLog(EJBConfigurationDAO.class);

    public EJBConfigurationDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public void addEJBConfiguration(EJBProviderDO ejbProviderDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            session.save(ejbProviderDO);
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to add EJB Configuration";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public EJBProviderDO[] getEJBConfigurations() {

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        EJBProviderDO[] ejbProviders = new EJBProviderDO[0];
        try {
            Criteria ejbProviderCriteria = session.createCriteria(EJBProviderDO.class);
            List list = ejbProviderCriteria.list();
            ejbProviders = (EJBProviderDO[]) list.toArray(new EJBProviderDO[list.size()]);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve all ejb provider configurations";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return ejbProviders;
    }

    public void deleteEJBConfiguration(String beanJNDIName, String jnpProviderUrl) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            Criteria ejbCriteria = session.createCriteria(EJBProviderDO.class);
            ejbCriteria.add(Expression.eq("beanJNDIName", beanJNDIName));
            ejbCriteria.add(Expression.eq("providerURL", jnpProviderUrl));

            EJBProviderDO ejbProviderDO = (EJBProviderDO) ejbCriteria.uniqueResult();
            if (ejbProviderDO == null) {
                tx.rollback();
                return;
            }
            session.delete(ejbProviderDO);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            e.printStackTrace();
            tx.rollback();
            String msg = "Unable to delete EJB Configuration";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }


    public EJBProviderDO getEJBConfiguration(String beanJNDIName, String jnpProviderUrl) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        EJBProviderDO ejbProviderDO = null;
        try {
            Criteria ejbCriteria = session.createCriteria(EJBProviderDO.class);
            ejbCriteria.add(Expression.eq("beanJNDIName", beanJNDIName));
            ejbCriteria.add(Expression.eq("providerURL", jnpProviderUrl));

            ejbProviderDO = (EJBProviderDO) ejbCriteria.uniqueResult();
            if (ejbProviderDO == null) {
                tx.rollback();
                return null;
            }

            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg;
            if (e instanceof org.hibernate.NonUniqueResultException) {
                msg = "More than one EJB Configuration found";
                log.error(msg, e);
                throw new RuntimeException(msg, e);
            } else {
                msg = "Unable to retrieve EJB Configuration";
                log.error(msg, e);
                throw new RuntimeException(msg, e);
            }
        } finally {
            hbConfig.closeSession();
        }
        return ejbProviderDO;
    }


}
