/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Expression;
import org.wso2.wsas.persistence.dataobject.EJBAppServerDO;
import org.wso2.wsas.util.HibernateConfig;

import java.util.List;

/**
 * Data access object managing persistence operations related to EJB configuration
 */
public class EJBAppServerDAO extends BaseDAO {

    private static final Log log = LogFactory.getLog(EJBAppServerDAO.class);

    public EJBAppServerDAO(HibernateConfig hbConfig) {
        super(hbConfig);
    }

    public void saveEJBAppServer(EJBAppServerDO ejbAppServerDO, boolean isUpdate) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            if (isUpdate) {
                session.update(ejbAppServerDO);
            } else {
                session.save(ejbAppServerDO);
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to save EJB Application Server";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public EJBAppServerDO[] getEJBAppServers() {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        EJBAppServerDO[] ejbAppServers = new EJBAppServerDO[0];
        try {
            Criteria ejbProviderCriteria = session.createCriteria(EJBAppServerDO.class);
            List list = ejbProviderCriteria.list();
            ejbAppServers = (EJBAppServerDO[]) list.toArray(new EJBAppServerDO[list.size()]);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve all ejb application servers";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return ejbAppServers;
    }

    public void deleteEJBAppServer(String jnpProviderUrl) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            Criteria ejbCriteria = session.createCriteria(EJBAppServerDO.class);
            ejbCriteria.add(Expression.eq("providerURL", jnpProviderUrl));

            EJBAppServerDO ejbAppServerDO = (EJBAppServerDO) ejbCriteria.uniqueResult();
            if (ejbAppServerDO == null) {
                tx.rollback();
                return;
            }
            session.delete(ejbAppServerDO);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            e.printStackTrace();
            tx.rollback();
            String msg = "Unable to delete EJB Application Server";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }


    public EJBAppServerDO getEJBAppServer(String jnpProviderUrl) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        EJBAppServerDO ejbAppServerDO = null;
        try {
            Criteria ejbCriteria = session.createCriteria(EJBAppServerDO.class);
            ejbCriteria.add(Expression.eq("providerURL", jnpProviderUrl));

            ejbAppServerDO = (EJBAppServerDO) ejbCriteria.uniqueResult();
            if (ejbAppServerDO == null) {
                tx.rollback();
                return null;
            }
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to retrieve EJB Configuration";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
        return ejbAppServerDO;
    }


}
