/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence.dao;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.exception.ConstraintViolationException;
import org.wso2.wsas.persistence.dataobject.AbstractDataObject;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.util.HibernateConfig;

import java.util.Date;

/**
 * Base class for all WSO2 WSAS data access objects.
 *
 * @see MessageDAO
 * @see ModuleDAO
 * @see OperationDAO
 * @see ServiceDAO
 * @see ServiceGroupDAO
 * @see ServerConfigurationDAO
 * @see UserDAO
 * @see UserRoleDAO
 */
public class BaseDAO {
    private static final Log log = LogFactory.getLog(BaseDAO.class);

    protected HibernateConfig hbConfig;

    public BaseDAO(HibernateConfig hbConfig) {
        this.hbConfig = hbConfig;
    }

    public Long create(AbstractDataObject trasientInstance) throws DuplicateEntityException {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            trasientInstance.setLastUpdatedTime(new Date());
            session.persist(trasientInstance);
            session.flush();
            tx.commit();
        } catch (ConstraintViolationException e) {
            String msg = "Trying to create duplicate entity";
            log.warn(msg, e);
            throw new DuplicateEntityException(msg, e);
        } catch (Exception e) {
            e.printStackTrace();
            tx.rollback();
            throw new RuntimeException("Cannot create entity", e);
        } finally {
            hbConfig.closeSession();
        }

        return trasientInstance.getId();
    }

    public Long createOrUpdate(AbstractDataObject instance) throws DuplicateEntityException {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            instance.setLastUpdatedTime(new Date());
            session.saveOrUpdate(instance);
            session.flush();
            tx.commit();
        } catch (ConstraintViolationException e) {
            String msg = "Trying to create duplicate entity";
            log.warn(msg, e);
            throw new DuplicateEntityException(msg, e);
        } catch (Exception e) {
            e.printStackTrace();
            tx.rollback();
            throw new RuntimeException("Cannot create entity", e);
        } finally {
            hbConfig.closeSession();
        }

        return instance.getId();
    }

    public void update(AbstractDataObject abstractDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            abstractDO.setLastUpdatedTime(new Date());
            session.merge(abstractDO);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to update";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public void delete(AbstractDataObject abstractDO) {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            session.delete(abstractDO);
            session.flush();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to delete";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }
}
