/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.persistence;

import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.util.Utils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyEngine;
import org.apache.neethi.PolicyReference;
import org.apache.neethi.PolicyRegistry;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.wso2.utils.ServerConfigurator;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.persistence.dao.AppenderDAO;
import org.wso2.wsas.persistence.dao.BaseDAO;
import org.wso2.wsas.persistence.dao.EJBAppServerDAO;
import org.wso2.wsas.persistence.dao.EJBConfigurationDAO;
import org.wso2.wsas.persistence.dao.KeyStoreDAO;
import org.wso2.wsas.persistence.dao.LoggerDAO;
import org.wso2.wsas.persistence.dao.MessageDAO;
import org.wso2.wsas.persistence.dao.ModuleDAO;
import org.wso2.wsas.persistence.dao.OperationDAO;
import org.wso2.wsas.persistence.dao.SecurityScenarioDAO;
import org.wso2.wsas.persistence.dao.SecurityTokenDAO;
import org.wso2.wsas.persistence.dao.ServerConfigurationDAO;
import org.wso2.wsas.persistence.dao.ServiceDAO;
import org.wso2.wsas.persistence.dao.ServiceGroupDAO;
import org.wso2.wsas.persistence.dao.TransportDAO;
import org.wso2.wsas.persistence.dao.UserDAO;
import org.wso2.wsas.persistence.dao.UserRoleDAO;
import org.wso2.wsas.persistence.dataobject.AbstractDataObject;
import org.wso2.wsas.persistence.dataobject.AppenderDO;
import org.wso2.wsas.persistence.dataobject.EJBAppServerDO;
import org.wso2.wsas.persistence.dataobject.EJBProviderDO;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.LoggerDO;
import org.wso2.wsas.persistence.dataobject.MessageDO;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.ModuleIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ModuleParameterDO;
import org.wso2.wsas.persistence.dataobject.ModulePolicyDO;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.OperationParameterDO;
import org.wso2.wsas.persistence.dataobject.SecurityScenarioDO;
import org.wso2.wsas.persistence.dataobject.SecurityTokenDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ServiceParameterDO;
import org.wso2.wsas.persistence.dataobject.ServicePolicyDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.dataobject.TransportDO;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.persistence.exception.EJBAppServerAlreadyExistsException;
import org.wso2.wsas.persistence.exception.EJBAppServerNotFoundException;
import org.wso2.wsas.persistence.exception.EJBConfigurationAlreadyExistsException;
import org.wso2.wsas.persistence.exception.EJBConfigurationNotFoundException;
import org.wso2.wsas.persistence.exception.KeyStoreAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ModuleAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ModuleNotFoundException;
import org.wso2.wsas.persistence.exception.ModulePolicyNotFoundException;
import org.wso2.wsas.persistence.exception.SecurityScenarioAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServiceAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServiceGroupNotFoundException;
import org.wso2.wsas.persistence.exception.ServiceNotFoundException;
import org.wso2.wsas.persistence.exception.ServicePolicyAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServicePolicyNotFoundException;
import org.wso2.wsas.persistence.exception.ServiceUserAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServiceUserNotFoundException;
import org.wso2.wsas.persistence.exception.TokenAlreadyExistsException;
import org.wso2.wsas.persistence.exception.TransportAlreadyExistsException;
import org.wso2.wsas.persistence.exception.UserRoleAlreadyExistsException;
import org.wso2.wsas.security.WSS4JUtil;
import org.wso2.wsas.util.HibernateConfig;
import org.wso2.wsas.util.HibernateConfigFactory;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.WsasUtils;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * Handles all the persistence related stuff for WSO2 WSAS
 */
public class PersistenceManager {
    private HibernateConfig hbConfig;
    private static Log log = LogFactory.getLog(PersistenceManager.class);

    public PersistenceManager() {
        this.hbConfig =
                HibernateConfigFactory.
                        getDefaultConfig(ServerConstants.WSO2WSAS_HB_CONFIG_KEY);
    }

    public Long addModule(ModuleDO module) throws ModuleAlreadyExistsException {
        try {
            return new ModuleDAO(hbConfig).create(module);
        } catch (DuplicateEntityException e) {
            throw new ModuleAlreadyExistsException(e);
        }
    }

    public ModuleDO getModule(String moduleName, String version) {
        return new ModuleDAO(hbConfig).getModule(moduleName, version);
    }

    public ModuleDO[] getAllModules() {
        return new ModuleDAO(hbConfig).getAllModules();
    }

    public void updateModule(ModuleDO module) throws ModuleNotFoundException {
        new ModuleDAO(hbConfig).update(module);
    }

    public void updateSecurityToken(SecurityTokenDO tokenDO) {
        new SecurityTokenDAO(hbConfig).update(tokenDO);
    }

    public SecurityTokenDO getSecurityToken(String tokenId) {
        return new SecurityTokenDAO(hbConfig).getToken(tokenId);
    }

    public Long addUserRole(ServiceUserRoleDO serviceUserRole)
            throws UserRoleAlreadyExistsException {
        try {
            Long id = new UserRoleDAO(hbConfig).create(serviceUserRole);
            serviceUserRole.notifyObservers();
            return id;
        } catch (DuplicateEntityException e) {
            throw new UserRoleAlreadyExistsException(e);
        }
    }

    public Long addOrUpdateUserRole(ServiceUserRoleDO serviceUserRole)
            throws UserRoleAlreadyExistsException {
        try {
            Long id = new UserRoleDAO(hbConfig).createOrUpdate(serviceUserRole);
            serviceUserRole.notifyObservers();
            return id;
        } catch (DuplicateEntityException e) {
            throw new UserRoleAlreadyExistsException(e);
        }
    }

    public Long addSecurityScenario(SecurityScenarioDO scenario)
            throws SecurityScenarioAlreadyExistsException {
        try {
            return new SecurityScenarioDAO(hbConfig).create(scenario);
        } catch (DuplicateEntityException e) {
            throw new SecurityScenarioAlreadyExistsException(e);
        }
    }

    public SecurityScenarioDO[] getAllSecurityScenarios() {
        return new SecurityScenarioDAO(hbConfig).getAllScenarios();
    }

    public Long addUser(ServiceUserDO serviceUser) throws ServiceUserAlreadyExistsException {
        try {
            Long id = new UserDAO(hbConfig).create(serviceUser);
            serviceUser.notifyObservers();
            return id;
        } catch (DuplicateEntityException e) {
            throw new ServiceUserAlreadyExistsException(e);
        }
    }

    public synchronized void assignUserToService(String username,
                                                 String serviceId,
                                                 String serviceVersion)
            throws ServiceUserNotFoundException, ServiceNotFoundException {

        ServiceUserDO user = new UserDAO(hbConfig).getUser(username);
        new ServiceDAO(hbConfig).addUser(serviceId, serviceVersion, user);
    }


    public synchronized void assignRoleToService(String roleName,
                                                 String serviceId,
                                                 String serviceVersion)
            throws ServiceUserNotFoundException, ServiceNotFoundException {
        ServiceUserRoleDO role = new UserRoleDAO(hbConfig).getUserRole(roleName);
        new ServiceDAO(hbConfig).addRole(serviceId, serviceVersion, role);

    }

    public ServiceUserDO getUser(String username) {
        return new UserDAO(hbConfig).getUser(username);
    }


    public ServiceUserRoleDO getRole(String role) {
        return new UserRoleDAO(hbConfig).getUserRole(role);
    }

    public void removeRoleFromService(String roleName,
                                      String serviceId,
                                      String serviceVersion,
                                      AxisService axisService)
            throws ServiceUserNotFoundException, ServiceNotFoundException {
        ServiceUserRoleDO role = new UserRoleDAO(hbConfig).getUserRole(roleName);
        new ServiceDAO(hbConfig).removeServiceRole(serviceId,
                                                   serviceVersion,
                                                   role,
                                                   axisService);
    }

    public void removeUserFromService(String userName,
                                      String serviceId,
                                      String serviceVersion,
                                      AxisService axisService)
            throws ServiceUserNotFoundException, ServiceNotFoundException {
        ServiceUserDO user = new UserDAO(hbConfig).getUser(userName);
        new ServiceDAO(hbConfig).removeServiceUser(serviceId,
                                                   serviceVersion,
                                                   user,
                                                   axisService);
    }

    public void deleteUser(String userName) {
        new UserDAO(hbConfig).deleteUser(userName);
    }

    public void deleteRole(String roleName) {
        new UserRoleDAO(hbConfig).deleteRole(roleName);
    }

    public void deleteRoleFromUser(String userName, String roleName) {
        new UserDAO(hbConfig).deleteRoleFromUser(userName, roleName);
    }

    public ServiceUserDO[] getUsers() {
        return new UserDAO(hbConfig).getUsers();
    }

    public void removeUser(String username) throws ServiceUserNotFoundException {
        new UserDAO(hbConfig).deleteUser(username);
    }

    public void deleteEntity(AbstractDataObject dataObject) {
        new BaseDAO(hbConfig).delete(dataObject);
    }

    public ServiceUserRoleDO[] getUserRoles() {
        return new UserRoleDAO(hbConfig).getUserRoles();
    }

    public ServiceDO getService(String serviceId, String serviceVersion) {
        if (serviceVersion == null || serviceVersion.trim().length() == 0) {
            serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        }
        return new ServiceDAO(hbConfig).getService(serviceId, serviceVersion);
    }

    public void addServiceGroup(ServiceGroupDO sgDO) {
        new ServiceGroupDAO(hbConfig).createServiceGroup(sgDO);
    }

    public ServiceGroupDO getServiceGroup(String serviceGroupId) {
        return new ServiceGroupDAO(hbConfig).getServiceGroup(serviceGroupId);
    }

    public ServiceDO[] getServices(ServiceGroupDO serviceGroup) {
        return new ServiceGroupDAO(hbConfig).getServices(serviceGroup);
    }

    public Long addService(ServiceDO service) throws ServiceAlreadyExistsException {
        try {
            return new ServiceDAO(hbConfig).create(service);
        } catch (DuplicateEntityException e) {
            throw new ServiceAlreadyExistsException(e);
        }
    }

    public void updateService(ServiceDO service) throws ServiceNotFoundException {
        new ServiceDAO(hbConfig).update(service);
    }

    public void updateOperation(OperationDO operation) throws ServiceNotFoundException {
        new OperationDAO(hbConfig).update(operation);
    }

    public void updateUser(ServiceUserDO user) throws ServiceUserNotFoundException {
        new UserDAO(hbConfig).update(user);
        user.notifyObservers();
    }

    public ServiceUserRoleDO getServiceUserRole(String role) {
        return new UserRoleDAO(hbConfig).getUserRole(role);
    }

    public Long addTransport(TransportDO protocol)
            throws TransportAlreadyExistsException {
        try {
            return new TransportDAO(hbConfig).create(protocol);
        } catch (DuplicateEntityException e) {
            throw new TransportAlreadyExistsException(e);
        }
    }

    public TransportDO getTransport(String protocol) {
        return new TransportDAO(hbConfig).getTransport(protocol);
    }

    public TransportDO[] getTransports() {
        return new TransportDAO(hbConfig).getTransports();
    }

    public Long addKeyStore(KeyStoreDO keyStore) throws KeyStoreAlreadyExistsException {
        try {
            return new KeyStoreDAO(hbConfig).create(keyStore);
        } catch (DuplicateEntityException e) {
            throw new KeyStoreAlreadyExistsException(e);
        }
    }

    public Long addToken(SecurityTokenDO tokenDO) throws TokenAlreadyExistsException {
        try {
            return new SecurityTokenDAO(hbConfig).create(tokenDO);
        } catch (DuplicateEntityException e) {
            throw new TokenAlreadyExistsException(e);
        }
    }

    public void deleteKeyStore(String keyStoreName) {
        new KeyStoreDAO(hbConfig).deleteKeyStore(keyStoreName);
    }

    public KeyStoreDO[] getKeyStores() {
        return new KeyStoreDAO(hbConfig).getKeyStores();
    }

    public KeyStoreDO getKeyStore(String ksName) {
        return new KeyStoreDAO(hbConfig).getKeyStore(ksName);
    }

    public String getConfigurationProperty(String propertyName) {
        return new ServerConfigurationDAO(hbConfig).getConfigurationProperty(propertyName);
    }

    public void updateConfigurationProperty(String propertyName, String value) {
        new ServerConfigurationDAO(hbConfig).updateConfigurationProperty(propertyName, value);
    }

    public void addServicePolicy(ServicePolicyDO servicePolicy)
            throws ServicePolicyAlreadyExistsException {

        String serviceId = servicePolicy.getService().getServiceIdentifierDO().getServiceId();
        String serviceVersion = servicePolicy.getService().getServiceIdentifierDO().getVersion();
        new ServiceDAO(hbConfig).addPolicy(serviceId, serviceVersion, servicePolicy);
    }

    public void updateServicePolicy(ServicePolicyDO servicePolicy)
            throws ServicePolicyNotFoundException {

        new ServiceDAO(hbConfig).updatePolicy(servicePolicy);
    }

    public void updateModulePolicy(ModulePolicyDO modulePolicy)
            throws ModulePolicyNotFoundException {

        new ModuleDAO(hbConfig).updatePolicy(modulePolicy);
    }

    public synchronized long getMaxMessageSequence(String serviceId,
                                                   String serviceVersion,
                                                   String operationName) {

        return new MessageDAO(hbConfig).getMaxMessageSequence(serviceId,
                                                              serviceVersion,
                                                              operationName);
    }

    public void addMessage(MessageDO message) {
        new MessageDAO(hbConfig).addMessage(message);
    }

    public MessageDO[] getMessages(MessageDO msgSearchCriteria) {
        return new MessageDAO(hbConfig).getMessages(msgSearchCriteria);
    }

    public void deleteService(String serviceId, String serviceVersion)
            throws ServiceNotFoundException {

        new ServiceDAO(hbConfig).deleteService(serviceId, serviceVersion);
    }

    public void deleteServiceGroup(AxisServiceGroup serviceGroup)
            throws ServiceGroupNotFoundException {
        Iterator services = serviceGroup.getServices();
        if (services.hasNext() && ((AxisService) services.next()).isClientSide()) {
            return;
        }
        new ServiceGroupDAO(hbConfig).deleteServiceGroup(serviceGroup.getServiceGroupName());
    }

    public ServiceUserDO[] getUsersForService(String serviceId,
                                              String serviceVersion) {
        return new ServiceDAO(hbConfig).getUsers(serviceId, serviceVersion);
    }


    public ServiceUserRoleDO[] getRolesForService(String serviceId,
                                                  String serviceVersion) {
        return new ServiceDAO(hbConfig).getRoles(serviceId, serviceVersion);
    }

    public void addEngagedOperationToModule(String moduleName,
                                            String moduleVersion,
                                            OperationDO op) {
        new ModuleDAO(hbConfig).addEngagedOperation(moduleName, moduleVersion, op);
    }

    public void addEngagedServiceToModule(String moduleName,
                                          String moduleVersion,
                                          ServiceDO service) {
        if (service == null) {
            return;
        }
        new ModuleDAO(hbConfig).addEngagedService(moduleName, moduleVersion, service);
    }

    public void addEngagedServiceGroupToModule(String moduleName,
                                               String moduleVersion,
                                               ServiceGroupDO serviceGrp) {
        if (serviceGrp == null) {
            return;
        }
        new ModuleDAO(hbConfig).addEngagedServiceGroup(moduleName, moduleVersion, serviceGrp);
    }

    public void addPolicyToService(String serviceId,
                                   String serviceVersion,
                                   ServicePolicyDO policy) {
        new ServiceDAO(hbConfig).addPolicy(serviceId, serviceVersion, policy);
    }

    public void addParameterToService(String serviceId,
                                      String serviceVersion,
                                      ServiceParameterDO param) {

        new ServiceDAO(hbConfig).addParameter(serviceId, serviceVersion, param);
    }

    public void addPolicyToModule(String moduleName, String moduleVersion, ModulePolicyDO policy) {
        new ModuleDAO(hbConfig).addPolicy(moduleName, moduleVersion, policy);
    }

    public void addParameterToModule(String moduleName,
                                     String moduleVersion,
                                     ModuleParameterDO param) {

        new ModuleDAO(hbConfig).addParameter(moduleName, moduleVersion, param);
    }

    public OperationDO[] getServiceOperations(String serviceId,
                                              String serviceVersion) {
        return new ServiceDAO(hbConfig).getOperations(serviceId, serviceVersion);
    }

    public OperationDO[] getServiceOperations(ServiceDO service) {
        return new ServiceDAO(hbConfig).getOperations(service);
    }

    public OperationDO[] getEngagedOperations(String moduleName,
                                              String moduleVersion) {
        return new ModuleDAO(hbConfig).getEngagedOperations(moduleName, moduleVersion);
    }

    public ServiceDO[] getEngagedServices(String moduleName,
                                          String moduleVersion) {
        return new ModuleDAO(hbConfig).getEngagedServices(moduleName, moduleVersion);
    }

    public ModuleDO[] getEngagedModules(String serviceId,
                                        String serviceVersion) {
        return new ServiceDAO(hbConfig).getEngagedModules(serviceId, serviceVersion);
    }

    public ModuleDO[] getEngagedModules(ServiceDO service) {
        return new ServiceDAO(hbConfig).getEngagedModules(service);
    }

    public ModuleDO[] getEngagedModules(OperationDO op) {
        return new OperationDAO(hbConfig).getEngagedModules(op);
    }

    public ServiceParameterDO[] getServiceParameters(ServiceDO service) {
        return new ServiceDAO(hbConfig).getParameters(service);
    }

    public ServiceParameterDO getServiceParameter(ServiceDO service, String paramName) {
        return new ServiceDAO(hbConfig).getParameter(service, paramName);
    }

    public ServiceGroupParameterDO getServiceGroupParameter(ServiceGroupDO serviceGroup,
                                                            String paramName) {
        return new ServiceGroupDAO(hbConfig).getParameter(serviceGroup, paramName);
    }

    public OperationParameterDO getOperationParameter(String serviceId,
                                                      String serviceVersion,
                                                      String operationId,
                                                      String paramName) {
        return new OperationDAO(hbConfig).getOperationParameter(serviceId,
                                                                serviceVersion,
                                                                operationId,
                                                                paramName);
    }

    public ServicePolicyDO[] getServicePolicies(ServiceDO service) {
        return new ServiceDAO(hbConfig).getPolicies(service);
    }

    public ModulePolicyDO[] getModulePolicies(ModuleDO module) {
        return new ModuleDAO(hbConfig).getPolicies(module);
    }

    public ModulePolicyDO[] getModulePolicies(String moduleName, String moduleVersion) {
        return new ModuleDAO(hbConfig).getPolicies(moduleName, moduleVersion);
    }

    public OperationParameterDO[] getOperationParameters(OperationDO op) {
        return new OperationDAO(hbConfig).getParameters(op);
    }

    public ModuleParameterDO[] getModuleParameters(ModuleDO module) {
        return new ModuleDAO(hbConfig).getParameters(module);
    }

    public ModuleParameterDO getModuleParameter(ModuleDO module, String paramName) {
        return new ModuleDAO(hbConfig).getParameter(module, paramName);
    }

    public ServiceUserRoleDO getUserRole(String role) {
        return new UserRoleDAO(hbConfig).getUserRole(role);
    }

    public ServiceUserRoleDO[] getUserRoles(ServiceUserDO user) {
        return new UserDAO(hbConfig).getUserRoles(user);
    }

    public ServiceUserRoleDO[] getUserSpecificRoles(String user) {
        return new UserDAO(hbConfig).getRoles(user);
    }

    public void addRole(String userName, ServiceUserRoleDO role) {
        new UserDAO(hbConfig).addRole(userName, role);
        role.notifyObservers();
    }

    public OperationDO getOperation(String serviceId, String serviceVersion, String opName) {
        return new OperationDAO(hbConfig).getOperation(serviceId, serviceVersion, opName);
    }

    public OperationDO[] getOperations(ServiceDO service) {
        return new ServiceDAO(hbConfig).getOperations(service);
    }

    public void addEntity(AbstractDataObject obj) throws DuplicateEntityException {
        new BaseDAO(hbConfig).create(obj);
    }

    public void updateEntity(AbstractDataObject obj) {
        new BaseDAO(hbConfig).update(obj);
    }

    public void removeEngagedServiceFromModule(String moduleName,
                                               String moduleVersion,
                                               ServiceDO service) {
        if (service != null) {
            new ModuleDAO(hbConfig).removeEngagedService(moduleName, moduleVersion, service);
        }
    }

    public void removeEngagedServiceGroupFromModule(String moduleName,
                                                    String moduleVersion,
                                                    ServiceGroupDO serviceGrp) {
        if (serviceGrp != null) {
            new ModuleDAO(hbConfig).removeEngagedServiceGroup(moduleName, moduleVersion, serviceGrp);
        }
    }

    public void removeEngagedOperationFromModule(String moduleName,
                                                 String moduleVersion,
                                                 OperationDO operation) {

        if (operation != null) {
            new ModuleDAO(hbConfig).removeEngagedOperation(moduleName, moduleVersion, operation);
        }
    }

    public void deleteModule(String moduleName, String moduleVersion) {
        new ModuleDAO(hbConfig).deleteModule(moduleName, moduleVersion);
    }

    public void handleNewServiceAddition(AxisService axisService) throws Exception {
        if (axisService.isClientSide()) {
            return;
        }
        String sgName = ((AxisServiceGroup) axisService.getParent()).getServiceGroupName();
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        ServiceGroupDO serviceGroupDO = new ServiceGroupDAO(hbConfig).getServiceGroup(sgName,
                                                                                      session);
        if (serviceGroupDO == null) {
            tx.rollback();
            return;
        }

        ServiceDO serviceDO = new ServiceDO();
        ServiceIdentifierDO serviceIdentifier = new ServiceIdentifierDO();
        serviceDO.setServiceIdentifierDO(serviceIdentifier);
        serviceIdentifier.setServiceId(axisService.getName());
        serviceIdentifier.setVersion(ServiceIdentifierDO.EMPTY_SERVICE_VERSION);

        serviceDO.setDocumentation(axisService.getDocumentation());

        // Add the Service Policies
        ServicePolicyDO[] servicePolicies = getServicePolicies(axisService);
        for (int i = 0; i < servicePolicies.length; i++) {
            serviceDO.addPolicy(servicePolicies[i]);
        }

        // Add the Service Operations
        for (Iterator iter = axisService.getOperations(); iter.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) iter.next();
            OperationDO op = new OperationDO();
            op.setName(axisOperation.getName().getLocalPart());
            op.setDocumentation(axisOperation.getDocumentation());
            serviceDO.addOperation(op);
        }

        // If the service scope='soapsession' we need to engage addressing if not already
        // engaged.
        if (axisService.getScope().equals(Constants.SCOPE_SOAP_SESSION)) {
            if (!axisService.isEngaged("addressing")) {
                axisService.engageModule(axisService.getAxisConfiguration().getModule("addressing"));
            }
        }

        // Add the Service Group
        serviceDO.setServiceGroup(serviceGroupDO);

        try {
            serviceDO.setLastUpdatedTime(new Date());
            serviceDO.setIsActive(true);
            session.persist(serviceDO);
            if (log.isDebugEnabled()) {
                log.debug("Added new service - " + axisService.getName());
            }

            // Add the Modules Engaged to this service
            ModuleDAO modDAO = new ModuleDAO(hbConfig);
            for (Iterator serviceModuleEngagementIter = axisService.getEngagedModules().iterator();
                 serviceModuleEngagementIter.hasNext();) {
                AxisModule axisModule = (AxisModule) serviceModuleEngagementIter.next();
                String moduleName = axisModule.getName();
                String moduleVersion = axisModule.getVersion();
                if (moduleVersion == null) {
                    moduleVersion = "";
                }
                if (!axisService.getParent().isEngaged(axisModule.getName())) {
                    ModuleDO moduleDO = modDAO.getModule(moduleName, moduleVersion, session);
                    if (moduleDO != null && !moduleDO.getEngagedServices().contains(serviceDO)) {
                        moduleDO.addService(serviceDO);
                        session.update(moduleDO);
                        session.update(serviceDO);
                    }
                }
            }

            // Save the operation-module engagements
            OperationDAO opDAO = new OperationDAO(hbConfig);
            for (Iterator iter = axisService.getOperations(); iter.hasNext();) {
                AxisOperation axisOperation = (AxisOperation) iter.next();
                for (Iterator opModuleEngagementIter = axisOperation.getEngagedModules().iterator();
                     opModuleEngagementIter.hasNext();) {
                    AxisModule axisModule = (AxisModule) opModuleEngagementIter.next();
                    if (!axisService.isEngaged(axisModule.getName())) {
                        OperationDO opDO = opDAO.getOperation(axisService.getName(),
                                                              ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                                              axisOperation.getName().getLocalPart(),
                                                              session);
                        String moduleName = axisModule.getName();
                        String moduleVersion = axisModule.getVersion();
                        if (moduleVersion == null) {
                            moduleVersion = "";
                        }
                        ModuleDO moduleDO = modDAO.getModule(moduleName, moduleVersion, session);
                        if (moduleDO != null &&
                            !moduleDO.getEngagedOperations().contains(serviceDO)) {
                            moduleDO.addOperation(opDO);
                            session.update(moduleDO);
                            session.update(opDO);
                        }
                    }
                }
            }

            // add the service parameters
            for (Iterator iter = axisService.getParameters().iterator(); iter.hasNext();) {
                Parameter parameter = (Parameter) iter.next();
                String paramName = parameter.getName();
                if (paramName != null && paramName.trim().length() != 0) {
                    ServiceParameterDO parameterDO = new ServiceParameterDO();
                    parameterDO.setName(paramName);
                    if (parameter.getParameterElement() != null) {
                        parameterDO.setValue(parameter.getParameterElement().toString());
                        parameterDO.setService(serviceDO);
                    } else if (parameter.getValue() != null &&
                               parameter.getValue() instanceof String) {
                        parameter = ParameterUtil.createParameter(paramName.trim(),
                                                                  (String) parameter.getValue());
                        parameterDO.setValue(parameter.getParameterElement().toString());
                        parameterDO.setService(serviceDO);
                    }
                }
            }

            // add the transports via which this service is exposed
            serviceDO.setIsExposedOnAllTransports(axisService.isEnableAllTransports());
            if (!axisService.isEnableAllTransports()) {
                List exposedTransports = axisService.getExposedTransports();
                TransportDAO transportDAO = new TransportDAO(hbConfig);
                for (Iterator iterator = exposedTransports.iterator(); iterator.hasNext();) {
                    TransportDO transportDO = transportDAO.getTransport((String) iterator.next(),
                                                                        session);
                    serviceDO.addTransport(transportDO);
                }
            }

            session.flush();
            tx.commit();
        } catch (Exception e) {
            log.error("Unable to handle new service addition", e);
            session.flush();
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
    }

    /**
     * Extract all the service policies from the AxisService.
     *
     * @param service
     * @return policies
     * @throws XMLStreamException
     */
    private ServicePolicyDO[] getServicePolicies(AxisService service) throws XMLStreamException {
        PolicyInclude include = service.getPolicyInclude();

        List servicePolicyDOs = new ArrayList();

        addServicePolicy(include, servicePolicyDOs, PolicyInclude.AXIS_SERVICE_POLICY);
        addServicePolicy(include, servicePolicyDOs, PolicyInclude.SERVICE_POLICY);
        addServicePolicy(include, servicePolicyDOs, PolicyInclude.PORT_POLICY);
        addServicePolicy(include, servicePolicyDOs, PolicyInclude.BINDING_POLICY);
        addServicePolicy(include, servicePolicyDOs, PolicyInclude.PORT_TYPE_POLICY);
        return (ServicePolicyDO[])
                servicePolicyDOs.toArray(new ServicePolicyDO[servicePolicyDOs.size()]);
    }

    /**
     * Extract all the service policies from the AxisService.
     *
     * @param axisModule
     * @return module policies
     * @throws XMLStreamException
     */
    private ModulePolicyDO[] getModulePolicies(AxisModule axisModule) throws XMLStreamException {
        PolicyInclude include = axisModule.getPolicyInclude();
        List modulePolicyDOs = new ArrayList();

        addModulePolicy(include, modulePolicyDOs, PolicyInclude.AXIS_MODULE_POLICY);
        return (ModulePolicyDO[])
                modulePolicyDOs.toArray(new ModulePolicyDO[modulePolicyDOs.size()]);
    }

    private void addServicePolicy(PolicyInclude include,
                                  List servicePolicyDOs,
                                  int policyType) throws XMLStreamException {
        for (Iterator iterator =
                include.getPolicyElements(policyType).iterator();
             iterator.hasNext();) {

            ServicePolicyDO servicePolicyDO = new ServicePolicyDO();
            servicePolicyDO.setType(policyType);

            Object o = iterator.next();
            Policy policy;
            if (o instanceof Policy) {
                policy = (Policy) o;
            } else if (o instanceof PolicyReference) {
                PolicyReference pRef = (PolicyReference) o;
                PolicyRegistry policyRegistry = include.getPolicyRegistry();

                String key = pRef.getURI();
                if (key.startsWith("#")) {
                    key = key.substring(key.indexOf("#") + 1);
                }

                policy = policyRegistry.lookup(key);

            } else {
                throw new IllegalArgumentException("Invalid Policy Item found in collection. " +
                                                   "This is a programmatic error");
            }
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(baos);

            policy.serialize(writer);
            writer.flush();

            servicePolicyDO.setUuid(policy.getId());
            servicePolicyDO.setPolicy(baos.toString());
            servicePolicyDOs.add(servicePolicyDO);
        }
    }

    private void addModulePolicy(PolicyInclude include,
                                 List modulePolicyDOs,
                                 int policyType) throws XMLStreamException {
        for (Iterator iterator =
                include.getPolicyElements(policyType).iterator();
             iterator.hasNext();) {

            ModulePolicyDO modulePolicyDO = new ModulePolicyDO();
            modulePolicyDO.setType(policyType);

            Object o = iterator.next();
            Policy policy;
            if (o instanceof Policy) {
                policy = (Policy) o;
            } else if (o instanceof PolicyReference) {
                PolicyReference pRef = (PolicyReference) o;
                PolicyRegistry policyRegistry = include.getPolicyRegistry();

                String key = pRef.getURI();
                if (key.startsWith("#")) {
                    key = key.substring(key.indexOf("#") + 1);
                }

                policy = policyRegistry.lookup(key);
            } else {
                throw new IllegalArgumentException("Invalid Policy Item found in collection. " +
                                                   "This is a programmatic error");
            }
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(baos);

            policy.serialize(writer);
            writer.flush();

            modulePolicyDO.setUuid(policy.getId());
            modulePolicyDO.setPolicy(baos.toString());
            modulePolicyDOs.add(modulePolicyDO);
        }
    }

    public void handleExistingServiceInit(ServiceDO serviceDO,
                                          AxisService axisService,
                                          AxisConfiguration axisConfig) throws Exception {

        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();

        try {
            serviceDO = (ServiceDO) session.load(ServiceDO.class, serviceDO.getId());
            String serviceId = serviceDO.getServiceIdentifierDO().getServiceId();
            String version = serviceDO.getServiceIdentifierDO().getVersion();

            // Handle Security stuff
//            if (serviceDO.getIsUTAuthEnabled()) {
//                try {
            axisService.addParameter(WSS4JUtil.
                    getPasswordCallBackRefParameter(serviceId, version));
//                } catch (AxisFault axisFault) {
//                    log.error("Cannot add parameter to AxisService", axisFault);
//                }
//            }

            // Load policies from DB in AxisService object. All other policies contained within
            // axisService will be rendered invalid by this time
            PolicyInclude policyInclude = axisService.getPolicyInclude();
            policyInclude.removeAllPolicyElements();
            for (Iterator iter = serviceDO.getPolicies().iterator(); iter.hasNext();) {
                ServicePolicyDO policyDO = (ServicePolicyDO) iter.next();
                String policyStr = policyDO.getPolicy();
                Policy policy = PolicyEngine.getPolicy(new ByteArrayInputStream(policyStr.getBytes()));
                policyInclude.addPolicyElement(policyDO.getType(), policy);
            }

            // Add the Service Parameters
            for (Iterator iter = serviceDO.getParameters().iterator(); iter.hasNext();) {
                ServiceParameterDO paramDO = (ServiceParameterDO) iter.next();
                paramDO.getValue();
                ByteArrayInputStream bais =
                        new ByteArrayInputStream(paramDO.getValue().getBytes());
                StAXOMBuilder builder;
                try {
                    builder = new StAXOMBuilder(bais);
                } catch (XMLStreamException e) {
                    String msg = "Unable to set service parameter";
                    log.error(msg, e);
                    throw new AxisFault(msg, e);
                }
                Parameter parameter = ParameterUtil.createParameter(builder.getDocumentElement());
                Parameter p = axisService.getParameter(paramDO.getName());
                if (p != null) {
                    if (!p.isLocked()) {
                        axisService.addParameter(parameter);
                    }
                } else {
                    axisService.addParameter(parameter);
                }
            }

            //add description
            String doc = serviceDO.getDocumentation();
            if (doc != null) {
                axisService.setDocumentation(doc);
            }

            // Disengage all the statically engaged modules (i.e. those module engaged from the services.xml file)
            axisService.getEngagedModules().clear();

            // Engage modules to service
            for (Iterator iter = serviceDO.getEngagedModules().iterator(); iter.hasNext();) {
                ModuleDO moduleDO = (ModuleDO) iter.next();
                ModuleIdentifierDO modID = moduleDO.getModuleIdentifierDO();
                AxisModule axisModule = axisConfig.getModule(modID.getName(), modID.getVersion());
                if (!isGloballyEngaged(modID.getName(), modID.getVersion(), session)) {
                    axisService.disengageModule(axisModule);
                    axisService.engageModule(axisModule);
                }
            }

            // Engage modules to operation
            for (Iterator iter = serviceDO.getOperations().iterator(); iter.hasNext();) {
                OperationDO opDO = (OperationDO) iter.next();
                AxisOperation axisOp = axisService.getOperation(new QName(opDO.getName()));

                // Disengage all the statically engaged modules (i.e. those module engaged from the services.xml file)
                axisOp.getEngagedModules().clear();

                for (Iterator iter2 = opDO.getEngagedModules().iterator(); iter2.hasNext();) {
                    ModuleDO moduleDO = (ModuleDO) iter2.next();
                    ModuleIdentifierDO modID = moduleDO.getModuleIdentifierDO();
                    AxisModule axisModule = axisConfig.getModule(modID.getName(),
                                                                 modID.getVersion());
                    if (!isGloballyEngaged(modID.getName(), modID.getVersion(), session)) {
                        if (!axisService.isEngaged(axisModule) && !axisOp.isControlOperation()) {
                            if (!axisOp.isEngaged(axisModule)) {
                                axisOp.engageModule(axisModule);
                            }
                        }
                    }
                }

                //set operation description
                String documentation = opDO.getDocumentation();
                if (documentation != null) {
                    axisOp.setDocumentation(documentation);
                }

                // Handle operation parameters
                for (Iterator opIter = opDO.getParameters().iterator(); opIter.hasNext();) {
                    OperationParameterDO opParamDO = (OperationParameterDO) opIter.next();
                    opParamDO.getValue();
                    ByteArrayInputStream bais =
                            new ByteArrayInputStream(opParamDO.getValue().getBytes());
                    StAXOMBuilder builder;
                    try {
                        builder = new StAXOMBuilder(bais);
                    } catch (XMLStreamException e) {
                        String msg = "Unable to set operation parameter";
                        log.error(msg, e);
                        throw new AxisFault(msg, e);
                    }
                    Parameter parameter =
                            ParameterUtil.createParameter(builder.getDocumentElement());
                    Parameter p = axisOp.getParameter(opParamDO.getName());
                    if (p != null) {
                        if (!p.isLocked()) {
                            axisOp.addParameter(parameter);
                        }
                    } else {
                        axisOp.addParameter(parameter);
                    }
                }
            }

            // Handle the exposed transports
            if (!serviceDO.getIsExposedOnAllTransports()) {
                axisService.setExposedTransports(new ArrayList());
                for (Iterator transpIter = serviceDO.getTransports().iterator();
                     transpIter.hasNext();) {
                    String transportProtocol =
                            ((TransportDO) transpIter.next()).getTransportProtocol();
                    axisService.addExposedTransport(transportProtocol);
                }
            }

            // Activate/Deactivate service
            axisService.setActive(serviceDO.getIsActive());
            tx.commit();
            if (log.isDebugEnabled()) {
                log.debug("Initialized service - " + serviceDO.getServiceIdentifierDO().getServiceId());
            }
        } catch (Exception e) {
            log.error("Unable to handle service initialization", e);
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
    }

    private boolean isGloballyEngaged(String moduleId, String moduleVersion, Session session) {
        String query = "from ModuleDO";
        List list = session.createQuery(query).list();
        ModuleDO[] modules = (ModuleDO[]) list.toArray(new ModuleDO[list.size()]);

        for (int i = 0; i < modules.length; i++) {
            ModuleDO moduleDO = modules[i];
            if (moduleDO.getIsGloballyEngaged()) {
                if (moduleDO.getModuleIdentifierDO().getName().equals(moduleId) &&
                    moduleDO.getModuleIdentifierDO().getVersion().equals(moduleVersion)) {
                    return true;
                }
            }
        }
        return false;
    }

    public void handleExistingServiceGroupInit(ServiceGroupDO sgDO,
                                               AxisServiceGroup serviceGroup) throws Exception {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();

        try {
            sgDO = (ServiceGroupDO) session.load(ServiceGroupDO.class, sgDO.getId());

            // Add the Service Group Parameters
            for (Iterator iter = sgDO.getParameters().iterator(); iter.hasNext();) {
                ServiceGroupParameterDO paramDO = (ServiceGroupParameterDO) iter.next();
                paramDO.getValue();
                ByteArrayInputStream bais =
                        new ByteArrayInputStream(paramDO.getValue().getBytes());
                StAXOMBuilder builder;
                try {
                    builder = new StAXOMBuilder(bais);
                } catch (XMLStreamException e) {
                    String msg = "Unable to set service group parameter";
                    log.error(msg, e);
                    throw new AxisFault(msg, e);
                }
                Parameter parameter = ParameterUtil.createParameter(builder.getDocumentElement());
                Parameter p = serviceGroup.getParameter(paramDO.getName());
                if (p != null) {
                    if (!p.isLocked()) {
                        serviceGroup.addParameter(parameter);
                    }
                } else {
                    serviceGroup.addParameter(parameter);
                }
            }

            AxisConfiguration axisConfig = serviceGroup.getAxisConfiguration();

            // Disengage all the statically engaged modules (i.e. those module engaged from the services.xml file)
            serviceGroup.getEngagedModules().clear();

            // Engage modules to service group
            for (Iterator iter = sgDO.getEngagedModules().iterator(); iter.hasNext();) {
                ModuleDO moduleDO = (ModuleDO) iter.next();
                ModuleIdentifierDO modID = moduleDO.getModuleIdentifierDO();
                AxisModule axisModule = axisConfig.getModule(modID.getName(), modID.getVersion());
                if (!isGloballyEngaged(modID.getName(), modID.getVersion(), session)) {
                    serviceGroup.disengageModule(axisModule);
                    serviceGroup.engageModule(axisModule);
                }
            }
            tx.commit();
            if (log.isDebugEnabled()) {
                log.debug("Initialized Service Group - " + sgDO.getName());
            }
        } catch (Exception e) {
            log.error("Unable to handle service group init", e);
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
    }

    public void handleNewServiceGroupAddition(AxisServiceGroup serviceGroup) throws Exception {
        Iterator services = serviceGroup.getServices();
        if (services.hasNext() && ((AxisService) services.next()).isClientSide()) {
            return;
        }
        ServiceGroupDO sgDO = new ServiceGroupDO();
        sgDO.setServiceArtifactUpdateTime(WsasUtils.lastUpdatedTime(serviceGroup));
        String sgName = serviceGroup.getServiceGroupName();
        sgDO.setName(sgName);
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        AxisConfiguration axisConfiguration = serviceGroup.getAxisConfiguration();
        try {
            ModuleDAO moduleDAO = new ModuleDAO(hbConfig);

            // Handle ServiceGroup-Module engagement
            for (Iterator iter = serviceGroup.getEngagedModules().iterator(); iter.hasNext();) {
                AxisModule axisModule = (AxisModule) iter.next();
                if (!axisConfiguration.isEngaged(axisModule.getName())) {
                    ModuleDO moduleDO =
                            moduleDAO.getModule(axisModule.getName(),
                                                axisModule.getVersion(),
                                                session);
                    if (moduleDO != null && moduleDO.getEngagedServiceGroups().contains(sgDO)) {
                        sgDO.addModule(moduleDO);
                    }
                }
            }

            // Handle Service Group Parameters
            for (Iterator iter = serviceGroup.getParameters().iterator(); iter.hasNext();) {
                Parameter parameter = (Parameter) iter.next();
                if (parameter.getName() != null && parameter.getName().trim().length() != 0) {
                    ServiceGroupParameterDO paramDO = new ServiceGroupParameterDO();
                    paramDO.setName(parameter.getName());
                    if (parameter.getParameterElement() != null) {
                        paramDO.setValue(parameter.getParameterElement().toString());
                        paramDO.setServiceGroup(sgDO);
//                        session.persist(paramDO);
                    }
                }
            }
            new ServiceGroupDAO(hbConfig).createServiceGroup(sgDO, session);
            session.flush();
            tx.commit();
        } catch (Exception e) {
            log.error("Unable to handle new service group addition", e);
            session.flush();
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
        if (log.isDebugEnabled()) {
            log.debug("Added new service group - " + sgName);
        }
    }

    public void handleNewModuleAddition(AxisModule axisModule,
                                        String moduleName,
                                        String moduleVersion) throws Exception {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();

        try {
            ModuleDO moduleDO;
            moduleDO = new ModuleDO();
            ModuleIdentifierDO modID = new ModuleIdentifierDO();

            modID.setName(moduleName);
            modID.setVersion(moduleVersion);
            moduleDO.setModuleIdentifierDO(modID);
            moduleDO.setIsGloballyEngaged(ServerConfigurator.getInstance().isGlobalyEngaged(axisModule));

            // add the module parameters
            for (Iterator iter = axisModule.getParameters().iterator(); iter.hasNext();) {
                Parameter param = (Parameter) iter.next();
                ModuleParameterDO paramDO = new ModuleParameterDO();
                paramDO.setName(param.getName());
                paramDO.setValue(param.getParameterElement().toString());
                moduleDO.addParameter(paramDO);
            }
            session.persist(moduleDO);

            // Persist module policies
            ModulePolicyDO[] modulePolicies = getModulePolicies(axisModule);
            for (int i = 0; i < modulePolicies.length; i++) {
                moduleDO.addPolicy(modulePolicies[i]);
            }
            session.flush();
            tx.commit();
            if (log.isDebugEnabled()) {
                log.debug("Added new module - " + axisModule.getName() + "-" + axisModule.getVersion());
            }
        } catch (Exception e) {
            log.error("Unable to handle new module addition", e);
            markFaultyModule(axisModule);
            session.flush();
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
    }

    private void markFaultyModule(AxisModule axisModule) {
        axisModule.getParent().
                getFaultyModules().
                put(Utils.getModuleName(axisModule.getName(), axisModule.getVersion()),
                    axisModule);
    }

    public void handleExistingModuleInit(ModuleDO moduleDO, AxisModule axisModule)
            throws Exception {
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();

        try {
            moduleDO = (ModuleDO) session.load(ModuleDO.class, moduleDO.getId());

            // Add the Module Parameters
            for (Iterator iter = moduleDO.getParameters().iterator(); iter.hasNext();) {
                ModuleParameterDO param = (ModuleParameterDO) iter.next();
                ByteArrayInputStream bais =
                        new ByteArrayInputStream(param.getValue().getBytes());
                StAXOMBuilder builder;
                try {
                    builder = new StAXOMBuilder(bais);
                } catch (XMLStreamException e) {
                    String msg = "Unable to set service parameter";
                    log.error(msg, e);
                    throw new AxisFault(msg, e);
                }
                Parameter parameter = ParameterUtil.createParameter(builder.getDocumentElement());
                Parameter p = axisModule.getParameter(param.getName());
                if (p != null) {
                    if (!p.isLocked()) {
                        axisModule.addParameter(parameter);
                    }
                } else {
                    axisModule.addParameter(parameter);
                }
            }

            // Load policies from DB into AxisModule. All other policies contained within
            // axisModule will be rendered invalid by this time
            PolicyInclude policyInclude = axisModule.getPolicyInclude();
            policyInclude.removeAllPolicyElements();
            for (Iterator iter = moduleDO.getPolicies().iterator(); iter.hasNext();) {
                ModulePolicyDO policyDO = (ModulePolicyDO) iter.next();
                String policyStr = policyDO.getPolicy();
                Policy policy = PolicyEngine.getPolicy(new ByteArrayInputStream(policyStr.getBytes()));
                policyInclude.addPolicyElement(policyDO.getType(), policy);
            }

            tx.commit();
            if (log.isDebugEnabled()) {
                log.debug("Initialized module - " + Utils.getModuleName(axisModule.getName(),
                                                                        axisModule.getVersion()));
            }
        } catch (Exception e) {
            log.error("Unable to handle module init", e);
            markFaultyModule(axisModule);
            tx.rollback();
            throw e;
        } finally {
            hbConfig.closeSession();
        }
    }

    public boolean userHasRole(String username, ServiceUserRoleDO role) {
        return new UserDAO(hbConfig).userHasRole(username, role);
    }

    public List getAllAppenders() {
        return new AppenderDAO(hbConfig).getAllAppenders();
    }

    public List getAllLoggers() {
        return new LoggerDAO(hbConfig).getAllLoggers();
    }

    public void addAppender(AppenderDO appenderDO) {
        new AppenderDAO(hbConfig).updateAppenderDO(appenderDO);
    }

    public void addLogger(LoggerDO loggerDO) {
        new LoggerDAO(hbConfig).updateLoggerDO(loggerDO);
    }

    public AppenderDO getAppenderDO(String appenderName) {
        return new AppenderDAO(hbConfig).getAppenderDO(appenderName);
    }

    public LoggerDO getLoggerDO(String loggerName) {
        return new LoggerDAO(hbConfig).getLoggerDO(loggerName);
    }

    public void updateAppenderData(String appenderName,
                                   String appenderPattern,
                                   String threshold,
                                   String logFileName,
                                   String sysLogHost,
                                   String facility,
                                   boolean isFileAppender,
                                   boolean isSysLogAppender) {

        AppenderDO appenderDO = new AppenderDO();
        appenderDO.setName(appenderName);
        appenderDO.setPattern(appenderPattern);
        appenderDO.setThreshold(threshold);
        appenderDO.setLogFileName(logFileName);
        appenderDO.setSysLogHost(sysLogHost);
        appenderDO.setFacility(facility);
        appenderDO.setIsFileAppender(isFileAppender);
        appenderDO.setIsSysLogAppender(isSysLogAppender);
        new AppenderDAO(hbConfig).updateAppenderDO(appenderDO);
    }

    public void updateLoggerData(String loggerName, String loggerLevel, boolean aditivity) {
        LoggerDO loggerDO = new LoggerDO();
        loggerDO.setName(loggerName);
        loggerDO.setLogLevel(loggerLevel);
        loggerDO.setAdditivity(aditivity);
        new LoggerDAO(hbConfig).updateLoggerDO(loggerDO);
    }

    public SecurityTokenDO[] getAllSecurityTokens() {
        return new SecurityTokenDAO(hbConfig).getAllTokens();
    }

    public SecurityScenarioDO getSecurityScenario(String scenarioId) {
        return new SecurityScenarioDAO(hbConfig).getSecurityScenario(scenarioId);
    }

    public void removeAllTrustedCertStores(ServiceDO serviceDO) {
        new ServiceDAO(hbConfig).removeAllTrustedCertStores(serviceDO);
    }

    public void removeAllLoggersAndAppenders() {
        String loggerQuery = "delete from LoggerDO";
        String appenderQuery = "delete from AppenderDO";
        Session session = hbConfig.currentSession();
        Transaction tx = session.beginTransaction();
        try {
            session.createQuery(loggerQuery).executeUpdate();
            session.createQuery(appenderQuery).executeUpdate();
            tx.commit();
        } catch (Throwable e) {
            tx.rollback();
            String msg = "Unable to removeAllLoggersAndAppenders";
            log.error(msg, e);
            throw new RuntimeException(msg, e);
        } finally {
            hbConfig.closeSession();
        }
    }

    public Long addEJBConfiguration(EJBProviderDO ejbProvider)
            throws EJBConfigurationAlreadyExistsException {
        try {
            return new EJBConfigurationDAO(hbConfig).create(ejbProvider);
        } catch (DuplicateEntityException e) {
            throw new EJBConfigurationAlreadyExistsException(e);
        }
    }

    public EJBProviderDO[] getEJBConfigurations() {
        return new EJBConfigurationDAO(hbConfig).getEJBConfigurations();
    }

    public EJBProviderDO getEJBConfiguration(String beanJNDIName, String jnpProviderUrl)
            throws EJBConfigurationNotFoundException {
        return new EJBConfigurationDAO(hbConfig).getEJBConfiguration(beanJNDIName, jnpProviderUrl);
    }

    public void removeEJBConfiguration(String beanJNDIName, String jnpProviderUrl)
            throws EJBConfigurationNotFoundException {
        new EJBConfigurationDAO(hbConfig).deleteEJBConfiguration(beanJNDIName, jnpProviderUrl);
    }

    public Long addEJBAppServerConfiguration(EJBAppServerDO ejbAppServerDO)
            throws EJBAppServerAlreadyExistsException {
        try {
            return new EJBAppServerDAO(hbConfig).create(ejbAppServerDO);
        } catch (DuplicateEntityException e) {
            throw new EJBAppServerAlreadyExistsException(e);
        }
    }

    public EJBAppServerDO[] getEJBAppServerConfigurations() {
        return new EJBAppServerDAO(hbConfig).getEJBAppServers();
    }

    public EJBAppServerDO getEJBAppServerConfiguration(String jnpProviderUrl) {
        return new EJBAppServerDAO(hbConfig).getEJBAppServer(jnpProviderUrl);
    }

    public void removeEJBAppServerConfiguration(String jnpProviderUrl)
            throws EJBAppServerNotFoundException {
        new EJBAppServerDAO(hbConfig).deleteEJBAppServer(jnpProviderUrl);
    }

    //TODO : Complete - Azeez
    public void restoreDefaults() {
//        //TODO: Method implementation
//        String serviceGroupQuery = "delete from ServiceGroupDO";
//        String serviceQuery = "delete from ServiceDO";
//        String moduleQuery = "delete from ModuleDO";
//        String loggerQuery = "delete from LoggerDO";
//        String appenderQuery = "delete from AppenderDO";
//        Session session = hbConfig.currentSession();
//        Transaction tx = session.beginTransaction();
//        try {
//            session.createQuery(loggerQuery).executeUpdate();
//            session.createQuery(appenderQuery).executeUpdate();
//            tx.commit();
//        } catch (Throwable e) {
//            tx.rollback();
//            String msg = "Unable to removeAllLoggersAndAppenders";
//            log.error(msg, e);
//            throw new RuntimeException(msg, e);
//        } finally {
//            hbConfig.closeSession();
//        }
    }
}