/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.installer;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMXMLParserWrapper;
import org.apache.axiom.om.impl.llom.factory.OMXMLBuilderFactory;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.XPath;
import org.wso2.utils.ArchiveManipulator;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.InputReader;
import org.wso2.wsas.serverinfo.GenericServerInfo;
import org.wso2.wsas.serverinfo.Tomcat41xServerInfo;
import org.wso2.wsas.serverinfo.Tomcat5xServerInfo;
import org.wso2.wsas.serverinfo.Tomcat6xServerInfo;
import org.wso2.wsas.transport.ServerPropertyKeys;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Iterator;
import java.util.Properties;

/**
 *
 */
public class TomcatInstaller extends AbstractAppServerInstaller {

    private FileManipulator fileMan = new FileManipulator();
    private String serverInfoImplClass;
    private String serverXml;
    private static final String TOMCAT_4_1_X = "tomcat-4.1.x";
    private static final String TOMCAT_5_0_X = "tomcat-5.0.x";
    private static final String TOMCAT_5_5_X = "tomcat-5.5.x";
    private static final String TOMCAT_6_X = "tomcat-6.x";

    public void install() throws InstallationException {

        // Get the Tomcat version
        System.out.println("Please select your Tomcat version : ");
        String[] versions = getSupportedVersions();
        for (int i = 0; i < versions.length; i++) {
            System.out.println((i + 1) + ". " + versions[i]);
        }
        int serverVersionIndex = getServerVersionIndex();
        String version = versions[serverVersionIndex];
        serverInfoImplClass = getServerInfoImplClasses()[serverVersionIndex];
        if (version.equals(OTHER_VERSION)) {
            new GenericInstaller().install();
            return;
        }

        // Capture the Tomcat Home
        if (this.serverHome != null && this.serverHome.trim().length() != 0 &&
            !this.serverHome.equals("INVALID")) {
            System.out.print("Please enter your Tomcat Home directory [" +
                             this.serverHome + "]: ");
        } else {
            System.out.print("Please enter your Tomcat Home directory : ");
        }
        serverHome = getAppServerHome();

        File tomcatHomeDir = new File(serverHome);
        if (!tomcatHomeDir.exists()) {
            String msg = "Tomcat home " + serverHome + " does not exist! Aborting installation.";
            System.err.println(" " + msg);
            throw new InstallationException(msg);
        }

        if (serverHome.toLowerCase().indexOf("tomcat") == -1) {
            System.out.print("[WARN] " + serverHome +
                             " does not look like a Tomcat home directory.");
            System.out.print(" Do you want to continue (y/n)? [y]: ");
            String response;
            do {
                try {
                    response = InputReader.readInput();
                } catch (IOException e) {
                    throw new InstallationException(e);
                }
            } while (!response.equalsIgnoreCase("y") && !response.equalsIgnoreCase("n") &&
                     !(response.trim().length() == 0));
            if (response.equalsIgnoreCase("n")) {
                System.out.println("[INFO] Installation aborted.");
                return;
            }
        }

        // copy the server.xml
        boolean isServerXMLReplaced = copyServerXML(serverHome, version);

        // Re-archive the wso2wsas.war and copy it to the webapps dir
        createWebArchive();

        copyWsasWAR(serverHome + File.separator + "webapps");
        if (version.equals(TOMCAT_4_1_X)) {
            try {
                File wso2wsasDir = new File(serverHome + File.separator +
                                            "webapps" + File.separator + "wso2wsas");
                if (!wso2wsasDir.mkdirs()) {
                    throw new InstallationException("Cannot create wso2wsas directory in webapps");
                }
                copyWsasWAR(serverHome + File.separator + "webapps" + File.separator + "wso2wsas");
                new ArchiveManipulator().extract(serverHome + File.separator + "webapps" +
                                                 File.separator + "wso2wsas" + File.separator + "wso2wsas.war",
                                                 serverHome + File.separator + "webapps" +
                                                 File.separator + "wso2wsas");
                new File(serverHome + File.separator + "webapps" + File.separator + "wso2wsas" +
                         File.separator + "wso2wsas.war").delete();
                fileMan.copyDir(new File(InstallerConstants.TMP_DIR), wso2wsasDir);
                fileMan.copyFile(new File("conf" + File.separator + "log4j.properties"),
                                 new File(serverHome + File.separator + "common" +
                                          File.separator + "classes" + File.separator +
                                          "log4j.properties"));

                // Copy the wos2utils.jar
                String wso2utilsJar = getItem("lib", "wso2utils-", ".jar");

                fileMan.copyFile(new File("lib" + File.separator + wso2utilsJar),
                                 new File(serverHome + File.separator + "common" +
                                          File.separator + "lib" + File.separator +
                                          wso2utilsJar));
                copyDerbyJAR(serverHome + File.separator + "shared" + File.separator + "lib");
            } catch (IOException e) {
                throw new InstallationException("Cannot create Tomcat 4.x specific directories & files.");
            }
        }

        // copy the jars
        if (version.equals(TOMCAT_6_X)) {
            copyDerbyJAR(serverHome + File.separator + File.separator + "lib");
            copyLog4jJAR(serverHome + File.separator + File.separator + "lib");
//            copyToolsJAR(serverHome + File.separator + File.separator + "lib");  //TODO: Some production servers do not run on boxes with JDK only JRE hence do not look for tools jar
        } else {
            copyDerbyJAR(serverHome + File.separator + "common" + File.separator + "lib");
            copyLog4jJAR(serverHome + File.separator + "common" + File.separator + "lib");
//            copyToolsJAR(serverHome + File.separator + "common" + File.separator + "lib"); //TODO: Some production servers do not run on boxes with JDK only JRE hence do not look for tools jar
        }

        if (isServerXMLReplaced) {
            System.out.println("WSO2 WSAS was successfully installed on Tomcat (" +
                               serverHome + ")");
        } else {
            System.out.println("\nWSO2 WSAS installation was successful.\n" +
                               "Please configure your Tomcat server.xml " +
                               "as described in the README.txt file.\n");
        }
    }

    private boolean copyServerXML(String tomcatHome, String version) throws InstallationException {
        boolean serverXMLReplaced = false;
        String ourServerXML = getOurServerXML(version);

        // Set the keystoreFile properly
        try {
            XMLStreamReader parser =
                    XMLInputFactory.newInstance().
                            createXMLStreamReader(new FileReader(ourServerXML));
            OMXMLParserWrapper builder =
                    OMXMLBuilderFactory.createStAXOMBuilder(OMAbstractFactory.getOMFactory(),
                                                            parser);
            OMElement docEle = builder.getDocumentElement();
            OMElement serviceEle =
                    docEle.getFirstChildWithName(new QName("Service"));
            if (version.equals(TOMCAT_4_1_X)) {
                for (Iterator iter = serviceEle.getChildrenWithName(new QName("Connector"));
                     iter.hasNext();) {

                    OMElement omEle = (OMElement) iter.next();
                    OMAttribute schemeAttr = omEle.getAttribute(new QName("scheme"));
                    if (schemeAttr != null &&
                        schemeAttr.getAttributeValue().equalsIgnoreCase("https")) {

                        OMElement factoryEle =
                                omEle.getFirstChildWithName(new QName("Factory"));
                        factoryEle.getAttribute(new QName("keystoreFile")).
                                setAttributeValue(wso2wsasHome + File.separator + "conf" +
                                                  File.separator + "wso2wsas.jks");
                        break;
                    }
                }

                // Set the Database path
                SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();
                XPath xp = new AXIOMXPath("//Service/Engine/Host/Context/ResourceParams/parameter/value[text()='jdbc:derby:${wso2wsas.home}/database/WSO2WSAS_DB']");
                xp.setNamespaceContext(nsCtx);
                OMElement omElement = (OMElement) xp.selectSingleNode(docEle);
                omElement.setText("jdbc:derby:" + wso2wsasHome + "/database/WSO2WSAS_DB");

                // Set the Database path
//                SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();
//                XPath xp =
//                        new AXIOMXPath("//Host/Context/ResourceParams[@name=\"jdbc/wso2wsas_db\"]/parameter/name");
//                xp.setNamespaceContext(nsCtx);
//                OMElement nameEle = (OMElement) xp.selectSingleNode(docEle);
//                nameEle.setText("jdbc:derby:" + wso2wsasHome + "/database/WSO2WSAS_DB");
            } else if (version.equals(TOMCAT_5_5_X) || version.equals(TOMCAT_6_X)) {
                SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();

                // Set the keystore path
                XPath xp1 = new AXIOMXPath("//Service/Connector/@keystoreFile");
                xp1.setNamespaceContext(nsCtx);
                OMAttribute ksFileAttrib = (OMAttribute) xp1.selectSingleNode(docEle);
                ksFileAttrib.setAttributeValue(wso2wsasHome + File.separator + "conf" +
                                               File.separator + "wso2wsas.jks");

                // Set the Database path
                XPath xp = new AXIOMXPath("//Service/Engine/Host/Context/Resource/@url");
                xp.setNamespaceContext(nsCtx);
                OMAttribute attrib = (OMAttribute) xp.selectSingleNode(docEle);
                attrib.setAttributeValue("jdbc:derby:" + wso2wsasHome + "/database/WSO2WSAS_DB");
            } else if (version.equals(TOMCAT_5_0_X)) {
                SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();

                // Set the keystore path
                XPath xp1 = new AXIOMXPath("//Service/Connector/@keystoreFile");
                xp1.setNamespaceContext(nsCtx);
                OMAttribute ksFileAttrib = (OMAttribute) xp1.selectSingleNode(docEle);
                ksFileAttrib.setAttributeValue(wso2wsasHome + File.separator + "conf" +
                                               File.separator + "wso2wsas.jks");

                // Set the Database path
                XPath xp = new AXIOMXPath("//Service/Engine/Host/Context/ResourceParams/parameter/value[text()='jdbc:derby:${wso2wsas.home}/database/WSO2WSAS_DB']");
                xp.setNamespaceContext(nsCtx);
                OMElement omElement = (OMElement) xp.selectSingleNode(docEle);
                omElement.setText("jdbc:derby:" + wso2wsasHome + "/database/WSO2WSAS_DB");
            }

            XMLStreamWriter writer =
                    XMLOutputFactory.newInstance().
                            createXMLStreamWriter(new FileWriter(ourServerXML));
            docEle.serialize(writer);
        } catch (Exception e) {
            String msg = "Could not modify server.xml file: " + e;
            System.err.println(" " + msg);
            throw new InstallationException(msg, e);
        }

        try {
            File tomcatConfDir = new File(tomcatHome + File.separator + "conf");
            System.out.println("Copying \"conf/tomcat/server.xml\" to \"" +
                               tomcatConfDir.getAbsolutePath() + "\"");
            serverXml = tomcatConfDir.getAbsolutePath() + File.separator + "server.xml";
            File tomcatServerXML = new File(serverXml);
            String replace;
            if (tomcatServerXML.exists()) {
                do {
                    System.out.println("\nWSO2 WSAS requires HTTPS to be enabled for the administration" +
                                       " services and the " +
                                       "WSO2 WSAS datasource to be configured for the WSO2 WSAS Database.\n" +
                                       "You can either add this to the " + tomcatConfDir.getAbsolutePath() +
                                       File.separator + "server.xml" +
                                       " yourself," +
                                       " or we provide a default server.xml with HTTPS enabled, " +
                                       "and datasource configured.\n" +
                                       "If you use our server.xml, the installer will backup the" +
                                       " original " + tomcatConfDir.getAbsolutePath() +
                                       File.separator + "server.xml.");
                    System.out.print("Do you want to replace the " +
                                     tomcatConfDir.getAbsolutePath() +
                                     File.separator + "conf" + File.separator +
                                     "server.xml file(y/n)? [n]: ");
                    replace = InputReader.readInput();
                } while (!replace.equalsIgnoreCase("y") && !replace.equalsIgnoreCase("n") &&
                         !(replace.trim().length() == 0));
                if (replace.equalsIgnoreCase("y")) {
                    File origServerXML = new File(tomcatConfDir.getAbsolutePath() +
                                                  File.separator + "server.xml");
                    String backedupServerXML = "server-" + System.currentTimeMillis() + ".xml.bak";
                    fileMan.copyFile(origServerXML, new File(tomcatConfDir.getAbsolutePath() +
                                                             File.separator + backedupServerXML));

                    System.out.println("[INFO] The original " + tomcatConfDir.getAbsolutePath() +
                                       File.separator + "conf" + File.separator + "server.xml " +
                                       "was backed up as " + tomcatConfDir.getAbsolutePath() +
                                       File.separator + "conf" + File.separator + backedupServerXML);
                    tomcatServerXML.delete();

                    fileMan.copyFile(new File(ourServerXML), tomcatServerXML);
                    serverXMLReplaced = true;
                }
            } else {
                serverXMLReplaced = true;
                fileMan.copyFile(new File(ourServerXML), tomcatServerXML);
            }
            System.out.println("OK");
        } catch (IOException e) {
            String msg = "Could not copy Tomcat server.xml file: " + e;
            System.err.println(" " + msg);
            throw new InstallationException(msg, e);
        }
        return serverXMLReplaced;
    }

    private String getOurServerXML(String version) throws InstallationException {
        String ourServerXML;
        if (version.indexOf("tomcat-4.") == 0) {
            ourServerXML = InstallerConstants.APPSERVER_TMP_DIR + File.separator + "tomcat" +
                           File.separator + "4.x" + File.separator + "server.xml";
        } else if (version.indexOf("tomcat-5.5") == 0) {
            ourServerXML = InstallerConstants.APPSERVER_TMP_DIR + File.separator + "tomcat" +
                           File.separator + "5.5.x" + File.separator + "server.xml";
        } else if (version.indexOf("tomcat-5.0") == 0) {
            ourServerXML = InstallerConstants.APPSERVER_TMP_DIR + File.separator + "tomcat" +
                           File.separator + "5.0.x" + File.separator + "server.xml";
        } else if (version.indexOf("tomcat-6") == 0) {
            ourServerXML = InstallerConstants.APPSERVER_TMP_DIR + File.separator + "tomcat" +
                           File.separator + "6.x" + File.separator + "server.xml";
        } else {
            throw new InstallationException("Unsupported Tomcat version");
        }
        return ourServerXML;
    }

    public String getDescription() {
        return "Tomcat Server";
    }

    public String getId() {
        return "1";
    }

    public String[] getSupportedVersions() {
        return new String[]{TOMCAT_4_1_X, TOMCAT_5_0_X, TOMCAT_5_5_X, TOMCAT_6_X, OTHER_VERSION};
    }

    public String[] getServerInfoImplClasses() {
        return new String[]{Tomcat41xServerInfo.class.getName(),
                            Tomcat5xServerInfo.class.getName(),
                            Tomcat5xServerInfo.class.getName(),
                            Tomcat6xServerInfo.class.getName(),
                            GenericServerInfo.class.getName()};
    }

    public String getServerInfoImplClass() {
        return serverInfoImplClass;
    }

    public Properties getServerProperties() {
        Properties props = new Properties();
        String serverXml = this.serverXml;
        if (File.separatorChar == '\\') {
            serverXml = serverXml.replace('\\', '/');
        }
        props.setProperty(ServerPropertyKeys.TOMCAT_SERVER_XML, serverXml);
        return props;
    }
}
