/*                                                                             
 * Copyright 2004,2005 The Apache Software Foundation.                         
 *                                                                             
 * Licensed under the Apache License, Version 2.0 (the "License");             
 * you may not use this file except in compliance with the License.            
 * You may obtain a copy of the License at                                     
 *                                                                             
 *      http://www.apache.org/licenses/LICENSE-2.0                             
 *                                                                             
 * Unless required by applicable law or agreed to in writing, software         
 * distributed under the License is distributed on an "AS IS" BASIS,           
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.    
 * See the License for the specific language governing permissions and         
 * limitations under the License.                                              
 */
package org.wso2.wsas.installer;

import org.wso2.utils.AntBuildException;
import org.wso2.utils.AntBuildInvoker;
import org.wso2.utils.ArchiveManipulator;
import org.wso2.utils.InputReader;

import java.io.File;
import java.io.IOException;

/**
 * This class coordinates the WSAS installation on a servlet container
 */
public class ServletContainerInstallationMode implements InstallationMode {

    private static final TomcatInstaller tomcatInstaller = new TomcatInstaller();
    private static final JBossInstaller jbossInstaller = new JBossInstaller();
    private static final GeronimoInstaller geronimoInstaller = new GeronimoInstaller();
    private static final GenericInstaller genericInstaller = new GenericInstaller();

    /**
     * If we need to support any more app servers out of the box, we need to write an Installable
     * class, and add an instance of it to this array.
     */
    private static final Installable[] INSTALLABLES
            = new Installable[]{ServletContainerInstallationMode.tomcatInstaller,
                                ServletContainerInstallationMode.jbossInstaller,
                                ServletContainerInstallationMode.geronimoInstaller,
                                ServletContainerInstallationMode.genericInstaller};

    private String wsasVersion;
    private String[] envVariables;

    public void setWsasVersion(String wsasVersion) {
        this.wsasVersion = wsasVersion;
    }

    public void setEnvVariables(String[] envVariables) {
        this.envVariables = envVariables;
    }

    public void doInstallation() {
        String wsasHome = InstallationManager.getWSASHome();
        System.setProperty("wso2wsas.home", wsasHome);

        String javaHome = "";
        for (int i = 0; i < envVariables.length; i++) {
            String arg = envVariables[i];
            if (arg.equals("-tomcat_home")) {
                if (++i < envVariables.length) {
                    tomcatInstaller.setServerHome(envVariables[i]);
                }
                continue;
            }

            if (arg.equals("-jboss_home")) {
                if (++i < envVariables.length) {
                    jbossInstaller.setServerHome(envVariables[i]);
                }
                continue;
            }

            if (arg.equals("-geronimo_home")) {
                if (++i < envVariables.length) {
                    geronimoInstaller.setServerHome(envVariables[i]);
                }
                continue;
            }

            if (arg.equals("-java_home")) {
                if (++i < envVariables.length) {
                    javaHome = envVariables[i];
                }
            }
        }

        System.out.println("Using Java Home : " + javaHome + "\n");

        // Extract the wso2wsas-installer.zip
        String installerZip = "wso2wsas-core-" + wsasVersion + "-installer" + ".zip";
        try {
            new ArchiveManipulator().extract("lib" + File.separator + installerZip,
                                             InstallerConstants.TMP_DIR);
        } catch (IOException e) {
            System.err.println("Cannot extract " + installerZip + " " + e);
            e.printStackTrace();
            System.exit(1);
        }

        // Ask for the WSO2 WSAS Home directory
        AntBuildInvoker buildInvoker = new AntBuildInvoker(new File(InstallerConstants.INSTALLER_TMP_DIR +
                                                                    File.separator + "build.xml"));
        try {
            buildInvoker.invokeDefaultTarget(false);
        } catch (AntBuildException e) {
            e.printStackTrace();
            System.err.println("Installation failed " + e);
            System.exit(1);
        }

        try {
            // Ask to select an Application server
            Installable installable = selectAppServer();
            installable.setWSASHome(wsasHome);
            installable.setJavaHome(javaHome);

            // App Server specific installation
            installable.install();

            // Cleanup files created during installation
            buildInvoker.invokeTarget("clean", false);
        } catch (Exception e) {
            System.err.println(" Installation failed : " + e);
            e.printStackTrace();
            System.exit(1);
        }
    }

    public String getDescription() {
        return "Servlet Container Installation";
    }


    public String toString() {
        return getDescription();
    }

    private Installable selectAppServer() throws InstallationException {
        System.out.println("Please select your application server:");
        for (int i = 0; i < INSTALLABLES.length; i++) {
            Installable installable = INSTALLABLES[i];
            System.out.println(installable.getId() + ". " + installable.getDescription());
        }

        boolean foundValidServer = false;
        String serverId;
        Installable installable = null;
        while (!foundValidServer) {
            System.out.print(": ");
            try {
                serverId = InputReader.readInput();
            } catch (IOException e) {
                throw new InstallationException(e);
            }
            for (int i = 0; i < INSTALLABLES.length; i++) {
                if (serverId.equals(INSTALLABLES[i].getId())) {
                    foundValidServer = true;
                    installable = INSTALLABLES[i];
                    break;
                }
            }
        }
        System.out.println();
        System.out.println("Installing on " + installable.getDescription() + "...");
        return installable;
    }
}
