/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.installer;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMXMLParserWrapper;
import org.apache.axiom.om.impl.llom.factory.OMXMLBuilderFactory;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.InputReader;
import org.wso2.wsas.serverinfo.GenericServerInfo;
import org.wso2.wsas.serverinfo.JBossServerInfo;
import org.wso2.wsas.transport.ServerPropertyKeys;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Iterator;
import java.util.Properties;

/**
 *
 */
public class JBossInstaller extends AbstractAppServerInstaller {

    private FileManipulator fileMan = new FileManipulator();
    private String configurationFileSet;
    private String serverInfoImplClass;
    private String tomcatServerXml;

    public void install() throws InstallationException {

        // Get the JBoss version
        System.out.println("Please select your JBoss version : ");
        String[] versions = getSupportedVersions();
        for (int i = 0; i < versions.length; i++) {
            System.out.println((i + 1) + ". " + versions[i]);
        }
        int serverVersionIndex = getServerVersionIndex();
        String version = versions[serverVersionIndex];
        serverInfoImplClass = getServerInfoImplClasses()[serverVersionIndex];
        if (version.equals(OTHER_VERSION)) {
            new GenericInstaller().install();
            return;
        }

        // Capture the JBoss Home
        if (this.serverHome != null && this.serverHome.trim().length() != 0 &&
            !this.serverHome.equals("INVALID")) {
            System.out.print("Please enter your JBoss Home directory [" +
                             this.serverHome + "]: ");
        } else {
            System.out.print("Please enter your JBoss Home directory : ");
        }
        serverHome = getAppServerHome();

        File jbossHomeDir = new File(serverHome);
        if (!jbossHomeDir.exists()) {
            String msg = "JBoss home " + serverHome + " does not exist! Aborting installation.";
            System.err.println(" " + msg);
            throw new InstallationException(msg);
        }

        if (serverHome.toLowerCase().indexOf("jboss") == -1) {
            System.out.print("[WARN] " + serverHome +
                             " does not look like a JBoss home directory.");
            System.out.print(" Do you want to continue (y/n)? [y]: ");
            String response;
            do {
                try {
                    response = InputReader.readInput();
                } catch (IOException e) {
                    throw new InstallationException(e);
                }
            }
            while (!response.equalsIgnoreCase("y") && !response.equalsIgnoreCase("n") &&
                   !(response.trim().length() == 0));
            if (response.equalsIgnoreCase("n")) {
                System.out.println("[INFO] Installation aborted.");
                return;
            }
        }

        // get the JBoss configuration file set
        String jbossServerDirName = serverHome + File.separator + "server";
        String[] configFileSets = new File(jbossServerDirName).list();

        if (configFileSets.length == 0) {
            System.out.println(" No configuration file sets found!");
            System.exit(1);
        }
        System.out.println("Please select the configuration file set you wish to use:");
        for (int i = 0; i < configFileSets.length; i++) {
            System.out.println((i + 1) + ". " + configFileSets[i]);
        }
        int configFileSetIndex = -1;
        do {
            System.out.print(": ");
            try {
                configFileSetIndex = Integer.parseInt(InputReader.readInput()) - 1;
            } catch (IOException e) {
                throw new InstallationException(e);
            } catch (NumberFormatException e) {      // should continue the loop
            }
        }
        while (configFileSetIndex == -1 || configFileSetIndex >= configFileSets.length);

        this.configurationFileSet = configFileSets[configFileSetIndex];
        String jbossDeploymentDir = serverHome + File.separator +
                                    "server" + File.separator + this.configurationFileSet +
                                    File.separator + "deploy";
        File jbossDeployDir = new File(jbossDeploymentDir);
        while (!jbossDeployDir.exists()) {
            System.out.println("[WARN] " + jbossDeployDir.getAbsolutePath() +
                               " does not exist.");
            System.out.print("Please specify the deployment directory : ");
            jbossDeploymentDir = "";
            while (jbossDeploymentDir == null || jbossDeploymentDir.trim().length() == 0) {
                try {
                    jbossDeploymentDir = InputReader.readInput();
                    jbossDeployDir = new File(jbossDeploymentDir);
                } catch (IOException e) {
                    throw new InstallationException(e);
                }
            }
        }

        // copy the wso2wsas-derby-ds.xml
        copyDerbyDsXml(jbossDeploymentDir);

        // copy the server.xml
        boolean isServerXMLReplaced = copyServerXML(serverHome);

        // Re-archive the wso2wsas.war and copy it to the webapps dir
        createWebArchive();
        copyWsasWAR(jbossDeploymentDir);

        // copy the derby.jar
        copyDerbyJAR(jbossDeploymentDir + File.separator + ".." + File.separator + "lib");

        if (isServerXMLReplaced) {
            System.out.println("WSO2 WSAS was successfully installed on JBoss (" +
                               serverHome + ")");
        } else {
            System.out.println("\nWSO2 WSAS installation was successful.\n" +
                               "Please configure your JBoss server.xml " +
                               "as described in the README.txt file.\n");
        }
    }

    private boolean copyServerXML(String jbossHome) throws InstallationException {
        boolean serverXMLReplaced = false;
        try {
            // Find the tomcat dir
            File deployDir = new File(jbossHome +
                                      File.separator + "server" +
                                      File.separator + this.configurationFileSet +
                                      File.separator + "deploy");
            File tomcatConfDir = null;
            String[] dirs = deployDir.list();
            boolean foundTomcat = false;
            for (int i = 0; i < dirs.length; i++) {
                String dir = dirs[i];
                if (dir.startsWith("jbossweb-tomcat") && dir.endsWith(".sar")) {
                    foundTomcat = true;
                    tomcatConfDir = new File(deployDir + File.separator + dir);
                    break;
                }
            }

            if (!foundTomcat) {
                throw new InstallationException("Tomcat not found");
            }

            System.out.println("Copying \"server.xml\" to \"" + tomcatConfDir.getAbsolutePath() + "\"");
            tomcatServerXml = tomcatConfDir.getAbsolutePath() +
                              File.separator + "server.xml";
            File tomcatServerXML = new File(tomcatServerXml);
            String delete;
            if (tomcatServerXML.exists()) {
                do {
                    System.out.println("\nWSO2 WSAS requires HTTPS to be enabled for the administration" +
                                       " services and the " +
                                       "WSO2 WSAS datasource to be configured for the WSO2 WSAS Database.\n" +
                                       "You can either add this to the " + tomcatServerXML.getAbsolutePath() +
                                       " yourself," +
                                       " \nor we provide a default server.xml with HTTPS enabled, " +
                                       "and datasource configured.\n" +
                                       "If you use our server.xml, the installer will backup the" +
                                       " original " + tomcatServerXML.getAbsolutePath());
                    System.out.print("Do you want to replace the " +
                                     tomcatServerXML.getAbsolutePath() +
                                     " file(y/n)? [n]: ");
                    delete = InputReader.readInput();
                }
                while (!delete.equalsIgnoreCase("y") && !delete.equalsIgnoreCase("n") &&
                       !(delete.trim().length() == 0));
                if (delete.equalsIgnoreCase("y")) {
                    File origServerXML = new File(tomcatConfDir.getAbsolutePath() +
                                                  File.separator + "server.xml");
                    String backedupServerXML = "server-" + System.currentTimeMillis() + ".xml.bak";
                    fileMan.copyFile(origServerXML, new File(tomcatConfDir.getAbsolutePath() +
                                                             File.separator + backedupServerXML));

                    System.out.println("[INFO] The original " + tomcatServerXML.getAbsolutePath() +
                                       " was backed up as " + tomcatConfDir.getAbsolutePath() +
                                       File.separator + backedupServerXML);
                    tomcatServerXML.delete();

                    String ourServerXML = InstallerConstants.APPSERVER_TMP_DIR + File.separator
                                          + "jboss" + File.separator + "server.xml";

                    // Set the keystoreFile properly
                    try {
                        XMLStreamReader parser =
                                XMLInputFactory.newInstance().
                                        createXMLStreamReader(new FileReader(ourServerXML));
                        OMXMLParserWrapper builder =
                                OMXMLBuilderFactory.
                                        createStAXOMBuilder(OMAbstractFactory.getOMFactory(),
                                                            parser);
                        OMElement docEle = builder.getDocumentElement();
                        OMElement serviceEle =
                                docEle.getFirstChildWithName(new QName("Service"));

                        for (Iterator iter = serviceEle.getChildrenWithName(new QName("Connector"));
                             iter.hasNext();) {

                            OMElement omEle = (OMElement) iter.next();
                            OMAttribute ksfAttr = omEle.getAttribute(new QName("keystoreFile"));
                            if (ksfAttr != null) {
                                ksfAttr.setAttributeValue(wso2wsasHome + File.separator + "conf" +
                                                          File.separator + "wso2wsas.jks");

                                XMLStreamWriter writer =
                                        XMLOutputFactory.newInstance().
                                                createXMLStreamWriter(new FileWriter(ourServerXML));
                                docEle.serialize(writer);
                                break;
                            }
                        }
                    } catch (XMLStreamException e) {
                        String msg = "Could not modify server.xml file: " + e;
                        System.err.println(" " + msg);
                        throw new InstallationException(msg, e);
                    }

                    fileMan.copyFile(new File(ourServerXML), tomcatServerXML);
                    serverXMLReplaced = true;
                }
            } else {
                serverXMLReplaced = true;
                fileMan.copyFile(new File(InstallerConstants.APPSERVER_TMP_DIR + File.separator +
                                          "jboss" + File.separator + "server.xml"),
                                 tomcatServerXML);
            }
            System.out.println("OK");
        } catch (IOException e) {
            String msg = "Could not copy server.xml file: " + e;
            System.err.println(" " + msg);
            throw new InstallationException(msg, e);
        }
        return serverXMLReplaced;
    }

    private void copyDerbyDsXml(String jbossDeploymentDir) throws InstallationException {
        try {
            File dst = new File(jbossDeploymentDir + File.separator + "wso2wsas-derby-ds.xml");
            System.out.println("Copying \"wso2wsas-derby-ds.xml\" to \"" +
                               dst.getAbsolutePath() + "\"");
            fileMan.copyFile(new File(InstallerConstants.APPSERVER_TMP_DIR + File.separator +
                                      "jboss" + File.separator + "wso2wsas-derby-ds.xml"),
                             dst);
            System.out.println("OK");
        } catch (IOException e) {
            String msg = "Could not copy wso2wsas-derby-ds.xml : " + e;
            System.err.println(" " + msg);
            throw new InstallationException(msg, e);
        }
    }

    public String getDescription() {
        return "JBoss Server";
    }

    public String getId() {
        return "2";
    }

    public String[] getSupportedVersions() {
        return new String[]{"jboss-3.2.x", "jboss-4.0.x", OTHER_VERSION};
    }

    public String[] getServerInfoImplClasses() {
        return new String[]{JBossServerInfo.class.getName(),
                            JBossServerInfo.class.getName(),
                            GenericServerInfo.class.getName()};
    }

    public String getServerInfoImplClass() {
        return serverInfoImplClass;
    }

    public Properties getServerProperties() {
        Properties props = new Properties();
        props.setProperty(ServerPropertyKeys.JBOSS_CONFIG_FILE_SET, this.configurationFileSet);
        String tomcatServerXml = this.tomcatServerXml;
        if (File.separatorChar == '\\') {
            tomcatServerXml = tomcatServerXml.replace('\\', '/');
        }
        props.setProperty(ServerPropertyKeys.TOMCAT_SERVER_XML, tomcatServerXml);
        return props;
    }
}
