/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.installer;

import org.wso2.utils.InputReader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;


/**
 * Manages the installation process of the WSO2 WSAS Embeddable distribution on different
 * application servers.
 */
public class InstallationManager {

    private static final ServletContainerInstallationMode scInstMode =
            new ServletContainerInstallationMode();
    private static final EclipseWtpPluginInstalltionMode eclipsePluginInstMode =
            new EclipseWtpPluginInstalltionMode();
    private static final WindowsNtServiceInstallation ntServiceInstMode =
            new WindowsNtServiceInstallation();
    private static final WindowsNtServiceUninstallation ntServiceUninstMode =
            new WindowsNtServiceUninstallation();

    /**
     * The supported installation modes. If we need to support any new installation modes, we have
     * to add an entry here.
     */
    private final InstallationMode[] INSTALLATION_MODES;

    public InstallationManager() {
        String osName = System.getProperty("os.name");
        if (osName.toLowerCase().indexOf("win") != -1) {
            INSTALLATION_MODES =
                    new InstallationMode[]{eclipsePluginInstMode, scInstMode,
                                           ntServiceInstMode, ntServiceUninstMode};
        } else {
            INSTALLATION_MODES =
                    new InstallationMode[]{eclipsePluginInstMode, scInstMode};
        }
    }

    private void doInstallation(String[] params) {
        Properties props = new Properties();
        try {
            props.load(new FileInputStream("bin" + File.separator + "version.txt"));
        } catch (IOException e) {
            System.err.println("Cannot load version.txt file " + e);
            e.printStackTrace();
            System.exit(1);
        }

        String wsasVersion = props.getProperty("version");

        System.out.println("\n###########################################################");
        System.out.println("#                                                         #");
        System.out.println("#               WSO2 WSAS v" + wsasVersion + " Installation               #");
        System.out.println("#                                                         #");
        System.out.println("###########################################################\n");

        scInstMode.setEnvVariables(params);
        scInstMode.setWsasVersion(wsasVersion);

        try {
            int installationMode = getInstallationMode();
            INSTALLATION_MODES[installationMode].doInstallation();
        } catch (InstallationException e) {
            System.err.println(" Installation failed : " + e);
            e.printStackTrace();
            System.exit(1);
        }
    }

    public static void main(String[] args) {
        new InstallationManager().doInstallation(args);
    }

    public static String getWSASHome() {
        File defaultWSASHome = new File(System.getProperty("user.home") +
                                        File.separator + ".wso2wsas");
        System.out.println("WSO2 WSAS Home directory is where the WSO2 WSAS database, conf files," +
                           " logs, and module & service repository will reside.");
        String wsasHome = defaultWSASHome.getAbsolutePath();
        do {
            System.out.print("Please enter the WSO2 WSAS Home directory [" +
                             defaultWSASHome.getAbsolutePath() + "]:");

            try {
                wsasHome = InputReader.readInput();
                if (wsasHome == null || wsasHome.trim().length() == 0) {
                    wsasHome = defaultWSASHome.getAbsolutePath();
                } else if (wsasHome.startsWith("~")) {
                    System.err.println("Please specify the path properly. The '~' character is not supported");
                    continue;
                } else if (new File(wsasHome).getCanonicalPath().
                        equals(new File(".").getCanonicalPath())) {
                    System.err.println("You cannot install WSO2 WSAS into this directory. " +
                                       "Please specify another directory.");
                    continue;
                }
            } catch (IOException e) {
                System.err.println(" Could not read WSO2 WSAS Home : " + e);
                System.exit(1);
            }
            File file = new File(wsasHome);
            if (file.isFile()) {
                System.err.println("WSO2 WSAS Home " + wsasHome + " is a file not a directory!");
            } else {
                break;
            }
        } while (true);
        return wsasHome.trim();
    }

    private int getInstallationMode() throws InstallationException {
        System.out.println("\nPlease select your installation mode: ");
        for (int i = 0; i < INSTALLATION_MODES.length; i++) {
            System.out.println(((i + 1) + ") ") +
                               INSTALLATION_MODES[i].getDescription());
        }

        int modeId = 0;
        while (true) {
            System.out.print(": ");
            try {
                modeId = Integer.parseInt(InputReader.readInput());
            } catch (Exception e) {
                System.err.println("Please enter a valid mode");
            }
            if (modeId > 0 && modeId <= INSTALLATION_MODES.length) {
                break;
            } else {
                System.err.println("Please enter a valid mode");
            }
        }
        System.out.println();
        System.out.println("Selection: " + INSTALLATION_MODES[modeId - 1]);
        return modeId - 1;
    }
}
