/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.installer;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.jaxen.JaxenException;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.XPath;
import org.wso2.utils.InputReader;
import org.wso2.wsas.serverinfo.GenericServerInfo;
import org.wso2.wsas.serverinfo.GeronimoServerInfo;
import org.wso2.wsas.transport.ServerPropertyKeys;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Properties;

/**
 *
 */
public class GeronimoInstaller extends AbstractAppServerInstaller {

    private String configXml;
    private static final String GERONIMO_PLAN_NAMESPACE =
            "http://geronimo.apache.org/xml/ns/j2ee/connector-1.1";

    public void install() throws InstallationException {

        // Get the Geronimo version
        System.out.println("Please select your Geronimo version : ");
        String[] versions = getSupportedVersions();
        for (int i = 0; i < versions.length; i++) {
            System.out.println((i + 1) + ". " + versions[i]);
        }
        int serverVersionIndex = getServerVersionIndex();
        String version = versions[serverVersionIndex];
        if (version.equals(OTHER_VERSION)) {
            new GenericInstaller().install();
            return;
        }

        // Capture the Geronimo Home
        if (this.serverHome != null && this.serverHome.trim().length() != 0 &&
            !this.serverHome.equals("INVALID")) {
            System.out.print("Please enter your Geronimo Home directory [" +
                             this.serverHome + "]: ");
        } else {
            System.out.print("Please enter your Geronimo Home directory : ");
        }
        serverHome = getAppServerHome();

        File geronimoHomeDir = new File(serverHome);
        if (!geronimoHomeDir.exists()) {
            String msg = "Geronimo home " + serverHome + " does not exist! Aborting installation.";
            System.err.println(" " + msg);
            throw new InstallationException(msg);
        }

        if (serverHome.toLowerCase().indexOf("geronimo") == -1) {
            System.out.print("[WARN] " + serverHome +
                             " does not look like a Geronimo home directory.");
            System.out.print(" Do you want to continue (y/n)? [y]: ");
            String response;
            do {
                try {
                    response = InputReader.readInput();
                } catch (IOException e) {
                    throw new InstallationException(e);
                }
            } while (!response.equalsIgnoreCase("y") && !response.equalsIgnoreCase("n") &&
                     !(response.trim().length() == 0));
            if (response.equalsIgnoreCase("n")) {
                System.out.println("[INFO] Installation aborted.");
                return;
            }
        }

        this.configXml = geronimoHomeDir.getAbsolutePath() + File.separator +
                         "var" + File.separator + "config" + File.separator + "config.xml";

        // Rearchive the wso2wsas.war and copy it to the webapps dir
        createWebArchive();

        // Deploy the plan.xml & wso2wsas.war
        try {
            String os = System.getProperty("os.version");
            System.out.print("Going to run Geronimo deployer tool.\n" +
                             "Please make sure the Geronimo server is running & press enter");
            InputReader.readInput();
            System.out.println();

            // Read the geronimo admin username & password
            String geronimoUsername;
            do {
                System.out.print("Geronimo Admin Username :");
                geronimoUsername = InputReader.readInput();
            } while (geronimoUsername == null || geronimoUsername.trim().length() == 0);

            String geronimoPassword;
            do {
                geronimoPassword = InputReader.readPassword("Geronimo Admin User Password :");
            } while (geronimoPassword == null || geronimoPassword.length() == 0);

            String script;
            if (os.toLowerCase().indexOf("win") != -1) {
                script = "deploy.bat";
            } else {
                script = "deploy.sh";
            }

            // modify the plan XML to contain the wso2wsas.home
            String planXml = InstallerConstants.APPSERVER_TMP_DIR + File.separator + "geronimo" +
                             File.separator + "wso2wsas-geronimo-plan.xml";
            updatePlanXml(planXml);

            System.out.println("Deploying the plan XML file");
            String deployPlanCmd = geronimoHomeDir.getAbsolutePath() + File.separator +
                                   "bin" + File.separator + script + " --user " +
                                   geronimoUsername + " --password " + geronimoPassword +
                                   " deploy " + planXml + " " + geronimoHomeDir.getAbsolutePath() + File.separator +
                                   "repository" + File.separator + "tranql" + File.separator + "tranql-connector" +
                                   File.separator + "1.2" + File.separator + "tranql-connector-1.2.rar";

            Process deployPlanProcess = Runtime.getRuntime().exec(deployPlanCmd);
            printOutput(deployPlanProcess);
            if (deployPlanProcess.waitFor() != 0) {
                throw new InstallationException("Geronimo deployer failed");
            }
            System.out.println("OK");
            System.out.println("Deploying the wso2wsas.war file");
            String deployWarCmd = geronimoHomeDir.getAbsolutePath() + File.separator +
                                  "bin" + File.separator + script + " --user " +
                                  geronimoUsername + " --password " + geronimoPassword +
                                  " deploy " + InstallerConstants.WSO2WSAS_WAR;
            Process deployWarProcess = Runtime.getRuntime().exec(deployWarCmd);
            printOutput(deployWarProcess);
            if (deployWarProcess.waitFor() != 0) {
                throw new InstallationException("Geronimo deployer failed");
            }
            System.out.println("OK");
        } catch (Exception e) {
            e.printStackTrace();
            throw new InstallationException(" Cannot run Geronimo deployer. \n" +
                                            "Please make sure that the Geronimo server is running.",
                                            e);
        }
        System.out.println("WSO2 WSAS was successfully installed on Geronimo (" + serverHome + ")");
    }

    private void updatePlanXml(String planXml) throws XMLStreamException,
                                                      JaxenException,
                                                      IOException {
        XMLStreamReader parser = XMLInputFactory.newInstance().
                createXMLStreamReader(new FileInputStream(new File(planXml)));
        StAXOMBuilder staxOMBuilder = new StAXOMBuilder(parser);
        OMElement docEle = staxOMBuilder.getDocumentElement();
        OMElement omElement = getOMElement("//ns:config-property-setting[@name='ConnectionURL']",
                                           docEle);
        omElement.setText("jdbc:derby:" + wso2wsasHome + "/database/WSO2WSAS_DB");
        XMLStreamWriter writer =
                XMLOutputFactory.newInstance().
                        createXMLStreamWriter(new FileWriter(planXml));
        docEle.serialize(writer);
        writer.flush();
    }

    private void printOutput(Process process) throws IOException {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String op;
            while ((op = reader.readLine()) != null) {
                System.err.println(op);
            }
        } finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    public String getDescription() {
        return "Geronimo Server";
    }

    public String getId() {
        return "3";
    }

    public String[] getServerInfoImplClasses() {
        return new String[]{GeronimoServerInfo.class.getName(),
                            GenericServerInfo.class.getName(),
                            GenericServerInfo.class.getName()};
    }

    public String[] getSupportedVersions() {
        return new String[]{"geronimo-tomcat-1.1.1", "geronimo-jetty-1.1.1", OTHER_VERSION};
    }

    public String getServerInfoImplClass() {
        return GeronimoServerInfo.class.getName();
    }

    public Properties getServerProperties() {
        Properties props = new Properties();
        String configXml = this.configXml;
        if (File.separatorChar == '\\') {
            configXml = configXml.replace('\\', '/');
        }
        props.setProperty(ServerPropertyKeys.GERONIMO_CONFIG_XML, configXml);
        props.setProperty(ServerPropertyKeys.APPSERVER_VERSION, "1.1.1"); //TODO: remove hard coding
        return props;
    }

    private OMElement getOMElement(String xpathString,
                                   OMElement parentElement) throws JaxenException {
        XPath xpath = getXPath(xpathString);
        return (OMElement) xpath.selectSingleNode(parentElement);
    }

    private XPath getXPath(String xpathString) throws JaxenException {
        SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();
        nsCtx.addNamespace("ns", GERONIMO_PLAN_NAMESPACE);
        XPath xpath = new AXIOMXPath(xpathString);
        xpath.setNamespaceContext(nsCtx);
        return xpath;
    }
}
