/*                                                                             
 * Copyright 2004,2005 The Apache Software Foundation.                         
 *                                                                             
 * Licensed under the Apache License, Version 2.0 (the "License");             
 * you may not use this file except in compliance with the License.            
 * You may obtain a copy of the License at                                     
 *                                                                             
 *      http://www.apache.org/licenses/LICENSE-2.0                             
 *                                                                             
 * Unless required by applicable law or agreed to in writing, software         
 * distributed under the License is distributed on an "AS IS" BASIS,           
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.    
 * See the License for the specific language governing permissions and         
 * limitations under the License.                                              
 */
package org.wso2.wsas.installer;

import org.wso2.utils.FileManipulator;
import org.wso2.utils.InputReader;

import java.io.File;
import java.io.IOException;

/**
 * This class coordinates the WSAS installation as an Eclipse WTP Plugin
 */
public class EclipseWtpPluginInstalltionMode implements InstallationMode {
    private static FileManipulator fileMan = new FileManipulator();

    public void doInstallation() throws InstallationException {

        System.out.println("\nStarting WSAS Eclipse WTP Plugin installation...");

        // Get the $WTP_HOME dir
        String wtpHomeDir = System.getProperty("WTP_HOME");
        System.out.println("Please shutdown the Eclipse instance, If Already Running ...");
        while (wtpHomeDir == null || wtpHomeDir.trim().length() == 0) {
            System.out.print("Please enter Eclipse WTP Home : ");
            try {
                wtpHomeDir = InputReader.readInput();
            } catch (IOException e) {
                String msg = "Could not read Eclipse WTP Home";
                System.err.println(" " + msg);
                throw new InstallationException(msg, e);
            }
        }
        File wtpDir = new File(wtpHomeDir);
        if (!wtpDir.exists()) {
            String msg = wtpHomeDir + " does not exist !! ";
            System.out.print(" " + msg);
        } else {
            File destination = new File(wtpHomeDir + File.separator + "plugins");
            if (validateEclipseWTPInstallation(destination)) {
                System.out.println("Using Eclipse WTP HOME [" + wtpHomeDir + "]");
                try {
                    System.out.println("Copying WSAS Eclipse WTP Plugins " + wtpDir.getAbsolutePath()
                                       + File.separator + "plugins");
                    //Delete the Existing Axis2 Plugins
                    //Fill in the Axis2 WTP Plugin Locations
                    File[] matchingFiles = fileMan.getMatchingFiles(destination.getAbsolutePath(),
                                                                    "org.eclipse.jst.ws.axis2", "jar");
                    for (int i = 0; i < matchingFiles.length; i++) {
                        File file = matchingFiles[i];
                        if (file.exists()) {
                            file.delete();
                        }
                    }

                    //Check for the previous WSF plugins existance
                    File[] matchingExistingWSFFiles = fileMan.getMatchingFiles(destination.getAbsolutePath(),
                                                                               "org.wso2.wsf", "jar");
                    if (matchingExistingWSFFiles.length > 0) {
                        String userInput;
                        //Get the user confirmation to uninstall the previous version
                        do {
                            System.out.print("Previous version of WSAS Eclipse Plugins Exists,\n" +
                                             " Uninstall previous version and install new version(y/n)? [y]: ");
                            userInput = InputReader.readInput();
                        }
                        while (!userInput.equalsIgnoreCase("y") && !userInput.equalsIgnoreCase("n") &&
                               !(userInput.trim().length() == 0));
                        if (userInput.equalsIgnoreCase("y") || userInput.trim().length() == 0) {
                            System.out.print(" Uninstalling the previous version of WSAS Eclipse Plugins ...");
                            for (int i = 0; i < matchingExistingWSFFiles.length; i++) {
                                File file = matchingExistingWSFFiles[i];
                                if (file.exists()) {
                                    file.delete();
                                }
                            }
                            installWSASEclipsePlugins(".", destination);
                        } else if (userInput.equalsIgnoreCase("n")) {
                            System.out.println("\nWSO2 WSAS Eclipse WTP Plugin installation canceled by user ...");
                        }
                    } else {
                        //No pervious plugins installed, go ahead with normal installation
                        installWSASEclipsePlugins(".", destination);
                    }
                } catch (IOException e) {
                    String msg = "Could not copy WSAS Eclipse WTP Plugins: " + e;
                    System.err.println(" " + msg);
                    throw new InstallationException(msg);
                }
            } else {
                System.err.println("\nWSO2 WSAS Eclipse WTP Plugin installation failed.\n" +
                                   "Please select a valid Eclipse WTP Installation. \n");
            }
        }
    }

    /**
     * Validates the given Eclipse installation for its WTP Plugins(JST,WST) existance
     *
     * @param wtpPluginDir : Eclipse WTP Installation Directory
     * @return : result of the basic validation of the WTP Plugins
     */
    private static boolean validateEclipseWTPInstallation(File wtpPluginDir) {
        File[] matchingJSTFiles = fileMan.getMatchingFiles(wtpPluginDir.getAbsolutePath(),
                                                           "org.eclipse.jst", "jar");
        File[] matchingWSTFiles = fileMan.getMatchingFiles(wtpPluginDir.getAbsolutePath(),
                                                           "org.eclipse.wst", "jar");
        return matchingJSTFiles.length > 0 && matchingWSTFiles.length > 0;
    }

    /**
     * Install the WSAS Eclipse Plugins
     *
     * @param wsasHome    The WSO2 WSAS home directory
     * @param destination The directory to which the files need to be copied
     * @throws IOException If file copying fails
     */
    private static void installWSASEclipsePlugins(String wsasHome,
                                                  File destination) throws IOException {
        //Adding the WSAS Plugins
        File wsfDir = new File(wsasHome + File.separator + "lib"
                               + File.separator + "wsf");
        //Fill in the WTP Plugin Locations
        File[] installableWSFPlugins = fileMan.getMatchingFiles(wsfDir.getAbsolutePath(),
                                                                "org.wso2.wsf", "jar");
        String delete;
        for (int i = 0; i < installableWSFPlugins.length; i++) {
            File checkFile = new File(destination + File.separator + installableWSFPlugins[i].getName());
            if (checkFile.exists()) {
                do {
                    System.out.print(checkFile.getAbsolutePath() +
                                     " already exists. Do you want to replace this file(y/n)? [y]: ");
                    delete = InputReader.readInput();
                }
                while (!delete.equalsIgnoreCase("y") && !delete.equalsIgnoreCase("n") &&
                       !(delete.trim().length() == 0));
                if (delete.equalsIgnoreCase("y") || delete.trim().length() == 0) {
                    checkFile.delete();
                }
            }
            fileMan.copyFile(installableWSFPlugins[i], checkFile);
        }
        System.out.println("OK");
        System.out.println("\nWSO2 WSAS Eclipse WTP Plugin installation was successful.\n" +
                           "Please restart Eclipse WTP Instance.. \n");
    }

    public String getDescription() {
        return "Eclipse WTP Plugin Installation";
    }

    public String toString() {
        return getDescription();
    }
}
