/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.installer;

import org.wso2.utils.ArchiveManipulator;
import org.wso2.wsas.transport.ServerPropertyKeys;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Iterator;
import java.util.Properties;

/**
 *
 */
public abstract class AbstractInstaller implements Installable {

    protected String serverHome;
    protected String wso2wsasHome;
    protected String javaHome;

    public void setServerHome(String serverHome) {
        this.serverHome = serverHome;
    }

    public String getServerHome() {
        return this.serverHome;
    }

    public String getWSASHome() {
        return this.wso2wsasHome;
    }

    public String setJavaHome(String javaHome) {
        return this.javaHome = javaHome;
    }

    public void setWSASHome(String wsasHome) {
        this.wso2wsasHome = wsasHome;
    }

    protected String getItem(String sourceDir, String itemPrefix, String extension) {
        File libDir = new File(sourceDir);
        String[] items = libDir.list();
        for (int i = 0; i < items.length; i++) {
            String item = items[i];
            if (item.startsWith(itemPrefix) && item.endsWith(extension)) {
                return item;
            }
        }
        return sourceDir;
    }

    protected void createWebArchive() throws InstallationException {
        try {
            // Archive the extracted wso2wsas.war
            System.out.println("Serializing " + InstallerConstants.WSO2WSAS_PROPERTIES +
                               " file ...");
            serializeWsasProperties(wso2wsasHome);
            System.out.println("OK");
            System.out.println("Creating Web Archive...");
            new ArchiveManipulator().archiveDir(InstallerConstants.WSO2WSAS_WAR,
                                                InstallerConstants.TMP_DIR + File.separator +
                                                "lib" + File.separator + "wso2wsas");
            System.out.println("OK");
        } catch (IOException e) {
            throw new InstallationException(e);
        }
    }

    private void serializeWsasProperties(String wso2wsasHome) throws InstallationException {
        System.out.println("Creating " + InstallerConstants.WSO2WSAS_PROPERTIES + " file...");
        File tmpDir = new File(InstallerConstants.TMP_DIR + File.separator + "lib" + File.separator +
                               "wso2wsas" + File.separator + "WEB-INF" + File.separator + "classes");
        if (!tmpDir.exists()) {
            tmpDir.mkdirs();
        }
        File wso2wsasPropFile = new File(tmpDir.getAbsolutePath() +
                                         File.separator + InstallerConstants.WSO2WSAS_PROPERTIES);

        BufferedWriter writer = null;
        try {
            if (wso2wsasPropFile.exists()) {
                wso2wsasPropFile.delete();
            }
            writer = new BufferedWriter(new FileWriter(wso2wsasPropFile));
            if (File.separatorChar == '\\') {
                wso2wsasHome = wso2wsasHome.replace(File.separatorChar, '/');
            }
            writer.write(ServerPropertyKeys.WSO2WSAS_HOME + "=" + wso2wsasHome + "\n");
            writer.write(ServerPropertyKeys.APPSERVER_INFO_IMPL + "=" +
                         getServerInfoImplClass() + "\n");
            Properties serverProps = getServerProperties();
            if (serverProps != null) {
                for (Iterator iter = serverProps.keySet().iterator(); iter.hasNext();) {
                    String key = (String) iter.next();
                    writer.write(key + "=" + serverProps.getProperty(key) + "\n");
                }
            }
            writer.flush();
        } catch (IOException e) {
            String msg = " Could not create " + InstallerConstants.WSO2WSAS_PROPERTIES +
                         " file :" + e;
            throw new InstallationException(msg);
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    System.out.println("[WARN] Could not close BufferedWriter : " + e);
                }
            }
        }
    }
}
