/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.deployment;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.AxisFault;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.engine.AxisEvent;
import org.apache.axis2.engine.AxisObserver;
import org.apache.axis2.rpc.receivers.ejb.EJBUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.rahas.impl.SAMLTokenIssuerConfig;
import org.apache.rahas.impl.TokenIssuerUtil;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.exception.EJBConfigurationNotFoundException;
import org.wso2.wsas.persistence.exception.ServiceGroupNotFoundException;
import org.wso2.wsas.persistence.exception.ServiceNotFoundException;
import org.wso2.wsas.security.ServerCrypto;
import org.wso2.wsas.security.util.RampartConfigUtil;
import org.wso2.wsas.util.SystemFilter;
import org.wso2.wsas.util.WsasUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * This deployment interceptor will be called whenever before a module is initialized or service is
 * deployed.
 *
 * @see AxisObserver
 */
public class DeploymentInterceptor implements AxisObserver {
    private static final Log log = LogFactory.getLog(DeploymentInterceptor.class);

    private final Map paramMap = new HashMap();
    private PersistenceManager persistenceMgr;
    private AxisConfiguration axisConfig;

    public void init(AxisConfiguration axisConfig) {
        persistenceMgr = new PersistenceManager();
        this.axisConfig = axisConfig;
    }

    public void serviceGroupUpdate(AxisEvent axisEvent, AxisServiceGroup axisServiceGroup) {
        // We do not persist Admin service events
        if (SystemFilter.isFilteredOutService(axisServiceGroup.getServiceGroupName())) {
            return;
        }

        int eventType = axisEvent.getEventType();
        if (eventType == AxisEvent.SERVICE_DEPLOY) {
            ServiceGroupDO sgDO =
                    persistenceMgr.getServiceGroup(axisServiceGroup.getServiceGroupName());
            if (sgDO == null) {
                addServiceGroup(axisServiceGroup);
            } else {

                // Check whether the artifact has been updated, if so we need to purge all
                // database entries and treat this as a new service group addition
                if (!sgDO.getServiceArtifactUpdateTime().
                        equals(WsasUtils.lastUpdatedTime(axisServiceGroup))) {

                    log.warn("The service artifact of the " +
                             axisServiceGroup.getServiceGroupName() +
                             " service group has changed. Removing all database entries and " +
                             "handling this as a new service addition.");
                    deleteServiceGroup(axisServiceGroup);
                    addServiceGroup(axisServiceGroup);
                } else {
                    try {
                        persistenceMgr.handleExistingServiceGroupInit(sgDO, axisServiceGroup);
                    } catch (Exception e) {
                        String msg = "Could not handle initialization of existing service group [" +
                                     axisServiceGroup.getServiceGroupName() + "]";
                        log.error(msg, e);
                    }
                }
            }
        } else if (eventType == AxisEvent.SERVICE_REMOVE) {
            deleteServiceGroup(axisServiceGroup);
        }
    }

    private void addServiceGroup(AxisServiceGroup axisServiceGroup) {
        try {
            persistenceMgr.handleNewServiceGroupAddition(axisServiceGroup);
        } catch (Exception e) {
            String msg = "Could not handle initialization of new service group [" +
                         axisServiceGroup.getServiceGroupName() + "]";
            log.error(msg, e);
        }
    }

    private void deleteServiceGroup(AxisServiceGroup axisServiceGroup) {
        try {
            persistenceMgr.deleteServiceGroup(axisServiceGroup);
        } catch (ServiceGroupNotFoundException e) {
            log.error("Service group not found", e);
        }
    }


    public void serviceUpdate(AxisEvent axisEvent, AxisService axisService) {
        // We do not persist Admin service events
        if (SystemFilter.
                isFilteredOutService(((AxisServiceGroup) axisService.getParent()).getServiceGroupName())) {
            return;
        }

        if (axisService.isClientSide()) {
            return;
        }
        //TODO: For the moment, there is no serviceDO version handling facility. hence we'll use
        // EMPTY_SERVICE_VERSION for all services
        int eventType = axisEvent.getEventType();

        String serviceName = axisService.getName();
        String serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        ServiceDO serviceDO = persistenceMgr.getService(serviceName, serviceVersion);

        // if (eventType == AxisEvent.SERVICE_STOP) do nothing

        if (eventType == AxisEvent.SERVICE_START) {
            ServiceDO service = persistenceMgr.getService(serviceName,
                                                          ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            service.setIsActive(true);
            persistenceMgr.updateEntity(service);
        } else if (eventType == AxisEvent.SERVICE_STOP) {
            ServiceDO service = persistenceMgr.getService(serviceName,
                                                          ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            service.setIsActive(false);
            persistenceMgr.updateEntity(service);
        } else if (eventType == AxisEvent.SERVICE_REMOVE) {
            if (serviceDO != null) {
                try {
                    persistenceMgr.deleteService(serviceName, serviceVersion);
                    //If the service is an EJB service, delete the saved 
                    //configuration info
                    removeEJBServiceConfiguration(axisService);
                } catch (ServiceNotFoundException ignored) {
                    // we have already verified that the serviceDO exists
                } catch (Exception e) {
                    String msg = "Cannot delete serviceDO [" + serviceName + "]";
                    log.error(msg, e);
                }
            }
        } else {

            // STS specific configurations
            if (axisService.getName().equals(ServerConstants.STS_NAME)
                && axisEvent.getEventType() == AxisEvent.SERVICE_DEPLOY) {

                // Add the STS Service
                ServerConfiguration config = ServerConfiguration.getInstance();
                KeyStoreDO[] keystores = persistenceMgr.getKeyStores();
                KeyStoreDO sysKs = null;
                for (int i = 0; i < keystores.length; i++) {
                    if (keystores[i].getIsPrimaryKeyStore()) {
                        sysKs = keystores[i];
                        break;
                    }
                }
                if (sysKs != null) {
                    String issuerName = config.getFirstProperty("HostName"); //TODO: Is this correct?
                    if (issuerName == null) {
                        //HostName not set :-( use wso2wsas-sts
                        issuerName = ServerConstants.STS_NAME;
                    }
                    String cryptoProvider = ServerCrypto.class.getName();
                    String ksId = sysKs.getKeyStoreName();
                    Properties props = RampartConfigUtil.getServerCryptoProperties(
                            new String[]{ksId}, ksId, sysKs.getPrivateKeyAlias());

                    SAMLTokenIssuerConfig stsSamlConfig = new SAMLTokenIssuerConfig(
                            issuerName, cryptoProvider, props);
                    stsSamlConfig.setIssuerKeyAlias(config.getFirstProperty("Security.KeyStore.KeyAlias"));
                    stsSamlConfig.setIssuerKeyPassword(config.getFirstProperty("Security.KeyStore.KeyPassword"));
                    stsSamlConfig.setAddRequestedAttachedRef(true);
                    stsSamlConfig.setAddRequestedUnattachedRef(true);
                    stsSamlConfig.setKeyComputation(2);
                    stsSamlConfig.setProofKeyType(TokenIssuerUtil.BINARY_SECRET);
                    try {
                        //remove param is exists
                        Parameter param = axisService.getParameter(SAMLTokenIssuerConfig.SAML_ISSUER_CONFIG.getLocalPart());
                        if (param == null) {
                            //Add new parameter
                            axisService.addParameter(stsSamlConfig.getParameter());
                        }
                    } catch (AxisFault e) {
                        log.error("Error setting STS-SAML configuration parameter", e);
                    }
                }
            }

            // Else, this is a AxisEvent.SERVICE_DEPLOY event.
            // This may be a new or existing serviceDO
            if (serviceDO == null) {  // is this a new serviceDO? If so, register it in the DB
                try {
                    persistenceMgr.handleNewServiceAddition(axisService);
                } catch (Exception e) {
                    String msg = "Could not handle initialization of new service [" +
                                 axisService.getName() + "]";
                    log.error(msg, e);
                }
            } else {  // The serviceDO is already registered in the DB
                try {
                    persistenceMgr.handleExistingServiceInit(serviceDO, axisService, axisConfig);
                } catch (Exception e) {
                    String msg = "Could not handle initialization of existing service [" +
                                 axisService.getName() + "]";
                    log.error(msg, e);
                }
            }
        }
    }


    private void removeEJBServiceConfiguration(AxisService axisService) {
        String serviceType = (String) axisService.getParameterValue(ServerConstants.SERVICE_TYPE);
        if (ServerConstants.SERVICE_TYPE_EJB.equals(serviceType)) {
            //No need for a NULL check.Its mandatory for a EJB service to have these params
            String beanJNDIName = (String) (axisService.getParameter(EJBUtil.EJB_JNDI_NAME)).getValue();
            String jnpProviderUrl = (String) (axisService.getParameter(EJBUtil.EJB_PROVIDER_URL)).getValue();
            try {
                persistenceMgr.removeEJBConfiguration(beanJNDIName, jnpProviderUrl);
            } catch (EJBConfigurationNotFoundException e) {
                log.error(e.getMessage());
            }
        }
    }

    public void moduleUpdate(AxisEvent axisEvent, AxisModule axisModule) {

        // We ignore admin module events
        String moduleName = axisModule.getName();
        if (moduleName.equals(ServerConstants.ADMIN_MODULE) ||
            moduleName.equals(ServerConstants.TRACER_MODULE) ||
            moduleName.equals(ServerConstants.STATISTICS_MODULE)) {
            return;
        }

        int eventType = axisEvent.getEventType();
        if (eventType == AxisEvent.MODULE_REMOVE) {
            return;
        }

        // Else this is a AxisEvent.MODULE_DEPLOY event. This may be a new or existing module
        String moduleVersion = axisModule.getVersion();
        if (moduleVersion == null) {
            moduleVersion = "";
        }
        ModuleDO moduleDO = persistenceMgr.getModule(moduleName, moduleVersion);

        if (moduleDO != null) {
            try {
                persistenceMgr.handleExistingModuleInit(moduleDO, axisModule);
            } catch (Exception e) {
                log.error("Could not handle initialization of existing module", e);
            }
        } else { // this is a new moduleDO which has not been registered in the DB yet
            try {
                persistenceMgr.handleNewModuleAddition(axisModule, moduleName, moduleVersion);
            } catch (Exception e) {
                log.error("Could not handle addition of new module", e);
            }
        }
    }

    public void addParameter(Parameter parameter) throws AxisFault {
        paramMap.put(parameter.getName(), parameter);
    }

    public void removeParameter(Parameter param) throws AxisFault {
        paramMap.remove(param.getName());
    }

    public void deserializeParameters(OMElement omElement) throws AxisFault {
        //No need to do anything here
    }

    public Parameter getParameter(String paramName) {
        return (Parameter) paramMap.get(paramName);
    }

    public ArrayList getParameters() {
        Collection collection = paramMap.values();
        ArrayList arr = new ArrayList();
        for (Iterator iterator = collection.iterator(); iterator.hasNext();) {
            arr.add(iterator.next());
        }
        return arr;
    }

    public boolean isParameterLocked(String paramName) {
        return ((Parameter) paramMap.get(paramName)).isLocked();
    }
}