/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas;

import org.apache.catalina.Context;
import org.apache.catalina.Host;
import org.apache.catalina.startup.Embedded;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.ArchiveManipulator;
import org.wso2.utils.FileManipulator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 *
 */
public class TomcatRepositoryListener implements Runnable {

    private static Log log = LogFactory.getLog(TomcatRepositoryListener.class);


    private Map repoMap = new HashMap();
    private Embedded embedded;
    private Host host;
    private File repo;
    private ArchiveManipulator arcMan = new ArchiveManipulator();
    private FileManipulator fileMan = new FileManipulator();

    public TomcatRepositoryListener(Embedded embedded,
                                    Host host,
                                    String repo) throws Exception {
        this.embedded = embedded;
        this.host = host;
        this.repo = new File(repo);
        if (!this.repo.isDirectory()) {
            this.repo = null;
            throw new Exception(repo + " is not a directory");
        }
    }

    /**
     * When an object implementing interface <code>Runnable</code> is used
     * to create a thread, starting the thread causes the object's
     * <code>run</code> method to be called in that separately executing
     * thread.
     * <p/>
     * The general contract of the method <code>run</code> is that it may
     * take any action whatsoever.
     *
     * @see Thread#run()
     */
    public void run() {
        while (true) {
            File[] files = repo.listFiles();
            Collection c = new ArrayList();

            // Deploy new files
            if (files != null) {
                for (int i = 0; i < files.length; i++) {
                    File file = files[i];
                    String filename = file.getName();
                    c.add(filename);
                    if (repoMap.containsKey(filename)) { // Hot update
                        long timestamp = ((WebContext) repoMap.get(filename)).getTimestamp();
                        if (timestamp < file.lastModified()) {
                            if (filename.endsWith(".war")) {
                                String dirName = filename.substring(0, filename.indexOf(".war"));
                                fileMan.deleteDir(
                                        new File(repo.getAbsolutePath() + File.separator + dirName));
                                try {
                                    arcMan.extract(repo.getAbsolutePath() + File.separator + filename,
                                                   repo.getAbsolutePath() + File.separator + dirName);
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }
                                WebContext webContext = (WebContext) repoMap.get(filename);
                                webContext.setTimestamp(file.lastModified());
                                repoMap.put(filename, webContext);
                            }
                        }
                    } else { // Hot deployment
                        if (file.isDirectory()) {
                            Context context;
                            if (filename.equals("ROOT")) {
                                context = embedded.createContext("/", file.getAbsolutePath());
                                host.addChild(context);
                            } else {
                                context =
                                        embedded.createContext("/" + filename, file.getAbsolutePath());
                                host.addChild(context);
                            }

                            if (log.isDebugEnabled()) {
                                log.debug("Deployed Web application: " + filename);
                            }

                            repoMap.put(filename, new WebContext(file.lastModified(), context));
                        } else if (filename.endsWith(".war")) { // Handle WAR
                            try {
                                arcMan.extract(repo.getAbsolutePath() + File.separator + filename,
                                               repo.getAbsolutePath() + File.separator +
                                               filename.substring(0,
                                                                  filename.indexOf(".war")));
                                repoMap.put(filename, new WebContext(file.lastModified(), null));
                            } catch (IOException e) {
                                log.error("Exception occurred while extracting archive " + filename,
                                          e);
                            }
                        }
                    }
                }

                // Undeploy old files
                Collection toRemove = new ArrayList();
                for (Iterator iterator = repoMap.keySet().iterator(); iterator.hasNext();) {
                    String filename = (String) iterator.next();
                    if (!c.contains(filename)) {
                        Context context = ((WebContext) repoMap.get(filename)).getContext();
                        if (context != null) {
                            host.removeChild(context);
                            toRemove.add(filename);
                            log.info("Undeployed Web application " + filename);
                        }
                    }
                }
                for (Iterator iterator = toRemove.iterator(); iterator.hasNext();) {
                    repoMap.remove(iterator.next());
                }
            }

            // Poll the repo every few seconds
            try {
                Thread.sleep(2000);
            } catch (InterruptedException e) {
                log.error("TomcatRepositoryListener thread was interrupted", e);
            }
        }
    }

    private static class WebContext {
        private long timestamp;
        private Context context;

        public WebContext(long timestamp, Context context) {
            this.timestamp = timestamp;
            this.context = context;
        }

        public long getTimestamp() {
            return timestamp;
        }

        public Context getContext() {
            return context;
        }

        public void setTimestamp(long timestamp) {
            this.timestamp = timestamp;
        }
    }
}
