/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas;

import org.apache.axis2.AxisFault;
import org.apache.axis2.clustering.ClusterManager;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.engine.ListenerManager;
import org.apache.axis2.transport.TransportListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.NetworkUtils;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerConfigurator;
import org.wso2.utils.ServerException;
import org.wso2.utils.WSO2ConfigurationContextFactory;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.exception.KeyStoreAlreadyExistsException;
import org.wso2.wsas.util.KeyStoreUtil;
import org.wso2.wsas.util.LoggingUtil;
import org.wso2.wsas.util.Monitor;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.WsasUtils;

import java.io.File;
import java.net.SocketException;

/**
 * This class is responsible for managing the WSO2 WSAS instance during startup.
 */
public final class ServerManager {
    private static final Log log = LogFactory.getLog(ServerManager.class);

    public String axis2RepoLocation;
    public Monitor monitor;
    public ConfigurationContext configContext;
    public String serverWorkDir;
    public String adminResourceBase;

    private ServerConfiguration serverConfig;

    private static ServerManager instance;
    private ListenerManager listenerManager;
    private int httpPort = 10001;
    private int httpsPort = 11001;

    public static ServerManager getInstance() {
        if (instance == null) {
            instance = new ServerManager();
        }
        return instance;
    }

    private ServerManager() {
        this.serverConfig = ServerConfiguration.getInstance();
    }

    public void setHttpPort(int httpPort) {
        this.httpPort = httpPort;
    }

    public void setHttpsPort(int httpsPort) {
        this.httpsPort = httpsPort;
    }

    public int getHttpPort() {
        return httpPort;
    }

    public int getHttpsPort() {
        return httpsPort;
    }

    public void start() throws ServerException {
        try {
            ServerConfigurator serverConfigurator = ServerConfigurator.getInstance();
            serverConfigurator.cleanup();
            if (!serverConfigurator.isInitialized()) {
                serverConfigurator.init(axis2RepoLocation,
                                        System.getProperty(ServerConstants.WEB_RESOURCE_LOCATION));
            }

            initLoggingConfiguration();

            persistSecurityKeyStore();

            ConfigurationContext oldContext = configContext;
            if (oldContext != null) {
                ListenerManager.getDefaultListenerManager().stop();
            }
            configContext =
                    WSO2ConfigurationContextFactory.createNewConfigurationContext(ServerConfigurator.getInstance());

            // Disable admin services, if needed
            if (!WsasUtils.isAdminConsoleEnabled()) {
                AxisConfiguration axisConfig = configContext.getAxisConfiguration();
                if (axisConfig.getServiceGroup(ServerConstants.ADMIN_SERVICE_GROUP) != null) {
                    axisConfig.removeServiceGroup(ServerConstants.ADMIN_SERVICE_GROUP);
                }
            }

            // If an old config context existed, use its cluster manager
            if (oldContext != null) {
                ClusterManager originalClusterManager =
                        oldContext.getAxisConfiguration().getClusterManager();
                configContext.getAxisConfiguration().setClusterManager(originalClusterManager);
            }

            initNetworkUtils(configContext.getAxisConfiguration());

            runIntializers();

            try {
                ServerStatus.setServerRunning();
            } catch (AxisFault e) {
                String msg = "Cannot set server to running mode";
                log.error(msg, e);
            }
        } catch (Exception e) {
            String msg = "Cannot start WSO2 WSAS";
            log.fatal(msg, e);
            throw new ServerException(msg, e);
        }
    }

    private void persistSecurityKeyStore() throws ServerException {

        String ksLocation = serverConfig.getFirstProperty("Security.KeyStore.Location");
        if (!KeyStoreUtil.keyStoreExists(ksLocation)) {
            try {
                KeyStoreUtil.persistKeyStore(new File(ksLocation).getAbsolutePath(),
                                             serverConfig.
                                                     getFirstProperty("Security.KeyStore.Password"),
                                             serverConfig.
                                                     getFirstProperty("Security.KeyStore.Type"),
                                             serverConfig.
                                                     getFirstProperty("Security.KeyStore.KeyAlias"),
                                             serverConfig.
                                                     getFirstProperty(
                                                     "Security.KeyStore.KeyPassword"),
                                             null,
                                             true);
            } catch (KeyStoreAlreadyExistsException e) {
                throw new ServerException(e);
            }
        }
    }

    public void status() {
    }

    public void stop() {
    }

    /**
     * Reads the database and check whether log setting already there. if so update the logger
     * settings with the database settings else update the data to data base
     */
    private void initLoggingConfiguration() {

        // check whether logging configurations are already persisted
        String isLoaded =
                new PersistenceManager().
                        getConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_IS_LOADED);
        if (Boolean.valueOf(isLoaded).booleanValue()) {
            LoggingUtil.loadCustomConfiguration();
        } else {
            // parse the logging file and get the information
            LoggingUtil.loadDefaultConfiguration();
        }
    }

    public void startListenerManager() {
        listenerManager = new ListenerManager();
        listenerManager.startSystem(configContext);
        ListenerManager.defaultConfigurationContext = configContext;
    }

    public void stopListenerManager() throws AxisFault {
        try {
            if (listenerManager != null) {
                listenerManager.destroy();
            }

            //we need to call this method to clean the temp files we created.
            if (configContext != null) {
                configContext.terminate();
            }
        } catch (Exception e) {
            log.error("Exception occurred while shutting down listeners", e);
        }
    }

    private void runIntializers() throws ServerException {

        String[] initializers =
                serverConfig.getProperties("ServerInitializers.Initializer");
        for (int i = 0; i < initializers.length; i++) {
            String clazzName = initializers[i];
            try {
                Class clazz = Class.forName(clazzName);
                ServerInitializer intializer = (ServerInitializer) clazz.newInstance();
                if (log.isDebugEnabled()) {
                    log.debug("Using ServerInitializer " + intializer.getClass().getName());
                }
                intializer.init(configContext);
            } catch (Exception e) {
                throw new ServerException(e);
            }
        }
    }

    private void initNetworkUtils(AxisConfiguration axisConfiguration)
            throws AxisFault, SocketException {
        String hostName = serverConfig.getFirstProperty("HostName");
        if (hostName != null) {
            Parameter param = axisConfiguration.getParameter(TransportListener.HOST_ADDRESS);
            if (param != null) {
                param.setEditable(true);
                param.setValue(hostName);
            } else {
                param = ParameterUtil.createParameter(TransportListener.HOST_ADDRESS, hostName);
                axisConfiguration.addParameter(param);
            }
        } else {
            Parameter param = axisConfiguration.getParameter(TransportListener.HOST_ADDRESS);
            if (param != null) {
                hostName = (String) param.getValue();
                log.info(TransportListener.HOST_ADDRESS + " has been selected from Axis2.xml.");
            }
        }
        NetworkUtils.init(hostName);
    }
}
