/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas;

import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.clustering.ClusterManager;
import org.apache.axis2.clustering.ClusteringConstants;
import org.apache.axis2.clustering.configuration.ConfigurationManager;
import org.apache.axis2.clustering.context.ContextManager;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.adminui.AdminUIServletFilter;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerConfigurationException;
import org.wso2.utils.ServerException;
import org.wso2.utils.transport.ProxyCache;
import org.wso2.wsas.serverinfo.ServerInfo;
import org.wso2.wsas.transport.ServerPropertyKeys;
import org.wso2.wsas.util.ClusteringUtil;
import org.wso2.wsas.util.Controllable;
import org.wso2.wsas.util.MIMEType2FileExtensionMap;
import org.wso2.wsas.util.Monitor;
import org.wso2.wsas.util.ServerController;
import org.wso2.wsas.util.SystemRestarter;
import org.wso2.wsas.util.WsasUtils;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

public class MainServlet extends HttpServlet implements Controllable {
    private static final Log log = LogFactory.getLog(MainServlet.class);
    private static final long serialVersionUID = 4517849288538613640L;

    protected ServerManager serverManager;
    protected String serverName;

    private ServletConfig servletConfig;
    private String wso2wsasHome;
    private ServerConfiguration serverConfig;
    private Thread shutdownHook;
    private ServerController serverController;

    protected void doGet(HttpServletRequest request,
                         HttpServletResponse response) throws ServletException,
                                                              IOException {
    }

    protected void doPost(HttpServletRequest request,
                          HttpServletResponse response) throws ServletException,
                                                               IOException {
    }

    public void init(ServletConfig servletConfig) throws ServletException {
        long startTime;
        if (System.getProperty(ServerConstants.WSO2WSAS_START_TIME) == null) {
            startTime = System.currentTimeMillis();
        } else {
            startTime = Long.parseLong(System.getProperty(ServerConstants.WSO2WSAS_START_TIME));
        }

        this.servletConfig = servletConfig;

        // Expanding of web content within AAR file
        try {
            System.setProperty(ServerConstants.WEB_RESOURCE_LOCATION,
                               servletConfig.getServletContext().getRealPath("/wservices"));
        } catch (Exception e) {
            log.warn("wso2wsas.war is not expanded. Web content within AAR file not supported.");
        }

        ServerInfo serverInfo;
        Properties props;
        try {
            InputStream is =
                    Thread.currentThread().getContextClassLoader().
                            getResourceAsStream("wso2wsas.properties");
            props = new Properties();
            if (is != null) {
                props.load(is);
            } else {
                String msg = "wso2wsas.properties file not found in classpath";
                log.fatal(msg);
                log.fatal(serverName + " startup failed.");
                throw new ServletException(msg);
            }
            is.close();
        } catch (IOException e) {
            String msg = "wso2wsas.properties file cannot be read";
            log.fatal(msg, e);
            log.fatal(serverName + " startup failed.");
            throw new ServletException(msg, e);
        }

        //Setting the WSAS home
        wso2wsasHome = System.getProperty(ServerConstants.WSO2WSAS_HOME);
        if (wso2wsasHome == null) {
            wso2wsasHome = props.getProperty(ServerConstants.WSO2WSAS_HOME);
            if (wso2wsasHome == null) {
                String msg = ServerConstants.WSO2WSAS_HOME +
                             " property not set in wso2wsas.properties file";
                log.fatal(msg);
                log.fatal(serverName + " startup failed.");
                throw new ServletException(msg);
            }
        }

        System.setProperty(ServerConstants.AXIS2_HOME, wso2wsasHome);
        System.setProperty(ServerConstants.WSO2WSAS_HOME, wso2wsasHome);

        // -------------------------------------------------------------------------------------
        serverConfig = ServerConfiguration.getInstance();
        try {
            serverConfig.init(WsasUtils.getWsasServerXml());
        } catch (ServerConfigurationException e) {
            String msg = "Could not initialize server configuration";
            log.fatal(msg);
            log.fatal(serverName + " startup failed.");
            throw new ServletException(msg);
        }
        serverManager = ServerManager.getInstance();
        // -------------------------------------------------------------------------------------

        serverName = serverConfig.getFirstProperty("Name");

        File file = new File(wso2wsasHome);
        if (!file.exists()) {
            String msg = ServerConstants.WSO2WSAS_HOME + " " + wso2wsasHome +
                         " does not exist.";
            log.fatal(msg);
            log.fatal(serverName + " startup failed.");
            throw new ServletException(msg);
        }
        if (!file.isDirectory()) {
            String msg = ServerConstants.WSO2WSAS_HOME + " " + wso2wsasHome +
                         " is not a directory.";
            log.fatal(msg);
            log.fatal(serverName + " startup failed.");
            throw new ServletException(msg);
        }

        // ServerInfo class
        String serverInfoImplClass = props.getProperty(ServerPropertyKeys.APPSERVER_INFO_IMPL);
        log.info("Using Server Info Class: " + serverInfoImplClass);
        try {
            serverInfo = (ServerInfo) Class.forName(serverInfoImplClass).newInstance();
            serverInfo.setProperties(props);
        } catch (Exception e) {
            String msg = "Could not instantiate ServerInfo implementation class " +
                         serverInfoImplClass;
            log.fatal(msg, e);
            log.fatal(serverName + " startup failed.");
            throw new ServletException(msg);
        }

        try {
            start(servletConfig);
        } catch (ServerException e) {
            String msg = "Could not start " + serverName;
            log.fatal(msg, e);
            throw new ServletException(msg);
        }

        ConfigurationContext configCtx = serverManager.configContext;

        // Listeners should be initialize before ServerInfo processing. 
        serverManager.startListenerManager();

        serverInfo.process();
        ServerManager.getInstance().setHttpPort(serverInfo.getHttpPort());
        ServerManager.getInstance().setHttpsPort(serverInfo.getHttpsPort());

        WsasUtils.setServletContextProperties(servletConfig.getServletContext(),
                                              (Map) configCtx
                                                      .getProperty(ServerConstants.GENERATED_PAGES),
                                              WsasUtils.isAdminConsoleEnabled(),
                                              ProxyCache.getInstance().getHttpPort() != -1 ?
                                              ProxyCache.getInstance().getHttpPort() :
                                              ServerManager.getInstance().getHttpPort(),
                                              ProxyCache.getInstance().getHttpsPort() != -1 ?
                                              ProxyCache.getInstance().getHttpsPort() :
                                              ServerManager.getInstance().getHttpsPort(),
                                              configCtx.getServicePath());
        // Print information about the system
        printInfo(startTime);
        System.getProperties().remove(ServerConstants.WSO2WSAS_START_TIME);
    }

    public void start(ServletConfig servletConfig) throws ServerException {
        ServletContext servletContext = servletConfig.getServletContext();

        System.setProperty(ServerConstants.DERBY_HOME,
                           new File(serverConfig.getFirstProperty(
                                   "Database.Home")).getAbsolutePath());
        String hostName = ServerConfiguration.getInstance().getFirstProperty("ClusteringHostName");
        if (System.getProperty(ClusteringConstants.LOCAL_IP_ADDRESS) == null &&
            hostName != null && hostName.trim().length() != 0) {
            System.setProperty(ClusteringConstants.LOCAL_IP_ADDRESS, hostName);
        }
        serverManager.adminResourceBase = servletContext.getRealPath(".");
        if (serverManager.adminResourceBase == null) {
            serverManager.adminResourceBase = "TODO"; //TODO: Get the path to the resource base. Handle unexpanded case.
        }
        serverManager.serverWorkDir =
                new File(serverConfig.getFirstProperty("WorkDirectory")).getAbsolutePath();
        System.setProperty("axis2.work.dir", serverManager.serverWorkDir);

        serverManager.axis2RepoLocation =
                serverConfig.getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        if (!serverManager.axis2RepoLocation.endsWith("/")) {
            serverConfig.setConfigurationProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION,
                                                  serverManager.axis2RepoLocation + "/");
            serverManager.axis2RepoLocation = serverManager.axis2RepoLocation + "/";
        }
        System.setProperty(Constants.AXIS2_REPO, serverManager.axis2RepoLocation);

        MIMEType2FileExtensionMap.mappingFileName = wso2wsasHome + File.separator +
                                                    "conf" + File.separator + "mime-mappings.xml";

        try {
            serverManager.start();
            addShutdownHook();
            ConfigurationContext configCtx = serverManager.configContext;
            servletConfig.getServletContext().setAttribute(ServerConstants.CONFIGURATION_CONTEXT,
                                                           configCtx);
            configCtx.setProperty(Constants.CONTAINER_MANAGED, "true");
            configCtx.setProperty(ServerConstants.WORK_DIR, serverManager.serverWorkDir);

            configCtx.setProperty(ServerConstants.WSO2WSAS_INSTANCE, this);
            startServerController();
        } catch (Exception e) {
            throw new ServerException(serverName + " startup failed.", e);
        }
    }

    private void printInfo(long startTime) {
        int httpPort = ServerManager.getInstance().getHttpPort();
        if (httpPort != -1) {
            log.info("HTTP port            : " + httpPort);
        }
        int httpsPort = ServerManager.getInstance().getHttpsPort();
        if (httpsPort != -1) {
            log.info("HTTPS port           : " + httpsPort);
        }
        log.info("");
        log.info(serverName + " started in " + (System.currentTimeMillis() - startTime) + " ms");
    }

    private void addShutdownHook() {
        if (shutdownHook != null) {
            return;
        }

        shutdownHook = new Thread() {
            public void run() {
                log.info("Shutting down " + serverName + "...");
                try {
                    new ServerManagement(serverManager.configContext.
                            getAxisConfiguration().getTransportsIn()).startMaintenance();
                    cleanupSystem();

                    try {
                        serverManager.stopListenerManager();
                    } catch (AxisFault e) {
                        String msg = "Error occurred while stopping Axis2 Listener Manager";
                        log.error(msg, e);
                        throw new ServerException(msg, e);
                    }
                    log.info("Shutdown complete");
                    log.info("Halting JVM");
                } catch (Exception e) {
                    log.warn("Error occurred while shutting down WSO2 WSAS", e);
                }
            }
        };
        Runtime.getRuntime().addShutdownHook(shutdownHook);
    }

    public void setAxis2RepoLocation(String axis2RepoLocation) {
        serverManager.axis2RepoLocation = axis2RepoLocation;
    }

    public void setMonitor(Monitor monitor) {
        serverManager.monitor = monitor;
    }

    public void startServer() throws ServerException {
        try {
            init(this.servletConfig);
            reinitializeServlets(this.servletConfig.getServletContext());
            try {
                ServerStatus.setServerRunning();
            } catch (AxisFault e) {
                String msg = "Cannot set server to running mode";
                log.error(msg, e);
            }
        } catch (ServletException e) {
            String msg = "Cannot start server";
            log.fatal(msg, e);
            throw new ServerException(msg, e);
        }
    }

    public void stopListeners() throws ServerException {
    }

    public void restart() {
        try {
            ServerStatus.setServerRestarting();
        } catch (AxisFault e) {
            String msg = "Cannot set server to restarting mode";
            log.error(msg, e);
        }
        SystemRestarter st = new SystemRestarter(false,
                                                 this,
                                                 this,
                                                 serverManager.axis2RepoLocation,
                                                 serverManager.monitor);
        cleanupSystem();
        new Thread(st).start();
    }

    public void restartGracefully() {
        try {
            ServerStatus.setServerRestarting();
        } catch (AxisFault e) {
            String msg = "Cannot set server to restarting mode";
            log.error(msg, e);
        }
        SystemRestarter st = new SystemRestarter(true,
                                                 this,
                                                 this,
                                                 serverManager.axis2RepoLocation,
                                                 serverManager.monitor);
        cleanupSystem();
        new Thread(st).start();
    }

    public void shutdown() {
        try {
            ServerStatus.setServerShuttingDown();
        } catch (AxisFault e) {
            String msg = "Cannot set server to shutdown mode";
            log.error(msg, e);
        }
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
        System.exit(0);
    }

    public void shutdownGracefully() {
        try {
            ServerStatus.setServerShuttingDown();
        } catch (AxisFault e) {
            String msg = "Cannot set server to shutdown mode";
            log.error(msg, e);
        }
        System.exit(0);
    }

    public Thread getShutdownHook() {
        return shutdownHook;
    }

    public void setConfigurationContext(ConfigurationContext newConfigCtx) throws
                                                                           ServerException {
        try {

            // We need to preserve the old ClusterManager
            ConfigurationContext oldConfigCtx = serverManager.configContext;
            AxisConfiguration oldAxisConfig = oldConfigCtx.getAxisConfiguration();
            ClusterManager originalClusterManager = oldAxisConfig.getClusterManager();
            newConfigCtx.getAxisConfiguration().setClusterManager(originalClusterManager);

            // We need to preserve the original properties which were in the config ctx
            for (Iterator iter = oldConfigCtx.getPropertyNames();
                 iter.hasNext();) {
                String key = (String) iter.next();
                newConfigCtx.setProperty(key, oldConfigCtx.getProperty(key));
            }

            // Remove all clustering references
            originalClusterManager.setConfigurationContext(newConfigCtx);
            ConfigurationManager configurationManager =
                    originalClusterManager.getConfigurationManager();
            if (configurationManager != null) {
                configurationManager.setConfigurationContext(newConfigCtx);
            }
            ContextManager contextManager = originalClusterManager.getContextManager();
            if (contextManager != null) {
                contextManager.setConfigurationContext(newConfigCtx);
            }
            oldAxisConfig.setClusterManager(null);

            // Stop the ListenerManager
            try {
                serverManager.stopListenerManager();
            } catch (AxisFault axisFault) {
                throw new ServerException(axisFault);
            }

            // Now we switch to the new Config ctx
            serverManager.configContext = newConfigCtx;

            // Disable admin services, if needed
            if (!WsasUtils.isAdminConsoleEnabled()) {
                newConfigCtx.getAxisConfiguration().
                        removeServiceGroup(ServerConstants.ADMIN_SERVICE_GROUP);
            }

            ServletContext servletContext = servletConfig.getServletContext();
            servletContext.removeAttribute(ServerConstants.CONFIGURATION_CONTEXT);
            servletContext.setAttribute(ServerConstants.CONFIGURATION_CONTEXT,
                                        newConfigCtx);
            reinitializeServlets(servletContext);

            // Need to enable clustering on the new configurationContext
            try {
                ClusteringUtil.enableClustering(newConfigCtx);
            } catch (AxisFault axisFault) {
                throw new ServerException(axisFault);
            }

            // Start the ListenerManager
            serverManager.startListenerManager();
        } catch (Exception e) {
            String msg = "Cannot set ConfigurationContext";
            log.fatal(msg, e);
            throw new ServerException(msg, e);
        }
    }

    public ConfigurationContext getConfigurationContext() {
        return serverManager.configContext;
    }

    private void reinitializeServlets(ServletContext servletContext) throws ServletException {
        // Reinitialize all of the servlets, for this puprpose,
        // the servlets should have added themselves
        // as ServletContext attributes in the previous init call
        Enumeration attributeNames = servletContext.getAttributeNames();
        while (attributeNames.hasMoreElements()) {
            Object attrib = servletContext.getAttribute((String) attributeNames.nextElement());
            if (attrib instanceof HttpServlet) {
                ((HttpServlet) attrib).init();
            } else if (attrib instanceof AdminUIServletFilter) {
                AdminUIServletFilter adminUIServletFilter = (AdminUIServletFilter) attrib;
                adminUIServletFilter.init();
            }
        }
    }

    private void startServerController() {
        if (serverController != null) {
            return;
        }
        serverController = new ServerController(this);
        int cmdListenerPort = WsasUtils.getCommandListenerPort();
        if (cmdListenerPort == -1) {
            return;
        }
        serverController.setPort(cmdListenerPort);
        Thread thread = new Thread(serverController);
        thread.start();
        log.info("Command port         : " + cmdListenerPort);
    }

    public void destroy() {
        try {
            shutdownGracefully();
        } catch (Exception e) {
            log.error("Error occurred while shutting down the system", e);
        }
    }

    private void cleanupSystem() {
        log.info("Cleaning up system...");
        new FileManipulator().deleteDir(new File(wso2wsasHome + File.separator +
                                                 serverConfig.
                                                         getFirstProperty("WorkDirectory")));
        ConfigurationContext configContext = serverManager.configContext;
        if (configContext != null) {
            Object property = configContext.getProperty(ServerConstants.FILE_RESOURCE_MAP);
            if (property != null) {
                ((Map) property).clear();
            }
        }
    }
}
